﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_DisplayLayerControl.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/spl/spl_Api.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Mutex.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/manager/vi_Manager.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <mutex>
#include <cstring>
#include <nn/am/service/am_StuckChecker.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>
#include <nn/am/service/display/am_DisplayWorker.h>
#include <nn/am/service/display/am_IntegratedDisplayManager.h>
#include <nn/am/service/display/am_ScreenShotRequestQueue.h>

#define NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX() std::lock_guard<decltype(*g_pDisplayMutex)> lk(*g_pDisplayMutex);
#define NN_AM_DISPCTRL_DUMP_ON_EXIT() NN_UTIL_SCOPE_EXIT{ g_pIntegratedDisplayManager->Dump(); };

namespace nn { namespace am { namespace service {

namespace {

    static const int DefaultFrameBufferSharingLevel = 2;
    static const int SupportedFrameBufferSharingLevelMin = 2;
    static const int SupportedFrameBufferSharingLevelMax = 2;

    NN_STATIC_ASSERT(DefaultFrameBufferSharingLevel >= SupportedFrameBufferSharingLevelMin);
    NN_STATIC_ASSERT(DefaultFrameBufferSharingLevel <= SupportedFrameBufferSharingLevelMax);

    template <typename T, typename U>
    T ReadFirmwareDebugSettings(const char* name, U&& defaultValue) NN_NOEXCEPT
    {
        T ret;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&ret, sizeof(ret), "am.display", name);
        if (!(sizeof(ret) == size))
        {
            NN_AM_SERVICE_LOG(error, "failed: ReadFirmwareDebugSettings(%s) size = %d(expected:%d)\n", name, static_cast<int>(size), static_cast<int>(sizeof(ret)));
            return defaultValue;
        }
        return ret;
    }

    util::optional<nn::os::Mutex> g_pDisplayMutex;
    nn::util::optional<display::IntegratedDisplayManager> g_pIntegratedDisplayManager;
    display::DisplayWorker g_DisplayWorker;
    display::ScreenShotRequestQueue g_ScreenShotRequestQueue;
}

void ScreenShotRequestCallbackFunction(const display::ScreenShotRequest& request, void*) NN_NOEXCEPT;

void InitializeDisplayLayerControl() NN_NOEXCEPT
{
    display::DisplayPrimitiveOperation::Initialize();

    g_pDisplayMutex.emplace(false);

    // バッファ共有レベルを取得
    int shareLevel = ReadFirmwareDebugSettings<int>("frame_buffer_sharing_level", DefaultFrameBufferSharingLevel);
    if(shareLevel < SupportedFrameBufferSharingLevelMin)
    {
        NN_AM_SERVICE_LOG(error, "FrameBufferSharingLevel %d is no longer supported. Using defulat level %d.\n", shareLevel, DefaultFrameBufferSharingLevel);
        shareLevel = DefaultFrameBufferSharingLevel;
    }
    else if(shareLevel > SupportedFrameBufferSharingLevelMax)
    {
        NN_AM_SERVICE_LOG(error, "FrameBufferSharingLevel %d is unknown. Using defulat level %d.\n", shareLevel, DefaultFrameBufferSharingLevel);
        shareLevel = DefaultFrameBufferSharingLevel;
    }

    // CallerApplet の互換モードの設定
    bool isImmediateCallerAppletCaptureForced = ReadFirmwareDebugSettings<bool>("force_immediate_update_caller_applet_capture", false);


    // パラメータを取得
    display::IntegratedDisplayParameter param = {};
    switch(shareLevel)
    {
    case 2:
        NN_AM_SERVICE_LOG(seq, "FrameBufferSharingLevel = 2 (enable-pure)\n");
        param = display::IntegratedDisplayParameter::GetValueForPureSharingMode();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    param.isImmediateCallerAppletCaptureForced = isImmediateCallerAppletCaptureForced;

    g_pIntegratedDisplayManager.emplace(param);
    g_DisplayWorker.Initialize();
    g_ScreenShotRequestQueue.Initialize(&g_DisplayWorker, ScreenShotRequestCallbackFunction, nullptr);

    g_DisplayWorker.Start();
}

Result CreateIndirectLayer(vi::IndirectLayerHandleType* pOut) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    NN_AM_SERVICE_LOG(seq, "Try CreateIndirectLayer()\n");

    NN_RESULT_DO(display::DisplayPrimitiveOperation::CreateIndirectLayer(pOut));
    NN_AM_SERVICE_LOG(seq, "CreateIndirectLayer() -> L_%lld\n", *pOut);
    NN_RESULT_SUCCESS;
}

void DestroyIndirectLayer(vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    NN_AM_SERVICE_LOG(seq, "DestroyIndirectLayer(L_%lld)\n", layerHandle);
    display::DisplayPrimitiveOperation::DestroyIndirectLayer(layerHandle);
}

Result CreateIndirectLayerConsumerEndPoint(vi::IndirectConsumerHandleType* pOut, vi::IndirectLayerHandleType layerHandle, applet::AppletResourceUserId userAruid) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    NN_AM_SERVICE_LOG(seq, "Try CreateIndirectConsumerEndPoint(L_%lld, A_%llx)\n", layerHandle, userAruid.lower);
    NN_RESULT_DO(display::DisplayPrimitiveOperation::CreateIndirectConsumerEndPoint(pOut, layerHandle, userAruid));
    NN_AM_SERVICE_LOG(seq, "CreateIndirectConsumerEndPoint(L_%lld, A_%llx) -> C_%lld\n", layerHandle, userAruid.lower, *pOut);
    NN_RESULT_SUCCESS;
}

void DestroyIndirectLayerConsumerEndPoint(vi::IndirectLayerHandleType layerHandle, vi::IndirectConsumerHandleType hConsumer) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    NN_AM_SERVICE_LOG(seq, "DestroyIndirectConsumerEndPoint(C_%lld)\n", layerHandle);
    display::DisplayPrimitiveOperation::DestroyIndirectConsumerEndPoint(layerHandle, hConsumer);
}

Result SetDisplayCmuLuma(const char* displayName, float targetLuma) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    vi::DisplayId displayId = 0;
    NN_RESULT_DO(display::DisplayPrimitiveOperation::OpenDisplay(&displayId, displayName));
    NN_UTIL_SCOPE_EXIT{ display::DisplayPrimitiveOperation::CloseDisplay(displayId); };

    return display::DisplayPrimitiveOperation::SetDisplayCmuLuma(displayId, targetLuma);
}

Result OpenDisplay(vi::DisplayId* pOutDisplayId, const char* displayName) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    return display::DisplayPrimitiveOperation::OpenDisplay(pOutDisplayId, displayName);
}

Result CloseDisplay(vi::DisplayId displayId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    display::DisplayPrimitiveOperation::CloseDisplay(displayId);
    NN_RESULT_SUCCESS;
}

Result GetDisplayHotplugState(vi::HotplugState* pOutValue, vi::DisplayId displayId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    vi::HotplugStateType value = {};
    NN_RESULT_DO(display::DisplayPrimitiveOperation::GetDisplayHotplugState(&value, displayId));
    *pOutValue = static_cast<vi::HotplugState>(value);
    NN_RESULT_SUCCESS;
}

Result GetDisplayHotplugEvent(os::SystemEventType* pOutValue, vi::DisplayId displayId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    nn::sf::NativeHandle handle;
    NN_RESULT_DO(display::DisplayPrimitiveOperation::GetDisplayHotplugEvent(handle, displayId));
    nn::os::AttachReadableHandleToSystemEvent(pOutValue, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();
    NN_RESULT_SUCCESS;
}

Result SetDisplayAlpha(vi::DisplayId displayId, float alpha) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    return display::DisplayPrimitiveOperation::SetDisplayAlpha(displayId, alpha);
}

Result SetDisplayPowerState(vi::DisplayId displayId, vi::PowerState value) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    return display::DisplayPrimitiveOperation::SetDisplayPowerState(displayId, static_cast<vi::PowerStateType>(value));
}

Result GetDisplayResolution(int* pOutWidth, int* pOutHeight, vi::DisplayId displayId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    int64_t width, height;
    NN_RESULT_DO(display::DisplayPrimitiveOperation::GetDisplayResolution(&width, &height, displayId));
    *pOutWidth = static_cast<int>(width);
    *pOutHeight = static_cast<int>(height);
    NN_RESULT_SUCCESS;
}

Result SetDefaultDisplay(vi::DisplayId displayId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
    return display::DisplayPrimitiveOperation::SetDefaultDisplay(displayId);
}

//-------------------------------------
// モード切替 API
//-------------------------------------

void ChangeDisplayLayerControlModeToSystem() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ChangeDisplayLayerControlModeToSystem()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->ChangeToSystemMode();
}

void ChangeDisplayLayerControlModeToApplet() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ChangeDisplayLayerControlModeToApplet()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->ChangeToAppletMode();
}

//-------------------------------------
// レイヤ管理 API
//-------------------------------------

bool IsSystemBufferSharingEnabled() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("IsSystemBufferSharingEnabled()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->IsSystemBufferSharingEnabled();
}

IntegratedDisplayLayerHandle CreateIntegratedDisplayLayer() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("CreateIntegratedDisplayLayer()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->CreateIntegratedDisplayLayer();
}

void DestroyIntegratedDisplayLayer(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("DestroyIntegratedDisplayLayer()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->DestroyIntegratedDisplayLayer(h);
}

bool IsIntegratedDisplayLayerUserSpecified(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("IsIntegratedDisplayLayerUserSpecified(h=%lld)\n", h);
    return g_pIntegratedDisplayManager->GetUserClass(h) != display::detail::IntegratedDisplayLayerUserClass_Unspecified;
}

bool IsIntegratedDisplayLayerBufferOwnerSpecified(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("IsIntegratedDisplayLayerBufferOwnerSpecified(h=%lld)\n", h);
    return g_pIntegratedDisplayManager->GetBufferOwner(h) != display::detail::IntegratedDisplayLayerBufferOwner_Unspecified;
}

bool IsIntegratedDisplayLayerDestinationSpecified(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("IsIntegratedDisplayLayerDestinationSpecified(h=%lld)\n", h);
    return g_pIntegratedDisplayManager->GetDestinationLayer(h) != display::detail::IntegratedDisplayLayerDestinationLayer_Unspecified;
}

bool SpecifyIntegratedDisplayLayerUserApplication(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerUserApplication(h=%lld,aruid=%lld)\n", h, aruid);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyUserClass(h, display::detail::IntegratedDisplayLayerUserClass_Application, aruid);
}

bool SpecifyIntegratedDisplayLayerUserFullscreenApplet(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerUserFullscreenApplet(h=%lld,aruid=%lld)\n", h, aruid);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyUserClass(h, display::detail::IntegratedDisplayLayerUserClass_FullscreenApplet, aruid);
}

bool SpecifyIntegratedDisplayLayerUserPartialApplet(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerUserPartialApplet(h=%lld,aruid=%lld)\n", h, aruid);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyUserClass(h, display::detail::IntegratedDisplayLayerUserClass_PartialApplet, aruid);
}

bool SpecifyIntegratedDisplayLayerUserOverlay(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerUserOverlay(h=%lld,aruid=%lld)\n", h, aruid);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyUserClass(h, display::detail::IntegratedDisplayLayerUserClass_Overlay, aruid);
}

bool SpecifyIntegratedDisplayLayerUserNoDisplay(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerUserNoDisplay(h=%lld,aruid=%lld)\n", h, aruid);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyUserClass(h, display::detail::IntegratedDisplayLayerUserClass_NoDisplay, aruid);
}

bool SpecifyIntegratedDisplayLayerDestinationDisplay(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerDestinationDisplay(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyDestinationLayer(h, display::detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer, {});
}

bool SpecifyIntegratedDisplayLayerDestinationIndirect(IntegratedDisplayLayerHandle h, nn::vi::IndirectLayerHandleType indirectLayerHandle) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerDestinationIndirect(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyDestinationLayer(h, display::detail::IntegratedDisplayLayerDestinationLayer_IndirectLayer, indirectLayerHandle);
}

bool SpecifyIntegratedDisplayLayerBufferOwnerApplet(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerBufferOwnerApplet(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyBufferOwner(h, display::detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet);
}

bool SpecifyIntegratedDisplayLayerBufferOwnerSystemShared(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SpecifyIntegratedDisplayLayerBufferOwnerSystemShared(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SpecifyBufferOwner(h, display::detail::IntegratedDisplayLayerBufferOwner_SystemShared);
}

void SetIntegratedDisplayLayerScreenShotPremissionGetterFunction(IntegratedDisplayLayerHandle h, bool (*func)(void*), void* userPtr) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetIntegratedDisplayLayerScreenShotPermissionGetterFunction()\n");
    //NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetScreenShotPermissionGetterFunction(h, func, userPtr);
}

nn::vi::fbshare::SharedBufferHandle AcquireIntegratedDisplayLayerSystemSharedBufferHandle(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("AcquireIntegratedDisplayLayerSystemSharedBufferHandle(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->AcquireSystemSharedBufferHandle(h);
}

nn::vi::LayerId GetIntegratedDisplayLayerLayerId(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("GetIntegratedDisplayLayerLayerId(h=%lld)\n", h);
    return g_pIntegratedDisplayManager->GetDirectLayerId(h);
}

nn::vi::IndirectProducerHandleType GetIntegratedDisplayLayerIndirectProducerHandle(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("GetIntegratedDisplayLayerIndirectProduverHandle(h=%lld)\n", h);
    return g_pIntegratedDisplayManager->GetIndirectProducerHandle(h);
}

nn::vi::fbshare::SharedLayerHandle GetIntegratedDisplayLayerSharedLayerHandle(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("GetIntegratedDisplayLayerSharedLayerHandle(h=%lld)\n", h);
    return g_pIntegratedDisplayManager->GetSharedLayerHandle(h);
}

void SetIntegratedDisplayLayerVisibility(IntegratedDisplayLayerHandle h, bool isVisible, int32_t zOrder, bool isCaptureTarget) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetIntegratedDisplayLayerVisibility(h=%lld,visible=%s,z=%d,capture=%s)\n", h, isVisible ? "y" : "n", zOrder, isCaptureTarget ? "y" : "n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetLayerVisibility(h, isVisible, zOrder, isCaptureTarget);
}

void SetIntegratedDisplayLayerTransitionLayerRequested(IntegratedDisplayLayerHandle h, bool value) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetIntegratedDisplayLayerTransitionLayerRequested(h=%lld,v=%s)\n", h, value ? "y" : "n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetLayerTransitionLayerRequested(h, value);
}

void SetIntegratedDisplayLayerAsConductorWhileForeground(IntegratedDisplayLayerHandle h, bool isConductor) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetIntegratedDisplayLayerAsConductorWhileForeground(h=%lld,conductor=%s)\n", h, isConductor ? "y" : "n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetLayerConductorIfForeground(h, isConductor);
}

void EnableIntegratedDisplayLayerToRecordingLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("EnableIntegratedDisplayLayerToRecordingLayerStackForApplication(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetRecordingLayerStackEnabled(h, true);
}

void DisableIntegratedDisplayLayerToRecordingLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("DisableIntegratedDisplayLayerToRecordingLayerStackForApplication(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetRecordingLayerStackEnabled(h, false);
}

void EnableIntegratedDisplayLayerToDebugLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("EnableIntegratedDisplayLayerToDebugLayerStackForApplication(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetDebugLayerStackEnabled(h, true);
}

void DisableIntegratedDisplayLayerToDebugLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("DisableIntegratedDisplayLayerToDebugLayerStackForApplication(h=%lld)\n", h);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetDebugLayerStackEnabled(h, false);
}

//---------------

void SetSystemLayerVisible(bool isVisible) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetSystemLayerVisible()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetSystemLayerVisibility(isVisible);
}

void ClearTransitionLayerImage(Bit32 clearColor) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ClearTransitionLayerImage()\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pIntegratedDisplayManager->ClearTransitionBufferImage(clearColor));
}

//---------------

void BeginOperationModeDisplay() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("BeginOperationModeDisplay()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->BeginOperationModeLayer();
}

void EndOperationModeDisplay() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("EndOperationModeDisplay()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->EndOperationModeLayer();
}

void SetOperationModeTexture(int x, int y, int width, int height, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetOperationModeTexture()\n");
    g_pIntegratedDisplayManager->SetOperationModeTexture(x, y, width, height, buffer, size);
}

void ShowOperationModeTexture() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ShowOperationModeTexture()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->ShowOperationModeTexture();
}

void ShowOperationModeStartupLogoTexture() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ShowOperationModeStartupLogoTexture()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->ShowOperationModeStartupLogoTexture();
}

void HideOperationModeTexture() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("HideOperationModeTexture()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->HideOperationModeTexture();
}

//---------------

nn::Result InitializeApplicationCopyrightSharedBuffer(nn::sf::NativeHandle& hTrmem, size_t trmemSize, int width, int height) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("InitializeApplicationCopyrightSharedBuffer()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->InitializeApplicationCopyrightSharedBuffer(std::move(hTrmem), trmemSize, width, height);
}

void FinalizeApplicationCopyrightSharedBuffer() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("FinalizeApplicationCopyrightSharedBuffer()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->FinalizeApplicationCopyrightSharedBuffer();
}

nn::Result SetApplicationCopyrightTexture(int x, int y, int width, int height, const void* buffer, size_t size, int originMode) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetApplicationCopyrightTexture()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->SetApplicationCopyrightTexture(x, y, width, height, buffer, size, originMode);
}

bool IsApplicationCopyrightTextureSet() NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("IsApplicationCopyrightTextureSet()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->IsApplicationCopyrightTextureSet();
}

void GetApplicationCopyrightInfoForScreenShot(nn::capsrv::ScreenShotAttribute* pOutValue) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("GetApplicationCopyrightInfoForScreenShot()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->GetApplicationCopyrightInfoForScreenShot(pOutValue);
}

void SetApplicationCopyrightVisibility(bool isVisible) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("SetApplicationCopyrightVisibility()\n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_pIntegratedDisplayManager->SetApplicationCopyrightVisibility(isVisible);
}

//---------------

void RequestScreenShotTask(const display::ScreenShotRequest& request) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_LOG_CALL("RequestScreenShotTask(tick=%lld)\n", request.GetRequestTick());
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    g_ScreenShotRequestQueue.EnqueueRequest(request);
}

void ScreenShotRequestCallbackFunction(const display::ScreenShotRequest& request, void*) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();

    switch(request.GetRequestType())
    {
    case display::ScreenShotRequest::RequestType_TakingScreenShot:
        return g_pIntegratedDisplayManager->RequestTakingScreenShot(request);
    case display::ScreenShotRequest::RequestType_NotifyingRefused:
        return g_pIntegratedDisplayManager->NotifyTakingScreenShotRefused(request);
    case display::ScreenShotRequest::RequestType_NotifyingFailure:
        return g_pIntegratedDisplayManager->NotifyTakingScreenShotFailed(request);
    default: return;
    }
}


//-------------------------------------
// キャプチャバッファAPI
//-------------------------------------

nn::Result AcquireCaptureBufferIndex(bool* pOutIsScreenShotEnabled, int* pOutBufferIndex, IntegratedDisplayLayerHandle handle, CaptureBufferIndex index) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("AcquireCaptureBufferIndex(h=%lld,index=%d)\n", handle, index);
    //NN_AM_DISPCTRL_DUMP_ON_EXIT();
    auto result = g_pIntegratedDisplayManager->AcquireCaptureBuffer(pOutIsScreenShotEnabled, pOutBufferIndex, handle, index);
    NN_AM_DISPCTRL_LOG_CALL("  -> result=%d-%d, iBuf=%d, screenshot=%s\n", result.GetModule(), result.GetDescription(), *pOutBufferIndex, (*pOutIsScreenShotEnabled ? "ok" : "ng"));
    return result;
}

nn::Result ReleaseCaptureBufferIndex(IntegratedDisplayLayerHandle handle, CaptureBufferIndex index) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ReleaseCaptureBufferIndex(h=%lld,index=%d)\n", handle, index);
    //NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->ReleaseCaptureBuffer(handle, index);
}

nn::Result CopyImageFromCaptureBuffer(bool* pOutIsScreenShotEnabled, void* dst, size_t dstSize, CaptureBufferIndex index) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    //NN_AM_DISPCTRL_DUMP_ON_EXIT();
    NN_AM_DISPCTRL_LOG_CALL("CopyImageFromCaptureBuffer(index=%d)\n", index);
    auto result = g_pIntegratedDisplayManager->ReadCaptureBufferImage(pOutIsScreenShotEnabled, dst, dstSize, index);
    NN_AM_DISPCTRL_LOG_CALL("  -> result=%d-%d, screenshot=%s\n", result.GetModule(), result.GetDescription(), (*pOutIsScreenShotEnabled ? "ok" : "ng"));
    return result;
}
nn::Result ClearCaptureBufferImage(CaptureBufferIndex index, Bit32 color, bool isCaptureEnabled) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("ClearCaptureBuffer(index=%d,0x%08X,ss=%s)\n", index, color, isCaptureEnabled ? "y" : "n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->ClearCaptureBufferImage(index, color, isCaptureEnabled);
}

nn::Result TryCaptureCurrentForegroundImage(CaptureBufferIndex index) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("TryCaptureCurrentForegroundImage(index=%d)\n", index);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->TryCaptureCurrentForegroundImage(index);
}

nn::Result TryCaptureCurrentLayerImage(CaptureBufferIndex index, IntegratedDisplayLayerHandle handle, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("TryCaptureCurrentLayerImage(index=%d,h=%lld,ss=%s,imm=%s)\n", index, handle, isScreenShotPermitted ? "y" : "n", isImmediate ? "y" : "n");
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->TryCaptureCurrentLayerImage(index, handle, isScreenShotPermitted, isImmediate);
}

nn::Result CopyBetweenCaptureBuffersImpl(CaptureBufferIndex dstIndex, CaptureBufferIndex srcIndex) NN_NOEXCEPT
{
    NN_AM_DISPCTRL_SCOPED_LOCK_MUTEX();
    NN_AM_DISPCTRL_LOG_CALL("CopyBetweenCaptureBuffersImpl(dstIndex=%d, srcIndex=%d)\n", dstIndex, srcIndex);
    NN_AM_DISPCTRL_DUMP_ON_EXIT();
    return g_pIntegratedDisplayManager->CopyCaptureBufferImage(dstIndex, srcIndex);
}

//-------------------------------------
// サポート終了
//-------------------------------------

bool TryAcquireCaptureBufferSemaphore() NN_NOEXCEPT
{
    return true;
}

void ReleaseCaptureBufferSemaphore() NN_NOEXCEPT
{
}

// キャプチャバッファを TransferMemory 化して返す。
nn::Result AcquireCaptureBufferTransferMemory(bool* pOutIsScreenShotEnabled, nn::sf::Out<nn::sf::NativeHandle> pOutTransferMemory, CaptureBufferIndex index) NN_NOEXCEPT
{
    NN_UNUSED(pOutIsScreenShotEnabled);
    NN_UNUSED(pOutTransferMemory);
    NN_UNUSED(index);
    NN_RESULT_THROW(nn::applet::ResultCaptureBufferOperationNotSupported());
}

nn::Result ReleaseCaptureBufferTransferMemory(CaptureBufferIndex index) NN_NOEXCEPT
{
    NN_UNUSED(index);
    NN_RESULT_SUCCESS;
}

}}}
