﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>

#include <mutex>
#include "am_AudioControl.h"

namespace nn { namespace am { namespace service {

namespace {

AudioControl g_AudioController;

}   // namespace

//-----------------------------------------------------------------------------

void AudioControl::AppletInfoForAudioControl::UpdateAudioSuspendState() NN_NOEXCEPT
{
    bool needSuspend = m_SuspendAudioForDebugEnabled || (m_SuspendCount > 0);
    if (needSuspend == m_PreviousNeedSuspend)
    {
        return;
    }

    if (needSuspend)
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(audio_SuspendAudio, 60);
        NN_AM_SERVICE_LOG(call, "RequestSuspendAudio() ARUID=%lld count=%lld debug=%s\n", m_Aruid, m_SuspendCount, m_SuspendAudioForDebugEnabled ? "true" : "false");
        // Audio をサスペンドする
        static const auto fadeTime = TimeSpan::FromMilliSeconds(10);

        audio::RequestSuspendAudioIns(m_Aruid, fadeTime);
        audio::RequestSuspendAudioOuts(m_Aruid, fadeTime);
        audio::RequestSuspendAudioRenderers(m_Aruid, fadeTime);
    }
    else
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(audio_ResumeAudio, 60);
        NN_AM_SERVICE_LOG(call, "RequestResumeAudio() ARUID=%lld count=%lld debug=%s\n", m_Aruid, m_SuspendCount, m_SuspendAudioForDebugEnabled ? "true" : "false");
        // Audio をレジュームする
        static const auto fadeTime = TimeSpan::FromMilliSeconds(10);

        audio::RequestResumeAudioIns(m_Aruid, fadeTime);
        audio::RequestResumeAudioOuts(m_Aruid, fadeTime);
        audio::RequestResumeAudioRenderers(m_Aruid, fadeTime);
    }
    m_PreviousNeedSuspend = needSuspend;
}

//-----------------------------------------------------------------------------

AudioControl::AppletInfoForAudioControl* AudioControl::FindAruidAndRegisterUnsafe(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // 既に登録済みのものがあればそれを返す
    for (auto&& p : m_List)
    {
        if (p.IsEqual(aruid))
        {
            return &p;
        }
    }
    // 空きスロットがあれば登録してそれを返す
    for (auto&& p : m_List)
    {
        if (p.IsEmpty())
        {
            p.Set(aruid);
            return &p;
        }
    }
    return nullptr;
}

AudioControl::AppletInfoForAudioControl* AudioControl::FindAruidUnsafe(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (auto&& p : m_List)
    {
        if (p.IsEqual(aruid))
        {
            return &p;
        }
    }
    return nullptr;
}

bool AudioControl::Register(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
    return FindAruidAndRegisterUnsafe(aruid) ? true : false;
}

bool AudioControl::Unregister(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
    auto p = FindAruidUnsafe(aruid);
    if (!p)
    {
        return false;
    }
    p->Clear();
    return true;
}

void AudioControl::SuspendAudio(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
    auto p = FindAruidAndRegisterUnsafe(aruid);
    if (p)
    {
        p->IncrementSuspendCount();
        p->UpdateAudioSuspendState();
    }
}

void AudioControl::ResumeAudio(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
    auto p = FindAruidAndRegisterUnsafe(aruid);
    if (p)
    {
        p->DecrementSuspendCount();
        p->UpdateAudioSuspendState();
    }
}

void AudioControl::SuspendAudioForDebug(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
#if 0
    {
        auto p = FindAruidAndRegisterUnsafe(aruid);
        if (p)
        {
            p->SetSuspendAudioForDebugEnabled(true);
            p->UpdateAudioSuspendState();
        }
    }
#else
    // audio 側には ForDebug ありとなしの API が既に用意されているものの、
    // これらの発行順序に制限があるため、現時点ではこちらは使用しない。
    {
        audio::RequestSuspendAudioInsForDebug(aruid);
        audio::RequestSuspendAudioOutsForDebug(aruid);
        audio::RequestSuspendAudioRenderersForDebug(aruid);
    }
#endif
}

void AudioControl::ResumeAudioForDebug(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
#if 0
    {
        auto p = FindAruidAndRegisterUnsafe(aruid);
        if (p)
        {
            p->SetSuspendAudioForDebugEnabled(false);
            p->UpdateAudioSuspendState();
        }
    }
#else
    // audio 側には ForDebug ありとなしの API が既に用意されているものの、
    // これらの発行順序に制限があるため、現時点ではこちらは使用しない。
    {
        audio::RequestResumeAudioInsForDebug(aruid);
        audio::RequestResumeAudioOutsForDebug(aruid);
        audio::RequestResumeAudioRenderersForDebug(aruid);
    }
#endif
}

//-----------------------------------------------------------------------------
// 対象 ARUID の Audio 制御対象を登録する
//
void RegisterAruidForAudioControl(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (g_AudioController.Register(aruid))
    {
        NN_AM_SERVICE_LOG(call, "Registered ARUID=%lld for audio control.\n", aruid);
    }
    else
    {
        NN_AM_SERVICE_LOG(call, "Cannot register ARUID=%lld for audio control.\n", aruid);
    }
}

//-----------------------------------------------------------------------------
// 対象 ARUID の Audio 制御対象を削除する
//
void UnregisterAruidForAudioControl(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (g_AudioController.Unregister(aruid))
    {
        NN_AM_SERVICE_LOG(call, "Unregistered ARUID=%lld for audio control.\n", aruid);
    }
    else
    {
        NN_AM_SERVICE_LOG(call, "Cannot unregister ARUID=%lld for audio control.\n", aruid);
    }
}

//-----------------------------------------------------------------------------
// 対象 ARUID の Audio を Suspend する（呼出回数カウンタに対応）
//
void SuspendAudioInProcess(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    g_AudioController.SuspendAudio(aruid);
}

//-----------------------------------------------------------------------------
// 対象 aruid の Audio を Resume する（呼出回数カウンタに対応）
//
void ResumeAudioInProcess(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    g_AudioController.ResumeAudio(aruid);
}

//-----------------------------------------------------------------------------
// 対象 pid の Audio を Suspend する（デバッグ通知用、カウンタなし）
//
void SuspendAudioInProcessForDebug(os::ProcessId pid) NN_NOEXCEPT
{
    g_AudioController.SuspendAudioForDebug({pid.value});
}

//-----------------------------------------------------------------------------
// 対象 pid の Audio を Resume する（デバッグ通知用、カウンタなし）
//
void ResumeAudioInProcessForDebug(os::ProcessId pid) NN_NOEXCEPT
{
    g_AudioController.ResumeAudioForDebug({pid.value});
}

}}}
