﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Span.h>

#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>

#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Result.h>

#include <nn/ae/ae_SystemAppletApi.h>
#include <nn/ae/ae_Result.h>
#include <nn/ae/ae_Types.h>

#include <nn/lr/lr_Result.h>
#include <nn/audio/audio_Applet.h>

#include <nv/nv_ServiceName.h>


namespace nn { namespace ae {

//-----------------------------------------------------------------------------

namespace {

    SystemAppletParameters  g_SystemAppletParameters;

    os::SystemEventType g_SystemGeneralChannelEvent;

}   // namespace anonymous


//-----------------------------------------------------------------------------
//  システムアプレットの初期化
//
void InvokeSystemAppletMain(AppletId appletId, SystemAppletMainFunction function) NN_NOEXCEPT
{
    nv::SetGraphicsServiceName("nvdrv:a");

    // TORIAEZU: TODO:
    // 既存 SA がいる場合に、その SA を終了してから自身の初期化を完了させる。
    // この処理は Proxy 側で行なうべきで、ここにあったコードは削除済みだが、
    // Proxy 側の対応がなされるまで、あえてコメントを残しておく。
    am::InitializeSystemAppletInterfaces();

    {
        sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetHomeMenuFunctions()->GetPopFromGeneralChannelEvent(&eventHandle));
        os::AttachReadableHandleToSystemEvent(&g_SystemGeneralChannelEvent, eventHandle.GetOsHandle(), eventHandle.IsManaged(), os::EventClearMode_ManualClear);
        eventHandle.Detach();
    }

    // SA は自動で fore を要求する
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetHomeMenuFunctions()->RequestToGetForeground());

    // アプレットパラメータの設定
    std::memset(&g_SystemAppletParameters, 0, sizeof(SystemAppletParameters));

    // アプレットのメイン関数の呼出し。
    // この後、FG 起動なら BG->FG 遷移要求が自動的に通知されてくる
    function( &g_SystemAppletParameters );

    os::DestroySystemEvent(&g_SystemGeneralChannelEvent);

    // 終了処理
    am::FinalizeSystemAppletInterfaces();
}


//-----------------------------------------------------------------------------
//  本体起動理由を返す
//
BootReason GetBootReason() NN_NOEXCEPT
{
    return BootReason_ColdBoot;
}


//-----------------------------------------------------------------------------
//  システムアプレットが Foreground へ遷移できるように要求を出す
//
Result RequestToGetForeground() NN_NOEXCEPT
{
    auto result = am::GetHomeMenuFunctions()->RequestToGetForeground();
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( applet::ResultAlreadyForeground )
        {
            NN_RESULT_THROW( ae::ResultAlreadyForeground() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// 強制的に FG 状態に遷移し、スクリーンをロックする。
//
void LockForeground() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetHomeMenuFunctions()->LockForeground() );
}


//-----------------------------------------------------------------------------
// スクリーンロックを解除し、強制 FG 状態を開放する。
//
void UnlockForeground() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetHomeMenuFunctions()->UnlockForeground() );
}


//-----------------------------------------------------------------------------

bool TryPopFromSystemGeneralChannel(applet::StorageHandle* pOut) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IStorage> ret;
    NN_RESULT_TRY(am::GetHomeMenuFunctions()->PopFromGeneralChannel(&ret))
        NN_RESULT_CATCH(am::ResultNoStorage)
        {
            return false;
        }
    NN_RESULT_END_TRY
    *pOut = {ret.Detach()};
    return true;
}

os::SystemEventType* GetPopFromSystemGeneralChannelEvent() NN_NOEXCEPT
{
    return &g_SystemGeneralChannelEvent;
}

void NotifyCecSettingsChanged() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetGlobalStateController()->NotifyCecSettingsChanged());
}


void SetDefaultHomeButtonLongPressTime(TimeSpan time) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetGlobalStateController()->SetDefaultHomeButtonLongPressTime(time.GetNanoSeconds()));
}

bool ShouldSleepOnBoot() NN_NOEXCEPT
{
    bool isShouldSleepOnBoot;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetGlobalStateController()->ShouldSleepOnBoot(&isShouldSleepOnBoot));
    return isShouldSleepOnBoot;
}

bool PopRequestLaunchApplicationForDebug(ncm::ApplicationId* pApplicationId, account::Uid* pUid) NN_NOEXCEPT
{
    ncm::ApplicationId applicationId;
    account::Uid uidBuffer[account::UserCountMax];
    auto uids = util::MakeSpan(uidBuffer);
    int n;
    NN_RESULT_TRY(am::GetHomeMenuFunctions()->PopRequestLaunchApplicationForDebug(&applicationId, &n, uids))
        NN_RESULT_CATCH(am::ResultNoData)
        {
            return false;
        }
    NN_RESULT_END_TRY
    *pApplicationId = applicationId;
    *pUid = n > 0 ? uids[0] : account::InvalidUid;
    return true;
}

}}  // namespace nn::ae

