﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/profile/account_ProfileTypes.h>

#include "../detail/account_PathUtil.h"
#include "../detail/account_UuidUtil.h"
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_SdkLog.h>

namespace nn { namespace account { namespace profile {

struct SaveDataHeader
{
    static const uint32_t Magic     = 0xa4871acd; // NOTE: 不変
    static const uint32_t Version   = 0x01606150; // NOTE: yyymmdd#
    static const uint64_t Header    = static_cast<uint64_t>(Magic) << 32 | Version;

    uint64_t header;
    uint64_t zero;
};
static_assert(std::alignment_of<SaveDataHeader>::value <= std::alignment_of<std::max_align_t>::value, "alignof(SaveDataHeader) > alignof(max_align_t)");
struct SaveDataEntry
{
    detail::SerializedUuid user;
    detail::SerializedUuid author;
    uint64_t timeStamp;
    char nickname[NicknameBytesMax];
    char userData[UserDataBytesMax];
};
static_assert(std::alignment_of<SaveDataEntry>::value <= std::alignment_of<std::max_align_t>::value, "alignof(SaveDataEntry) > alignof(max_align_t)");

const SaveDataEntry EmptyEntry = {{}, {}, 0x00ull, "", {}};

class SaveDataAccessor
{
private:
    static bool IsContained(const Uid& u, const Uid* users, int count)
    {
        for (auto i = 0; i < count && users[i]; ++ i)
        {
            if (users[i] == u)
            {
                return true;
            }
        }
        return false;
    }

    static Result LoadImpl(
        Profile *outProfiles, int profileCount,
        void *buffer, size_t bufferSize,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(profileCount == UserCountMax);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<std::max_align_t>::value == 0, "alignof(buffer) == alignof(std::max_align_t)");
#endif
        NN_SDK_ASSERT(bufferSize >= sizeof(SaveDataHeader), "sizeof(buffer) >= sizeof(SaveDataHeader)");
        NN_SDK_ASSERT(bufferSize >= sizeof(SaveDataEntry), "sizeof(buffer) >= sizeof(SaveDataEntry)");
        NN_UNUSED(bufferSize);

        auto& fs = storage.GetFileSystem();
        char path[64];
        detail::PathUtil::GetProfileListPath(path, sizeof(path), storage.GetRootPath());

        size_t size;
        NN_RESULT_DO(fs.GetSize(&size, path));
        NN_RESULT_THROW_UNLESS(
            size == sizeof(SaveDataHeader) + sizeof(SaveDataEntry) * UserCountMax,
            ResultSaveDataBroken());

        {
            SaveDataHeader& header = *reinterpret_cast<SaveDataHeader*>(buffer);
            NN_RESULT_DO(fs.Read(&size, &header, sizeof(SaveDataHeader), path));
            NN_RESULT_THROW_UNLESS(detail::ConvertNetworkToHost(header.header) == SaveDataHeader::Header, ResultSaveDataBroken());
            NN_RESULT_THROW_UNLESS(header.zero == 0x00ull, ResultSaveDataBroken());
        }

        SaveDataEntry& entry = *reinterpret_cast<SaveDataEntry*>(buffer);
        int filledCount = 0;
        for (auto i = 0; i < profileCount; ++ i)
        {
            NN_RESULT_DO(fs.Read(&size, &entry, sizeof(SaveDataEntry), path, sizeof(SaveDataHeader) + i * sizeof(SaveDataEntry)));

            auto& p = outProfiles[filledCount];
            p.user = detail::ConvertToUid(entry.user.Deserialize());
            if (p.user)
            {
                p.base.author = detail::ConvertToUid(entry.author.Deserialize());
                p.base.timeStamp = detail::ConvertNetworkToHost(entry.timeStamp);
                std::memcpy(p.base.nickname, entry.nickname, sizeof(p.base.nickname));
                std::memcpy(p.userData.data, entry.userData, sizeof(p.userData.data));
                ++ filledCount;
            }
        }
        while (filledCount < profileCount)
        {
            outProfiles[filledCount ++].Clear();
        }
        NN_RESULT_SUCCESS;
    }
public:
    static Result Load(
        Profile* outProfiles, int profileCount,
        void *buffer, size_t bufferSize,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        auto r = LoadImpl(outProfiles, profileCount, buffer, bufferSize, storage);
        if (fs::ResultPathNotFound::Includes(r))
        {
            for (auto i = 0; i < profileCount; ++ i)
            {
                outProfiles[i].Clear();
            }
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(r);
        NN_RESULT_SUCCESS;
    }
    static Result Store(
        const Profile* profiles, int profileCount,
        void *buffer, size_t bufferSize,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(profileCount == UserCountMax);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<std::max_align_t>::value == 0, "alignof(buffer) == alignof(std::max_align_t)");
#endif
        NN_SDK_ASSERT(bufferSize >= sizeof(SaveDataHeader), "sizeof(buffer) >= sizeof(SaveDataHeader)");
        NN_SDK_ASSERT(bufferSize >= sizeof(SaveDataEntry), "sizeof(buffer) >= sizeof(SaveDataEntry)");
        NN_UNUSED(bufferSize);

        auto& fs = storage.GetFileSystem();
        char path[64];
        detail::PathUtil::GetProfileListPath(path, sizeof(path), storage.GetRootPath());

        {
            SaveDataHeader& header = *reinterpret_cast<SaveDataHeader*>(buffer);
            header.header = detail::ConvertHostToNetwork(SaveDataHeader::Header);
            header.zero = 0x00ull;
            NN_RESULT_DO(fs.Write(path, &header, sizeof(SaveDataHeader)));
        }
        SaveDataEntry& entry = *reinterpret_cast<SaveDataEntry*>(buffer);
        for (auto i = 0; i < profileCount; ++ i)
        {
            const auto& p = profiles[i];
            if (p)
            {
                entry.user = detail::ConvertToUuid(p.user).Serialize();
                entry.author = detail::ConvertToUuid(p.base.author).Serialize();
                entry.timeStamp = detail::ConvertHostToNetwork(p.base.timeStamp);
                std::memcpy(entry.nickname, p.base.nickname, sizeof(entry.nickname));
                std::memcpy(entry.userData, p.userData.data, sizeof(entry.userData));
            }
            else
            {
                entry = EmptyEntry;
            }
            NN_RESULT_DO(fs.Append(path, sizeof(SaveDataHeader) + i * sizeof(SaveDataEntry), &entry, sizeof(SaveDataEntry)));
            size_t size;
            NN_RESULT_DO(fs.Read(&size, &entry, sizeof(SaveDataEntry), path, sizeof(SaveDataHeader) + i * sizeof(SaveDataEntry)));
        }
        NN_RESULT_SUCCESS;
    }

    static Result StoreImage(
        const Uid& uid, const void* image, size_t size,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        return storage.GetFileSystem().Write(
            detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, storage.GetRootPath()), image, size);
    }
    static Result GetImageSizeForApplications(
        size_t* pOutSize, const Uid& uid,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        size_t size;
        NN_RESULT_TRY(storage.GetFileSystem().GetSize(
            &size, detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, storage.GetRootPath())))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultUserNotExist());
            }
        NN_RESULT_END_TRY
        *pOutSize = size;
        NN_RESULT_SUCCESS;
    }
    static Result LoadImageForApplications(
        size_t* pOutActualSize, void* pOut, size_t size, const Uid& uid,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        size_t actualSize;
        NN_RESULT_TRY(storage.GetFileSystem().Read(
            &actualSize, pOut, size, detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, storage.GetRootPath())))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultUserNotExist());
            }
        NN_RESULT_END_TRY
        *pOutActualSize = actualSize;
        NN_RESULT_SUCCESS;
    }
    static void DeleteImage(const Uid& uid, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        auto r = storage.GetFileSystem().Delete(
            detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, storage.GetRootPath()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
};

}}} // ~namespace nn::account::profile
