﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/profile/account_ProfileStorage.h>

#include "account_SaveDataAccessor.h"
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_Types.h>

#include <mutex>
#include <type_traits>

#include <nn/nn_Result.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace account { namespace profile {

namespace {
uint64_t UpdateModifier(uint64_t modifier) NN_NOEXCEPT
{
    // それっぽい値になればよい
    const uint64_t Mask = 0xf4ec5eed7cc2486cull;
    return (((modifier << 23) | (modifier >> 41)) + 1) ^ Mask;
}
} // ~namespace nn::account::profile::<anonymous>

/** -------------------------------------------------------------------------------------------
    内部で使う
 */

void ProfileStorage::DeleteUnsafe(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    for (auto& p: m_Profiles)
    {
        if (p.user == uid)
        {
            p.Clear();
            return;
        }
    }
    // 削除対象がいなくても不問とする
}
void ProfileStorage::AddUnsafe(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(
        [&]() -> bool
        {
            for (auto& p : m_Profiles)
            {
                if (p.user == uid)
                {
                    return false;
                }
            }
            return true;
        }());
    for (auto i = 0; i < std::extent<decltype(m_Profiles)>::value; ++ i)
    {
        auto& p = m_Profiles[i];
        if (!p.user)
        {
            p.ClearWith(uid);
            m_Modifier[i] = 0;
            return;
        }
    }
    NN_ABORT(
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: No more slot to store profile (internal DB broken)\n");
}
Result ProfileStorage::UpdateImplUnsafe(
    const Uid& uid,
    const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(profileBase);

    int index;
    NN_RESULT_DO(FindUnsafe(&index, uid));

    Profile& profile = m_Profiles[index];
    profile.base = profileBase;
    profile.userData = userData;

    auto lock = m_pStorage->AcquireWriterLock();
    if (pImage)
    {
        NN_SDK_ASSERT(size > 0);
        NN_RESULT_DO(SaveDataAccessor::StoreImage(uid, pImage, size, *m_pStorage));
    }
    StoreSaveData();
    m_Modifier[index] = UpdateModifier(m_Modifier[index]);
    NN_RESULT_SUCCESS;
}
Result ProfileStorage::ApplyExternalDataUnsafe(
    const Uid& uid,
    const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size,
    bool force) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(profileBase);

    int index;
    NN_RESULT_DO(FindUnsafe(&index, uid));

    Profile& profile = m_Profiles[index];
    NN_RESULT_THROW_UNLESS(
        force || profileBase.timeStamp >= profile.base.timeStamp,
        ResultProfileObsoleted());
    profile.base = profileBase;
    profile.userData = userData;

    auto lock = m_pStorage->AcquireWriterLock();
    if (pImage)
    {
        NN_SDK_ASSERT(size > 0);
        NN_RESULT_DO(SaveDataAccessor::StoreImage(uid, pImage, size, *m_pStorage));
    }
    else
    {
        SaveDataAccessor::DeleteImage(uid, *m_pStorage);
    }
    StoreSaveData();
    m_Modifier[index] = UpdateModifier(m_Modifier[index]);
    NN_RESULT_SUCCESS;
}
void ProfileStorage::StoreSaveData() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(SaveDataAccessor::Store(
        m_Profiles, sizeof(m_Profiles) / sizeof(m_Profiles[0]),
        &m_IoBuffer, sizeof(m_IoBuffer),
        *m_pStorage));
}
Result ProfileStorage::FindUnsafe(int* pOut, const Uid& uid) const NN_NOEXCEPT
{
    for (auto i = 0; i < std::extent<decltype(m_Profiles)>::value; ++ i)
    {
        if (m_Profiles[i].user == uid)
        {
            *pOut = i;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

/** -------------------------------------------------------------------------------------------
    サービス起動, 終了時の処理
    */
Result ProfileStorage::Initialize(
    const Uid* users, int userCount,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Initialized);

    NN_RESULT_DO(SaveDataAccessor::Load(
        m_Profiles, sizeof(m_Profiles) / sizeof(m_Profiles[0]),
        &m_IoBuffer, sizeof(m_IoBuffer),
        storage));

    // いるべきユーザーがいるか？
    // (nn::account::DeleteUser() の途中で終了した場合などが該当する。)
    for (auto i = 0; i < userCount && users[i]; ++ i)
    {
        bool missing = true;
        for (const auto& p: m_Profiles)
        {
            if (p.user == users[i])
            {
                missing = false;
                break;
            }
        }
        if (missing)
        {
            std::lock_guard<decltype(m_Lock)> lock(m_Lock);
            AddUnsafe(users[i]);
        }
    }
    m_pStorage = &storage;
    m_Initialized = true;
    NN_RESULT_SUCCESS;
};

/** -------------------------------------------------------------------------------------------
    UserRegistration と連動する処理
 */

void ProfileStorage::Add(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    AddUnsafe(uid);
}

// Unregister はこれのあとに呼ぶ
void ProfileStorage::Delete(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    auto lock2 = m_pStorage->AcquireWriterLock();
    SaveDataAccessor::DeleteImage(uid, *m_pStorage);
    DeleteUnsafe(uid);
    StoreSaveData();
}

/** -------------------------------------------------------------------------------------------
    ProfileUpdateNotifier
 */

void ProfileStorage::ReleaseSystemEvent(const os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_SDK_ASSERT(pEvent);

    Notifiable::ReleaseSystemEvent(pEvent);
}

nn::Result ProfileStorage::GetProfileUpdateEvent(os::SystemEventType** pOutEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    return Notifiable::AcquireSystemEvent(pOutEvent, ProfileStorageTag_Update);
}

/** -------------------------------------------------------------------------------------------
    IProfile と ProfileAdaptor で呼ばれる機能の実体
 */

Result ProfileStorage::GetDigest(ProfileDigest* pOut, const Uid& uid) const NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(*pOut) == sizeof(uint64_t) * 2);

    NN_SDK_ASSERT(m_Initialized);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    int index;
    NN_RESULT_DO(FindUnsafe(&index, uid));

    uint64_t raw[2] = {m_Modifier[index], m_Profiles[index].base.timeStamp};
    std::memcpy(pOut, raw, sizeof(*pOut));
    NN_RESULT_SUCCESS;
}
Result ProfileStorage::GetProfile(ProfileBase* pOutProfileBase, UserData* pOutUserData, const Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    int index;
    NN_RESULT_DO(FindUnsafe(&index, uid));

    *pOutProfileBase = m_Profiles[index].base;
    if (pOutUserData != nullptr)
    {
        *pOutUserData = m_Profiles[index].userData;
    }
    NN_RESULT_SUCCESS;
}
Result ProfileStorage::GetImageSize(size_t* pOutSize, const Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return SaveDataAccessor::GetImageSizeForApplications(pOutSize, uid, *m_pStorage);
}
Result ProfileStorage::LoadImage(size_t* pOutActualSize, void* pOut, size_t size, const Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return SaveDataAccessor::LoadImageForApplications(pOutActualSize, pOut, size, uid, *m_pStorage);
}
Result ProfileStorage::Update(const Uid& uid, const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_SDK_ASSERT(uid);
    NN_RESULT_THROW_UNLESS(profileBase.author == uid, ResultBadProfileAuthor());
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_RESULT_DO(UpdateImplUnsafe(uid, profileBase, userData, pImage, size));
    Notifiable::SignalEvents(ProfileStorageTag_Update);
    NN_RESULT_SUCCESS;
}
Result ProfileStorage::ApplyExternalData(const Uid& uid, const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size, bool force) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    NN_SDK_ASSERT(uid);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_RESULT_DO(ApplyExternalDataUnsafe(uid, profileBase, userData, pImage, size, force));
    NN_RESULT_SUCCESS;
}

}}}
