﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "account_NasProblemDetails.h"
#include "../http/account_RedirectUriParser.h"
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/json/account_JsonAdaptor.h>
#include <nn/account/json/account_RapidJsonStream.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/nas/account_NasUserResourceCache.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_Result.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn {
namespace account {
namespace nas {

/*
    - User リソース定義, (screenName)
      https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/schema/1.0.0/resources/user.json
    - 言語, タイムゾーン
      https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/schema/1.0.0/common_types.json
    - ニックネーム, メールアドレス, 性別, ログイン ID
      https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/schema/1.0.0/accounts/common.json
    - 国
      https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/schema/1.0.0/country.json
    - 地域
      https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/schema/1.0.0/region.json
    - Mii
      https://e03a97819c9711e59510d820a52f298a-sb.accounts.nintendo.com/schema/1.0.0/resources/mii.json
*/

// システムに User リソースのキャッシュを作成するためのアダプタ。
// このアダプタ自体は NintendoAccountId 以外のデータを返さない。
//
class NasUserAdaptor final
    : public NasCommonAdaptor<12, 64>
{
    NN_DISALLOW_COPY(NasUserAdaptor);

private:
    static const size_t SizeToReserve = 1024;

    const detail::AbstractLocalStorage& m_Storage;

    json::LookupEntry m_IdEntry;
    json::LookupEntry m_IsChildEntry;
    json::LookupEntry m_ScreenNameEntry;
    json::LookupEntry m_EmailEntry; // null の場合もある
    json::LookupEntry m_LoginIdEntry; // null の場合もある
    json::LookupEntry m_NicknameEntry;
    json::LookupEntry m_BirthdayEntry;
    json::LookupEntry m_GenderEntry;
    json::LookupEntry m_AnalyticsPermissionsForTargetMarketingEntry;
    json::LookupEntry m_AnalyticsPermissionsForInternalAnalysisEntry;
    json::LookupEntry m_LanguageEntry;
    json::LookupEntry m_CountryEntry;
    json::LookupEntry m_RegionEntry; // null の場合もある
    json::LookupEntry m_TimezoneEntry;

    json::LookupEntry m_NnIdEntry;
    json::LookupEntry m_NnCreatedAtEntry;
    json::LookupEntry m_TwitterIdEntry;
    json::LookupEntry m_TwitterCreatedAtEntry;
    json::LookupEntry m_FacebookIdEntry;
    json::LookupEntry m_FacebookCreatedAtEntry;
    json::LookupEntry m_GoogleIdEntry;
    json::LookupEntry m_GoogleCreatedAtEntry;

    bool m_IsAdapted;
    NintendoAccountId m_Id;
    detail::Uuid m_CacheId;
    char m_Path[128];
    char m_StackBuffer[128];

    json::BufferedFileOutputStreamForRapidJson m_Stream;
    json::FixedSizeOneTimeAllocatorForRapidJson m_StackAllocator;
    json::DefaultRapidJsonFileWriter m_Writer;

public:
    typedef NasCommonAdaptor<12, 64>::JsonPathType JsonPathType;

public:
    NasUserAdaptor(
        const detail::AbstractLocalStorage& storage,
        char* buffer, size_t bufferSize) NN_NOEXCEPT;
    ~NasUserAdaptor() NN_NOEXCEPT;

    Result PullUserResourceCache(NintendoAccountId* pOutNaId, NasUserResourceCache& userResourceCache) NN_NOEXCEPT;

    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, std::nullptr_t) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE;
};

// システムでキャッシュしている User リソースの値を読みだすためのアダプタ。
// デフォルト値をいい感じに埋めてくれるはず。
class NasUserCacheAdaptor
{
    NN_DISALLOW_COPY(NasUserCacheAdaptor);
private:
    json::LookupEntry m_IsChildEntry;
    json::LookupEntry m_ScreenNameEntry;
    json::LookupEntry m_EmailEntry;
    json::LookupEntry m_LoginIdEntry;
    json::LookupEntry m_NicknameEntry;
    json::LookupEntry m_BirthdayEntry;
    json::LookupEntry m_GenderEntry;
    json::LookupEntry m_AnalyticsOptedInEntry;
    json::LookupEntry m_AnalyticsForTargetMarketingPermittedEntry;
    json::LookupEntry m_AnalyticsForInternalAnalysisPermittedEntry;
    json::LookupEntry m_LanguageEntry;
    json::LookupEntry m_CountryEntry;
    json::LookupEntry m_RegionEntry;
    json::LookupEntry m_TimezoneEntry;

    json::LookupEntry m_IsNnLinkedEntry;
    json::LookupEntry m_IsTwitterLinkedEntry;
    json::LookupEntry m_IsFacebookLinkedEntry;
    json::LookupEntry m_IsGoogleLinkedEntry;

    NasUserBase* m_pOutNasUserBase;
public:
    typedef json::JsonPath<8, 64> JsonPathType;
public:
    explicit NasUserCacheAdaptor(NasUserBase* pOut) NN_NOEXCEPT;
    Result Adapt() NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;
    // 以下考慮しない入力
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {} // NOP
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {} // NOP
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {} // NOP
    void Update(const JsonPathType&, double) NN_NOEXCEPT {} // NOP
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {} // NOP
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {} // NOP
};

}
}
} // ~namespace nn::account::nas
