﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasSessionPool.h>

#include "../detail/account_UuidUtil.h"
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>


namespace nn {
namespace account {
namespace nas {

NasSessionPoolBase::NasSessionPoolBase() NN_NOEXCEPT
{
    for (auto& session : m_Sessions)
    {
        session.sessionId = detail::InvalidUuid;
    }
}

const NasSessionPoolBase::Session* NasSessionPoolBase::FindUnsafe(const detail::Uuid& sessionId) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(sessionId);
    for (const auto& session : m_Sessions)
    {
        if (session.sessionId == sessionId)
        {
            return &session;
        }
    }
    return nullptr;
}
NasSessionPoolBase::Session* NasSessionPoolBase::FindUnsafe(const detail::Uuid& sessionId, const Validation& validation) NN_NOEXCEPT
{
    NN_SDK_ASSERT(sessionId);
    for (auto& session : m_Sessions)
    {
        if (true
            && session.sessionId == sessionId
            && session.validation.uid == validation.uid
            && session.validation.tag == validation.tag)
        {
            return &session;
        }
    }
    return nullptr;
}

Result NasSessionPoolBase::Register(detail::Uuid* pOutSessionId, uintptr_t ptr, const Validation& validation) NN_NOEXCEPT
{
    NN_SDK_ASSERT(ptr != 0u);
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    detail::Uuid sessionId = detail::InvalidUuid;
    auto retryCount = 10; // TORIAEZU
    while (retryCount-- > 0)
    {
        auto uuid = detail::GenerateUuid();
        NN_SDK_ASSERT(uuid);
        auto p = FindUnsafe(uuid);
        if (p == nullptr)
        {
            sessionId = uuid;
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    }
    NN_RESULT_THROW_UNLESS(sessionId, ResultNotSupported());

    Session* pReserved = nullptr;
    for (auto& session : m_Sessions)
    {
        if (!session.sessionId)
        {
            pReserved = &session;
            break;
        }
    }
    NN_RESULT_THROW_UNLESS(pReserved != nullptr, ResultNasAuthorizationSessionFull());

    // 登録
    pReserved->sessionId = sessionId;
    pReserved->validation = validation;
    pReserved->count = 1;
    pReserved->ptr = ptr;
    *pOutSessionId = sessionId;
    NN_RESULT_SUCCESS;
}

Result NasSessionPoolBase::Acquire(uintptr_t* pOutPtr, const detail::Uuid& sessionId, const Validation& validation) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    auto p = FindUnsafe(sessionId, validation);
    NN_RESULT_THROW_UNLESS(p != nullptr, ResultNintendoAccountAuthorizationSessionNotExist());
    NN_SDK_ASSERT(p->ptr != 0u);
    NN_SDK_ASSERT(p->count > 0);
    NN_SDK_ASSERT(p->count < std::numeric_limits<decltype(p->count)>::max());

    ++ p->count;
    *pOutPtr = p->ptr;
    NN_RESULT_SUCCESS;
}
uintptr_t NasSessionPoolBase::Release(const detail::Uuid& sessionId, const Validation& validation) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    auto p = FindUnsafe(sessionId, validation);
    NN_ABORT_UNLESS(p != nullptr);
    NN_SDK_ASSERT(p->ptr != 0u);
    NN_SDK_ASSERT(p->count > 0);

    -- p->count;
    if (!(p->count > 0))
    {
        p->sessionId = detail::InvalidUuid;
        return p->ptr;
    }
    return 0u;
}

} // ~namespace nn::account::nas
}
}
