﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_NasResourceResolver.h"

#include <cstring>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/nsd/nsd_ApiForNasServicePrivate.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace account { namespace nas {
namespace {
const char FqdnForLogin[] = "https://accounts.nintendo.com";
const char FqdnForUser[] = "https://api.accounts.nintendo.com";
} // ~namespace nn::account::nas::<anonymous>

// DD1: {0xb266f39b541d814full, "nintendo://account.nx.sys"};
NasClientInfo NasResourceResolver::m_ClientInfoForAccountSystem = {0x00ull, "nintendo://nobody"};

bool NasResourceResolver::m_Initialized = false;

void NasResourceResolver::InitializeImplicitly() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(os::SdkMutex, s_Lock);
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);
    if (m_Initialized)
    {
        return;
    }

    nsd::NasServiceSetting settings;
    nsd::GetNasServiceSettingEx(&settings, nsd::NasServiceNameOfNxAccount);

    NasClientInfo clientInfo;
    clientInfo.clientId = settings.clientId;
    std::strncpy(clientInfo.redirectUri, settings.redirectUri.value, sizeof(clientInfo.redirectUri));

    m_ClientInfoForAccountSystem = clientInfo;
    m_Initialized = true;
}

int NasResourceResolver::GetUrlForToken(char* url, size_t bufferSize) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(url, bufferSize, "%s/connect/1.0.0/api/token", FqdnForLogin);
    NN_ABORT_UNLESS(l < static_cast<int>(bufferSize)); // 終端チェック
    return l;
}
int NasResourceResolver::GetUrlForAuthorizationRequest(char* url, size_t bufferSize) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(url, bufferSize, "%s/connect/1.0.0/authorize", FqdnForLogin);
    NN_ABORT_UNLESS(l < static_cast<int>(bufferSize)); // 終端チェック
    return l;
}
int NasResourceResolver::GetUrlForUserResource(char* url, size_t bufferSize) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(url, bufferSize, "%s/2.0.0/users/me", FqdnForUser);
    NN_ABORT_UNLESS(l < static_cast<int>(bufferSize)); // 終端チェック
    return l;
}
NasClientInfo NasResourceResolver::GetClientIdForAccountSystem() NN_NOEXCEPT
{
    InitializeImplicitly();
    return m_ClientInfoForAccountSystem;
}
Scope NasResourceResolver::GetScopeForAccountSystem() NN_NOEXCEPT
{
    static const Scope Scope = {""};
    return Scope;
}
Scope NasResourceResolver::GetScopeForOp2Membership() NN_NOEXCEPT
{
    static const Scope Scope = {"user"}; // TODO SIGLO-71933 user.membership に変更する
    return Scope;
}

}}} //
