﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/json/account_JsonAdaptor.h>

#include <utility>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace account { namespace nas {

// NAS API が定めるエラーコードの表現形式
// 参照: http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=161304662

const int NasErrorCodeLengthMax = 32;
const int NasErrorDetailLengthMax = 48;

struct NasProblem
{
    char error[NasErrorCodeLengthMax];

    bool operator==(const NasProblem& rts) const NN_NOEXCEPT;
    bool operator!=(const NasProblem& rts) const NN_NOEXCEPT;
};

struct NasProblemWithDetail
{
    char error[NasErrorCodeLengthMax];
    char detail[NasErrorDetailLengthMax];

    bool operator==(const NasProblemWithDetail& rts) const NN_NOEXCEPT;
    bool operator!=(const NasProblemWithDetail& rts) const NN_NOEXCEPT;
};
std::pair<bool, Result> FindResultForAuthorizationRequest(const NasProblemWithDetail& problem) NN_NOEXCEPT;

template <int NodeDepthMax, int PathLengthMax>
class NasProblemAdaptor
{
    NN_STATIC_ASSERT(NodeDepthMax >= 4 && PathLengthMax >= 16);

    NN_DISALLOW_COPY(NasProblemAdaptor);

public:
    typedef json::JsonPath<NodeDepthMax, PathLengthMax> JsonPathType;

private:
    json::LookupEntry m_ErrorCodeEntry;

    NasProblem m_Problem;

public:
    NasProblemAdaptor() NN_NOEXCEPT;

    Result Adapt(int32_t httpCode) const NN_NOEXCEPT;

    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;

    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, bool) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, double) NN_NOEXCEPT
    { /*NOP*/ }
};

template <int NodeDepthMax, int PathLengthMax>
class NasTokenRequestProblemAdaptor
{
    NN_STATIC_ASSERT(NodeDepthMax >= 4 && PathLengthMax >= 16);

    NN_DISALLOW_COPY(NasTokenRequestProblemAdaptor);

public:
    typedef json::JsonPath<NodeDepthMax, PathLengthMax> JsonPathType;

private:
    json::LookupEntry m_ErrorCodeEntry;
    json::LookupEntry m_ErrorDetailEntry;

    NasProblemWithDetail m_Problem;

public:
    NasTokenRequestProblemAdaptor() NN_NOEXCEPT;

    Result Adapt(int32_t httpCode) NN_NOEXCEPT;

    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;

    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, bool) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
    { /*NOP*/ }
    void Update(const JsonPathType&, double) NN_NOEXCEPT
    { /*NOP*/ }
};


template <int NodeDepthMax, int PathLengthMax>
class NasCommonAdaptor
    : public json::ExtensibleJsonAdaptorBase<NasProblemAdaptor<NodeDepthMax, PathLengthMax>>
{
};

template <int NodeDepthMax, int PathLengthMax>
class NasTokenRequestAdaptorBase
    : public json::ExtensibleJsonAdaptorBase<NasTokenRequestProblemAdaptor<NodeDepthMax, PathLengthMax>>
{
};

}}} // ~namespace nn::account::baas

/* --------------------------------------------------------------------------------------------
    実装
*/

#include "../http/account_HttpUtil.h"

#include <cstring>
#include <limits>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace account { namespace nas {

/* --------------------------------------------------------------------------------------------
    NasProblem
*/
std::pair<bool, Result> FindResult(const NasProblem& problem) NN_NOEXCEPT;

inline bool NasProblem::operator==(const NasProblem& rhs) const NN_NOEXCEPT
{
    return strncmp(error, rhs.error, sizeof(error)) == 0;
}
inline bool NasProblem::operator!=(const NasProblem& rhs) const NN_NOEXCEPT
{
    return !(*this == rhs);
}

template <int NodeDepthMax, int PathLengthMax>
inline NasProblemAdaptor<NodeDepthMax, PathLengthMax>::NasProblemAdaptor() NN_NOEXCEPT
    : m_ErrorCodeEntry("$.error")
{
}
template <int NodeDepthMax, int PathLengthMax>
inline Result NasProblemAdaptor<NodeDepthMax, PathLengthMax>::Adapt(int32_t httpCode) const NN_NOEXCEPT
{
    if (httpCode / 100 == 2)
    {
        // HttpStatus が 2xx なら ResultSuccess
        NN_RESULT_SUCCESS;
    }

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
    NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

    NN_SDK_LOG("[nn::account] -----------------------------------------------\n");
    NN_SDK_LOG("  Error: NAS ProblemDetails for HTTP Status %d\n", httpCode);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_ErrorCodeEntry);

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

    if (m_ErrorCodeEntry)
    {
        auto r = FindResult(m_Problem);
        if (r.first)
        {
            NN_RESULT_THROW(r.second);
        }
    }
    NN_RESULT_THROW(http::HandleHttpStatusCode(httpCode));
}
template <int NodeDepthMax, int PathLengthMax>
inline void NasProblemAdaptor<NodeDepthMax, PathLengthMax>::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(valueLength >= 0);
    if (m_ErrorCodeEntry.CanAccept(jsonPath))
    {
        if (!(static_cast<uint32_t>(valueLength) <= sizeof(m_Problem.error)))
        {
            // エラーコードが長すぎる
            return;
        }
        std::strncpy(m_Problem.error, value, sizeof(m_Problem.error));
        m_ErrorCodeEntry.MarkAccepted();
    }
}

/* --------------------------------------------------------------------------------------------
    NasTokenRequestProblem
*/
std::pair<bool, Result> FindResultForTokenRequest(const NasProblemWithDetail& problem) NN_NOEXCEPT;

inline bool NasProblemWithDetail::operator==(const NasProblemWithDetail& rhs) const NN_NOEXCEPT
{
    return (true
        && strncmp(error, rhs.error, sizeof(error)) == 0
        && strncmp(detail, rhs.detail, sizeof(detail)) == 0);
}
inline bool NasProblemWithDetail::operator!=(const NasProblemWithDetail& rhs) const NN_NOEXCEPT
{
    return !(*this == rhs);
}

template <int NodeDepthMax, int PathLengthMax>
inline NasTokenRequestProblemAdaptor<NodeDepthMax, PathLengthMax>::NasTokenRequestProblemAdaptor() NN_NOEXCEPT
    : m_ErrorCodeEntry("$.error")
    , m_ErrorDetailEntry("$.error_detail")
{
}
template <int NodeDepthMax, int PathLengthMax>
inline Result NasTokenRequestProblemAdaptor<NodeDepthMax, PathLengthMax>::Adapt(int32_t httpCode) NN_NOEXCEPT
{
    if (httpCode / 100 == 2)
    {
        // HttpStatus が 2xx なら ResultSuccess
        NN_RESULT_SUCCESS;
    }

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
    NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

    NN_SDK_LOG("[nn::account] -----------------------------------------------\n");
    NN_SDK_LOG("  Error: NAS ProblemDetails for HTTP Status %d\n", httpCode);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_ErrorCodeEntry);
    NN_ACCOUNT_PRINT_ENTRY_STATE(m_ErrorDetailEntry);

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

    if (m_ErrorCodeEntry)
    {
        if (!m_ErrorDetailEntry)
        {
            m_Problem.detail[0] = '\0';
        }

        auto r = FindResultForTokenRequest(m_Problem);
        if (r.first)
        {
            NN_RESULT_THROW(r.second);
        }
    }
    NN_RESULT_THROW(http::HandleHttpStatusCode(httpCode));
}
template <int NodeDepthMax, int PathLengthMax>
inline void NasTokenRequestProblemAdaptor<NodeDepthMax, PathLengthMax>::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(valueLength >= 0);
    if (m_ErrorCodeEntry.CanAccept(jsonPath))
    {
        NN_SDK_LOG("[nn::account] NAS $.error       : %s\n", value);
        if (!(static_cast<uint32_t>(valueLength) <= sizeof(m_Problem.error)))
        {
            // エラーコードが長すぎる
            return;
        }
        std::strncpy(m_Problem.error, value, sizeof(m_Problem.error));
        m_ErrorCodeEntry.MarkAccepted();
    }
    else if (m_ErrorDetailEntry.CanAccept(jsonPath))
    {
        NN_SDK_LOG("[nn::account] NAS $.error_detail : %s\n", value);
        if (!(static_cast<uint32_t>(valueLength) <= sizeof(m_Problem.detail)))
        {
            // エラーコードが長すぎる
            return;
        }
        std::strncpy(m_Problem.detail, value, sizeof(m_Problem.detail));
        m_ErrorDetailEntry.MarkAccepted();
    }
}

}}} // ~namespace nn::account::baas
