﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasLoginDriver.h>

#include "account_NasAuthorizationHeader.h"
#include "account_NasCredentialHolder.h"
#include "account_NasLoginAdaptor.h"
#include "account_NasResourceResolver.h"
#include "../detail/account_ByteUtil.h"
#include "../detail/account_CacheUtil.h"
#include "../http/account_OAuthUtil.h"
#include "../http/account_UrlEncoder.h"
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_RedirectionCaptor.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/account_ResultPrivate.h>

#include <cctype>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace nas {

namespace {

inline bool IsValidRedirectUri(const char* url, size_t size) NN_NOEXCEPT
{
    size_t offset = 0u;
    for (; offset < size && url[offset] != '\0'; ++ offset)
    {
        auto c = url[offset];
        if (!(std::isalnum(c) || c == ':' || c == '/' || c == '-' || c == '.' || c == '_' || c == '~'))
        {
            return false;
        }
    }
    return offset < size;
}

template <size_t N>
inline bool IsValidParameter(const char (&data)[N]) NN_NOEXCEPT
{
    int i = 0;
    for (; i < N && data[i] != '\0' && http::IsUnreservedCharacterForUri(data[i]); ++ i)
    {
    }
    return (i == N || data[i] == '\0');
}

template <size_t M, size_t N>
Result FillUserParameter(size_t* sizeActual, char* dst, size_t dstSize, const char (&label)[M], const char (&value)[N]) NN_NOEXCEPT
{
    NN_SDK_ASSERT(label[M - 1] == '\0');
    NN_SDK_ASSERT(strnlen(value, N) < N);
    NN_RESULT_THROW_UNLESS(dstSize > (M - 1) + 3 * (N - 1) + 1, ResultInsufficientBuffer());

    size_t offset = 0u;
    strncpy(dst + offset, label, M - 1);
    offset += M - 1;

    auto l = http::UrlEncode(dst + offset, dstSize - offset, value, sizeof(value));
    NN_SDK_ASSERT(l < (N - 1) * 3);
    offset += l;

    dst[offset] = '\0';
    *sizeActual = offset;
    NN_RESULT_SUCCESS;
}

// TODO: Mii を扱う場合には追加する →  user.mii
static const char NxScopeForTokenRequest[] = ""
    "openid"
    "+offline"
    "+napps"
    "+urn%3Aoauth%3Ainit-sso"
    "+user"
    "+user.birthday"
    "+user.email"
    "+user.links"
    "+user.links%5B%5D.id"
    "+user.loginId"
    // "+user.membership" // TODO SIGLO-71933 OP2 用
    "+user.screenName"
    "+user.terms";

Result SetupStreamForTokenRequest(
    http::CurlInputStream& stream,
    char* postDataBuffer, size_t postDataBufferSize,
    const NasClientInfo& clientInfo, const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
    const ndas::NdasOperator& ndasOperator,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());
    NN_RESULT_THROW_UNLESS(IsValidParameter(codeVerifier.data), ResultInvalidStringFormat());

    // POST データ作成
    static const char PostDataFormat[] =
        "grant_type=authorization_code"
        "&client_id=%016llx"
        "&scope=%s"
        "&code_verifier=%.*s"
        "&code=";
    size_t l;
    l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        clientInfo.clientId, NxScopeForTokenRequest,
        sizeof(codeVerifier.data), codeVerifier.data);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < postDataBufferSize);
    NN_UNUSED(l);

    // Authorization code
    auto codeBuffer = &postDataBuffer[l];
    auto codeBufferSize = postDataBufferSize - l;
    NN_RESULT_THROW_UNLESS(codeBufferSize > detail::NasAuthorizationCodeSizeMax, ResultInsufficientBuffer());

    size_t codeSize;
    NN_RESULT_DO((detail::CacheUtil::LoadCacheFile<detail::NasAuthorizationCodeSizeMax, ResultInvalidArrayLength>(
        &codeSize, codeBuffer, codeBufferSize, codeCacheId, storage)));
    l += codeSize;

    // Device authentication token
    NN_RESULT_THROW_UNLESS(l + sizeof("&device_authentication_token=") <= postDataBufferSize, ResultInsufficientBuffer());
    std::strncpy(&postDataBuffer[l], "&device_authentication_token=", postDataBufferSize - l);
    l += sizeof("&device_authentication_token=") - 1;

    auto tokenBuffer = &postDataBuffer[l];
    auto tokenBufferSize = postDataBufferSize - l;
    NN_RESULT_THROW_UNLESS(tokenBufferSize > detail::NdasDevAuthTokenSizeMax, ResultInsufficientBuffer());
    size_t tokenSize;
    NN_RESULT_DO(ndasOperator.LoadServiceAuthenticationTokenCacheForNintendoAccount(&tokenSize, tokenBuffer, tokenBufferSize));
    l += tokenSize;

    // Redirect URI
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, postDataBuffer + l, postDataBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    l += filledSize;
    NN_SDK_ASSERT(postDataBuffer[l] == '\0');

    // リクエスト生成
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = NasResourceResolver::GetUrlForToken(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result SetupStreamForTokenRefresh(
    http::CurlInputStream& stream,
    char* postDataBuffer, size_t postDataBufferSize,
    const NintendoAccountId& id, const NasClientInfo& clientInfo, const char* scope,
    const ndas::NdasOperator& ndasOperator,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    http::CodeVerifier codeVerifier;
    NN_RESULT_DO(NasCredentialHolder::LoadCodeVerifier(&codeVerifier, id, storage));
    NN_RESULT_THROW_UNLESS(IsValidParameter(codeVerifier.data), ResultInvalidStringFormat());

    // POST データ作成
    int l = 0;
    if (scope[0] == '\0')
    {
        static const char PostDataFormat[] =
            "grant_type=refresh_token"
            "&client_id=%016llx"
            "&code_verifier=%.*s"
            "&refresh_token=";
        l = nn::util::SNPrintf(
            postDataBuffer, postDataBufferSize,
            PostDataFormat,
            clientInfo.clientId,
            sizeof(codeVerifier.data), codeVerifier.data);
    }
    else
    {
        static const char PostDataFormat[] =
            "grant_type=refresh_token"
            "&client_id=%016llx"
            "&scope=%s"
            "&code_verifier=%.*s"
            "&refresh_token=";
        l = nn::util::SNPrintf(
            postDataBuffer, postDataBufferSize,
            PostDataFormat,
            clientInfo.clientId,
            scope,
            sizeof(codeVerifier.data), codeVerifier.data);
    }
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < postDataBufferSize);

    // Refresh token
    auto tokenBuffer = postDataBuffer + l;
    auto tokenBufferSize = postDataBufferSize - l;
    NN_RESULT_THROW_UNLESS(tokenBufferSize > detail::NasRefreshTokenSizeMax, ResultInsufficientBuffer());

    size_t tokenSize;
    NN_RESULT_DO(NasCredentialHolder::LoadRefreshToken(
        &tokenSize, tokenBuffer, tokenBufferSize, id, storage));
    NN_ABORT_UNLESS(tokenSize <= detail::NasRefreshTokenSizeMax);
    l += static_cast<int>(tokenSize);

    // Device authentication token
    NN_RESULT_THROW_UNLESS(l + sizeof("&device_authentication_token=") <= postDataBufferSize, ResultInsufficientBuffer());
    std::strncpy(&postDataBuffer[l], "&device_authentication_token=", postDataBufferSize - l);
    l += sizeof("&device_authentication_token=") - 1;

    tokenBuffer = &postDataBuffer[l];
    tokenBufferSize = postDataBufferSize - l;
    NN_RESULT_THROW_UNLESS(tokenBufferSize > detail::NdasDevAuthTokenSizeMax, ResultInsufficientBuffer());
    NN_RESULT_DO(ndasOperator.LoadServiceAuthenticationTokenCacheForNintendoAccount(&tokenSize, tokenBuffer, tokenBufferSize));
    l += static_cast<int>(tokenSize);

    tokenBuffer[tokenSize] = '\0';
    NN_SDK_ASSERT(static_cast<int>(strnlen(postDataBuffer, postDataBufferSize)) == l);

    // リクエスト生成
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = NasResourceResolver::GetUrlForToken(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result SetupRedirectCaptorForApplicationAuthorizationRequestWithoutPrompt(
    http::RedirectionCaptor& captor,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const detail::Uuid& accessTokenCacheId,
    const NasClientInfo& clientInfo, const Scope& scope, const State& state, const Nonce& nonce,
    const detail::AbstractLocalStorage& storage,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());

    // POST データ作成
    static const char PostDataFormat[] =
        "response_type=code+id_token&prompt=none"
        "&client_id=%016llx";
    size_t l;
    l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        clientInfo.clientId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < postDataBufferSize);

    // Redirect URI, Scope, State, Nonce
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, postDataBuffer + l, postDataBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    NN_SDK_ASSERT(filledSize < postDataBufferSize - l);
    l += filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, postDataBuffer + l, postDataBufferSize - l, "&scope=", scope.data));
    NN_SDK_ASSERT(filledSize < postDataBufferSize - l);
    l += filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, postDataBuffer + l, postDataBufferSize - l, "&state=", state.data));
    NN_SDK_ASSERT(filledSize < postDataBufferSize - l);
    l += filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, postDataBuffer + l, postDataBufferSize - l, "&nonce=", nonce.data));
    NN_SDK_ASSERT(filledSize < postDataBufferSize - l);
    l += filledSize;
    NN_SDK_ASSERT(postDataBuffer[l] == '\0');

    // Authorization ヘッダの準備
    NN_RESULT_DO(CreateAuthorizationHeader(workBuffer, workBufferSize, accessTokenCacheId, storage));
    *headers = curl_slist_append(*headers, workBuffer);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(captor.SetHeaders(*headers));
    NN_RESULT_DO(captor.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    l = NasResourceResolver::GetUrlForAuthorizationRequest(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(captor.SetUrl(url));
    // NN_SDK_LOG("%s?%s\n", url, postDataBuffer);
    // NN_SDK_LOG("%s\n", workBuffer);

    NN_RESULT_SUCCESS;
}

const char* GetThemeParameter(NintendoAccountAuthorizationPageTheme theme)
{
    switch (theme)
    {
    case NintendoAccountAuthorizationPageTheme_Register:
        return "register";
    case NintendoAccountAuthorizationPageTheme_Intro:
        return "intro";
    case NintendoAccountAuthorizationPageTheme_EmailAuthentication:
        return "email_authentication";
    case NintendoAccountAuthorizationPageTheme_SimpleAuthentication:
        return "simple_authenticate";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

} // ~namespace nn::account::nas::<anonymous>

const size_t NasLoginDriver::RequiredBufferSize = 12 * 1024u;

Result NasLoginDriver::GetUrlForAuthorizationRequest(
    char* urlBuffer, size_t urlBufferSize,
    const NasClientInfo& clientInfo, const http::CodeVerifier& codeVerifier,
    const State& state, NintendoAccountAuthorizationPageTheme theme) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());
    NN_RESULT_THROW_UNLESS(IsValidParameter(state.data), ResultInvalidStringFormat());
    NN_SDK_ASSERT(urlBufferSize >= 4096);

    // code_challenge の生成
    char codeChallenge[256];
    http::GenerateCodeChallenge(codeChallenge, sizeof(codeChallenge), codeVerifier);

    static const char UrlFormat[] =
        "%s?" // protocol + fqdn
        "response_type=code&code_challenge_method=S256&scope=nx"
        "&client_id=%016llx"
        "&state=%.*s"
        "&code_challenge=%s"
        "&theme=%s";

    // FQDN 解決
    char url[256];
    size_t l;
    l = NasResourceResolver::GetUrlForAuthorizationRequest(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    // URL 生成
    l = util::SNPrintf(
        urlBuffer, urlBufferSize,
        UrlFormat,
        url,
        clientInfo.clientId,
        strnlen(state.data, sizeof(state.data)), state.data,
        codeChallenge,
        GetThemeParameter(theme));
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < urlBufferSize);

    // Redirect URI
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    NN_SDK_ASSERT(filledSize < urlBufferSize - l);
    l += filledSize;
    NN_SDK_ASSERT(urlBuffer[l] == '\0');

    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::GetUrlForAuthorizationRequestWithIdTokenHint(
    char* urlBuffer, size_t urlBufferSize,
    const NintendoAccountId& id, const NasClientInfo& clientInfo, const http::CodeVerifier& codeVerifier,
    const State& state) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());
    NN_RESULT_THROW_UNLESS(IsValidParameter(state.data), ResultInvalidStringFormat());
    NN_SDK_ASSERT(urlBufferSize >= 4096);

    // code_challenge の生成
    char codeChallenge[256];
    http::GenerateCodeChallenge(codeChallenge, sizeof(codeChallenge), codeVerifier);

    static const char UrlFormat[] =
        "%s?" // protocol + fqdn
        "response_type=code&code_challenge_method=S256&scope=nx"
        "&client_id=%016llx"
        "&state=%.*s"
        "&code_challenge=%s"
        "&id_token_hint=";

    // FQDN 解決
    char url[256];
    size_t l;
    l = NasResourceResolver::GetUrlForAuthorizationRequest(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    // URL 生成
    l = util::SNPrintf(
        urlBuffer, urlBufferSize,
        UrlFormat,
        url,
        clientInfo.clientId,
        strnlen(state.data, sizeof(state.data)), state.data,
        codeChallenge);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < urlBufferSize);

    // id_token_hint の付与
    NN_RESULT_THROW_UNLESS(urlBufferSize - l > detail::NasIdTokenSizeMax, ResultInsufficientBuffer());
    size_t sizeActual;
    NN_RESULT_DO(NasCredentialHolder::LoadIdToken(&sizeActual, urlBuffer + l, urlBufferSize - l, id, m_Storage));
    NN_ABORT_UNLESS(sizeActual <= detail::NasIdTokenSizeMax);
    l += sizeActual;

    // Redirect URI
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    l += filledSize;
    NN_SDK_ASSERT(urlBuffer[l] == '\0');

    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::GetUrlForNnidLinkageRequest(
    char* urlBuffer, size_t urlBufferSize,
    const NintendoAccountId& id, const NasClientInfo& clientInfo,
    const State& state) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());
    NN_RESULT_THROW_UNLESS(IsValidParameter(state.data), ResultInvalidStringFormat());
    NN_SDK_ASSERT(urlBufferSize >= 4096);

    static const char UrlFormat[] =
        "%s"
        "?response_type="   "id_token"
        "&claims="          "%s"
        "&scope="           "%s"
        "&client_id="       "%016llx"
        "&state="           "%.*s"
        "&id_token_hint=";

    // FQDN 解決
    char url[256];
    size_t l;
    l = NasResourceResolver::GetUrlForAuthorizationRequest(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    // URL 生成
    l = util::SNPrintf(
        urlBuffer, urlBufferSize,
        UrlFormat,
        url,
        "%7B%22links.nintendoNetwork.id%22%3A%7B%22essential%22%3Atrue%7D%7D", // claims, {"links.nintendoNetwork.id":{"essential":true}}
        "user.links+user.links%5B%5D.id", // scope, user.links user.links[].id
        clientInfo.clientId,
        strnlen(state.data, sizeof(state.data)), state.data);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < urlBufferSize);

    // id_token_hint の付与
    NN_RESULT_THROW_UNLESS(urlBufferSize - l > detail::NasIdTokenSizeMax, ResultInsufficientBuffer());
    size_t sizeActual;
    NN_RESULT_DO(NasCredentialHolder::LoadIdToken(&sizeActual, urlBuffer + l, urlBufferSize - l, id, m_Storage));
    NN_ABORT_UNLESS(sizeActual <= detail::NasIdTokenSizeMax);
    l += sizeActual;

    // Redirect URI
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    l += filledSize;
    NN_SDK_ASSERT(urlBuffer[l] == '\0');

    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::GetUrlForApplicationAuthorizationRequest(
    char* urlBuffer, size_t urlBufferSize,
    const NintendoAccountId& id,
    const NasClientInfo& clientInfo, const Scope& scope, const State& state, const Nonce& nonce) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());
    NN_SDK_ASSERT(urlBufferSize >= 4096);

    static const char UrlFormat[] =
        "%s?" // protocol + fqdn
        "response_type=code+id_token"
        "&client_id=%016llx"
        "&id_token_hint=";

    // FQDN 解決
    char url[256];
    size_t l;
    l = NasResourceResolver::GetUrlForAuthorizationRequest(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    // URL 生成
    l = util::SNPrintf(
        urlBuffer, urlBufferSize,
        UrlFormat,
        url,
        clientInfo.clientId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < urlBufferSize);

    // id_token_hint の付与
    NN_RESULT_THROW_UNLESS(urlBufferSize - l > detail::NasIdTokenSizeMax, ResultInsufficientBuffer());
    size_t sizeActual;
    NN_RESULT_DO(NasCredentialHolder::LoadIdToken(&sizeActual, urlBuffer + l, urlBufferSize - l, id, m_Storage));
    NN_ABORT_UNLESS(sizeActual <= detail::NasIdTokenSizeMax);
    l += sizeActual;

    // Redirect URI, Scope, State, Nonce
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    NN_SDK_ASSERT(filledSize < urlBufferSize - l);
    l += filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&scope=", scope.data));
    NN_SDK_ASSERT(filledSize < urlBufferSize - l);
    l += filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&state=", state.data));
    NN_SDK_ASSERT(filledSize < urlBufferSize - l);
    l += filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&nonce=", nonce.data));
    NN_SDK_ASSERT(filledSize < urlBufferSize - l);
    l += filledSize;

    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::GetUrlForGuestLoginRequest(
    char* urlBuffer, size_t urlBufferSize,
    const NasClientInfo& clientInfo, const State& state) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidRedirectUri(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), ResultInvalidStringFormat());
    NN_RESULT_THROW_UNLESS(IsValidParameter(state.data), ResultInvalidStringFormat());
    NN_SDK_ASSERT(urlBufferSize >= 4096);

    static const char UrlFormat[] =
        "%s?" // protocol + fqdn
        "response_type=id_token"
        "&client_id=%016llx"
        "&scope=openid" // ユーザーリソース等にはアクセスしない
        "&state=%.*s";

    // FQDN 解決
    char url[256];
    size_t l;
    l = NasResourceResolver::GetUrlForAuthorizationRequest(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    // URL 生成
    l = util::SNPrintf(
        urlBuffer, urlBufferSize,
        UrlFormat,
        url,
        clientInfo.clientId,
        strnlen(state.data, sizeof(state.data)), state.data);
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < urlBufferSize);

    // Redirect URI
    size_t filledSize;
    NN_RESULT_DO(FillUserParameter(
        &filledSize, urlBuffer + l, urlBufferSize - l, "&redirect_uri=", clientInfo.redirectUri));
    NN_SDK_ASSERT(filledSize < urlBufferSize - l);
    l += filledSize;
    NN_SDK_ASSERT(urlBuffer[l] == '\0');

    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::AcquirePrimaryTokens(
    NasCredentialCache* pOutCredentialCache, NasAccessTokenCache* pOutAccessTokenCache,
    const NasClientInfo& clientInfo,
    const detail::Uuid& codeCacheId, const http::CodeVerifier& codeVerifier,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        char postDataBuffer[detail::NasAuthorizationCodeSizeMax + detail::NdasDevAuthTokenSizeMax + 2048];
        union
        {
            struct
            {
                char stringBuffer[detail::NasRefreshTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutCredentialCache);
    NN_SDK_REQUIRES_NOT_NULL(pOutAccessTokenCache);
    NN_SDK_REQUIRES(codeCacheId);
    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(rawBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForTokenRequest(
        stream,
        buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        clientInfo, codeVerifier, codeCacheId, m_NdasOperator, m_Storage));

    // 通信の実行と結果の適用
    NasPrimaryTokenAdaptor adaptor(m_Storage);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // キャッシュの取得
    NN_RESULT_DO(adaptor.PullPrimaryTokens(pOutCredentialCache, pOutAccessTokenCache));
    NN_SDK_ASSERT(pOutCredentialCache->refreshTokenCacheId);
    NN_SDK_ASSERT(pOutCredentialCache->idTokenCacheId);
    NN_SDK_ASSERT(pOutAccessTokenCache->cacheId);
    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::RefreshAccessToken(
    NasAccessTokenCache* pOutAccessTokenCache,
    const NintendoAccountId& id, const NasClientInfo& clientInfo, const Scope& scope,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        char postDataBuffer[detail::NasRefreshTokenSizeMax + detail::NdasDevAuthTokenSizeMax + 2048];
        union
        {
            struct
            {
                char stringBuffer[detail::NasAccessTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutAccessTokenCache);
    NN_SDK_REQUIRES(id);
    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(rawBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForTokenRefresh(
        stream,
        buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        id, clientInfo, scope.data, m_NdasOperator, m_Storage));

    // 通信の実行と結果の適用
    NasAccessTokenAdaptor adaptor(m_Storage);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // キャッシュの取得
    NasAccessTokenCache accessTokenCache;
    NN_RESULT_DO(adaptor.PullAccessToken(&accessTokenCache));
    NN_SDK_ASSERT(accessTokenCache.cacheId);
    *pOutAccessTokenCache = accessTokenCache;
    NN_RESULT_SUCCESS;
}

Result NasLoginDriver::AcquireApplicationAuthorizationWithoutPrompt(
    NasApplicationAuthorization* pOutAuthorization,
    const detail::Uuid& accessTokenCacheId,
    const NasClientInfo& clientInfo, const Scope& scope, const State& state, const Nonce& nonce,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        char postDataBuffer[detail::NasRedirectedUriLengthMax + 1024];
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char location[detail::NasRedirectedUriLengthMax + 1];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutAuthorization);
    NN_SDK_REQUIRES(accessTokenCacheId);
    NN_SDK_REQUIRES_NOT_NULL(curlHandle);
    NN_SDK_REQUIRES_NOT_NULL(rawBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    // 通信の実行
    http::RedirectionCaptor captor(curlHandle, pCancellable);
    NN_RESULT_DO(captor.Initialize(buffer->u.io.location, sizeof(buffer->u.io.location)));
    NN_RESULT_DO(SetupRedirectCaptorForApplicationAuthorizationRequestWithoutPrompt(
        captor, &headers,
        buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        accessTokenCacheId, clientInfo, scope, state, nonce,
        m_Storage, buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));
    NN_RESULT_DO(captor.Invoke());

    // 結果の適用
    NasApplicationAuthorizationAdaptor adaptor(pOutAuthorization, clientInfo.redirectUri, state);
    NN_RESULT_THROW_UNLESS(adaptor.Parse(captor.GetLocation(), sizeof(buffer->u.io.location)), ResultNasDataBroken());
    NN_RESULT_DO(adaptor.Adapt());
    NN_RESULT_SUCCESS;
}

}
}
} // ~namespace nn::account::nas
