﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "account_ByteUtil.h"
#include <nn/account/account_Types.h>
#include <nn/account/detail/account_InternalTypes.h>

#include <type_traits>
#include <nn/util/util_BitPack.h>


namespace nn {
namespace account {
namespace detail {

class AbstractFileSystem;

} // ~namespace nn::account::detail
}
}

namespace nn {
namespace account {
namespace detail {

// 構造体の大きさ等の検査
static_assert(std::is_pod<Uuid>::value, "SerializedUuid is not POD");
static_assert(sizeof(Uuid) == 16, "UidString size error (!= 16 bytes))");
static_assert(std::is_pod<UuidString>::value, "UidString is not POD");
static_assert(sizeof(UuidString) == UuidString::Bytes, "UidString size error (!= sizeof(\"f81d4fae-7dec-11d0-a765-00a0c91e6bf6\"))");
static_assert(std::is_pod<SerializedUuid>::value, "SerializedUuid is not POD");
static_assert(sizeof(SerializedUuid) == SerializedUuid::Bytes, "UidString size error (!= 16 bytes))");

// UUID と UID の相互変換
inline const Uuid ConvertToUuid(const Uid& uid) NN_NOEXCEPT
{
    Uuid uuid = {
        {
            static_cast<Bit32>(uid._data[0] & 0xFFFFFFFFull),
            static_cast<Bit32>(uid._data[0] >> 32),
            static_cast<Bit32>(uid._data[1] & 0xFFFFFFFFull),
            static_cast<Bit32>(uid._data[1] >> 32),
        }
    };
    return uuid;
}
inline const Uid ConvertToUid(const Uuid& uuid) NN_NOEXCEPT
{
    Uid uid = {
        {
            (static_cast<Bit64>(uuid._data[1]) << 32) | uuid._data[0],
            (static_cast<Bit64>(uuid._data[3]) << 32) | uuid._data[2]
        }
    };
    return uid;
}

// 文字列化 / 文字列化解除
UuidString UuidToString(const Uuid& uuid) NN_NOEXCEPT;
Uuid UuidUnstring(const UuidString& us) NN_NOEXCEPT;

// 直列化 / 直列化解除
SerializedUuid SerializeUuid(const Uuid& uuid) NN_NOEXCEPT;
Uuid DeserializeUuid(const SerializedUuid& su) NN_NOEXCEPT;

Uuid GenerateUuid() NN_NOEXCEPT;
Uuid GenerateUuid(const AbstractFileSystem& fs) NN_NOEXCEPT;

} // ~namespace nn::account::detail
}
}

/* --------------------------------------------------------------------------------------------
 * 実装
 */
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Uuid.h>

namespace nn { namespace account { namespace detail {

inline const Uuid UuidString::Unstring() const NN_NOEXCEPT
{
    return UuidUnstring(*this);
}

inline const Uuid SerializedUuid::Deserialize() const NN_NOEXCEPT
{
    return DeserializeUuid(*this);
}

inline const UuidString Uuid::ToString() const NN_NOEXCEPT
{
    return UuidToString(*this);
}

inline const SerializedUuid Uuid::Serialize() const NN_NOEXCEPT
{
    return SerializeUuid(*this);
}

struct UuidTraits
{
    // Byte 0
    typedef nn::util::BitPack32::Field<0,                       32, uint32_t>   TimeLow;
    // Byte 1
    typedef nn::util::BitPack32::Field<0,                       16, uint16_t>   TimeMid;
    typedef nn::util::BitPack32::Field<TimeMid::Next,           16, uint16_t>   TimeHiAndVersion;
    // Byte 2
    typedef nn::util::BitPack32::Field<0,                       8,  uint8_t>    ClkSeqHiAndRes;
    typedef nn::util::BitPack32::Field<ClkSeqHiAndRes::Next,    8,  uint8_t>    ClkSeqLow;
    typedef nn::util::BitPack32::Field<ClkSeqLow::Next,         16, uint16_t>   NodeLo;
    // Byte 3
    typedef nn::util::BitPack32::Field<0,                       32, uint32_t>   NodeHi;

    static const Uuid Generate() NN_NOEXCEPT;
    static const Uuid GenerateWithContext() NN_NOEXCEPT;
};

inline UuidString UuidToString(const Uuid& uuid) NN_NOEXCEPT
{
    util::BitPack32 data[4] = {
        {static_cast<uint32_t>(uuid._data[0])},
        {static_cast<uint32_t>(uuid._data[1])},
        {static_cast<uint32_t>(uuid._data[2])},
        {static_cast<uint32_t>(uuid._data[3])}
    };

    UuidString str;
    util::SNPrintf(
        str._data, sizeof(str._data),
        "%08x-%04x-%04x-%02x%02x-%08x%04x",
        data[0].Get<UuidTraits::TimeLow>(), data[1].Get<UuidTraits::TimeMid>(), data[1].Get<UuidTraits::TimeHiAndVersion>(),
        data[2].Get<UuidTraits::ClkSeqHiAndRes>(), data[2].Get<UuidTraits::ClkSeqLow>(),
        data[3].Get<UuidTraits::NodeHi>(), data[2].Get<UuidTraits::NodeLo>());
    return str;
}
inline Uuid UuidUnstring(const UuidString& us) NN_NOEXCEPT
{
    util::BitPack32 data[4] = {{0}, {0}, {0}, {0}};
    const char* str = us._data;

    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint32_t) * 2);
    data[0].Set<UuidTraits::TimeLow>(ExtractHexadecimal<uint32_t>(str, sizeof(uint32_t) * 2));
    str += sizeof(uint32_t) * 2;
    ++ str; // -
    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint16_t) * 2);
    data[1].Set<UuidTraits::TimeMid>(ExtractHexadecimal<uint16_t>(str, sizeof(uint16_t) * 2));
    str += sizeof(uint16_t) * 2;
    ++ str; // -
    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint16_t) * 2);
    data[1].Set<UuidTraits::TimeHiAndVersion>(ExtractHexadecimal<uint16_t>(str, sizeof(uint16_t) * 2));
    str += sizeof(uint16_t) * 2;
    ++ str; // -
    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint8_t) * 2);
    data[2].Set<UuidTraits::ClkSeqHiAndRes>(ExtractHexadecimal<uint8_t>(str, sizeof(uint8_t) * 2));
    str += sizeof(uint8_t) * 2;
    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint8_t) * 2);
    data[2].Set<UuidTraits::ClkSeqLow>(ExtractHexadecimal<uint8_t>(str, sizeof(uint8_t) * 2));
    str += sizeof(uint8_t) * 2;
    ++ str; // -
    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint32_t) * 2);
    data[3].Set<UuidTraits::NodeHi>(ExtractHexadecimal<uint32_t>(str, sizeof(uint32_t) * 2));
    str += sizeof(uint32_t) * 2;
    NN_SDK_ASSERT(UuidString::Bytes - (str - us._data) >= sizeof(uint16_t) * 2);
    data[2].Set<UuidTraits::NodeLo>(ExtractHexadecimal<uint16_t>(str, sizeof(uint16_t) * 2));
    str += sizeof(uint16_t) * 2;
    ++ str; // '\0'
    NN_SDK_ASSERT(str - us._data == UuidString::Bytes);

    Uuid uuid = {{data[0].storage, data[1].storage, data[2].storage, data[3].storage}};
    return uuid;
}
inline SerializedUuid SerializeUuid(const Uuid& src) NN_NOEXCEPT
{
    util::BitPack32 host[4] = {
        {static_cast<uint32_t>(src._data[0])},
        {static_cast<uint32_t>(src._data[1])},
        {static_cast<uint32_t>(src._data[2])},
        {static_cast<uint32_t>(src._data[3])}
    };

    util::BitPack32 serialized[4] = {{0}, {0}, {0}, {0}};
    serialized[0].Set<UuidTraits::TimeLow>(ConvertHostToNetwork(host[0].Get<UuidTraits::TimeLow>()));
    serialized[1].Set<UuidTraits::TimeMid>(ConvertHostToNetwork(host[1].Get<UuidTraits::TimeMid>()));
    serialized[1].Set<UuidTraits::TimeHiAndVersion>(ConvertHostToNetwork(host[1].Get<UuidTraits::TimeHiAndVersion>()));
    serialized[2].Set<UuidTraits::ClkSeqHiAndRes>(host[2].Get<UuidTraits::ClkSeqHiAndRes>()); // 1byte なので無変換
    serialized[2].Set<UuidTraits::ClkSeqLow>(host[2].Get<UuidTraits::ClkSeqLow>()); // 1byte なので無変換
    auto nodeLo = static_cast<uint64_t>(host[2].Get<UuidTraits::NodeLo>());
    auto nodeHi = static_cast<uint64_t>(host[3].Get<UuidTraits::NodeHi>());
    uint64_t node = ConvertHostToNetwork48((nodeHi << 16) | nodeLo); // Node は 48byte なので特殊に扱う
    serialized[2].Set<UuidTraits::NodeLo>(static_cast<uint16_t>(node & 0xFFFFull));
    serialized[3].Set<UuidTraits::NodeHi>(static_cast<uint32_t>((node >> 16) & 0xFFFFFFFFull));

    SerializedUuid uuid;
    std::memcpy(&uuid._data[0], &serialized[0].storage, sizeof(serialized[0].storage));
    std::memcpy(&uuid._data[4], &serialized[1].storage, sizeof(serialized[1].storage));
    std::memcpy(&uuid._data[8], &serialized[2].storage, sizeof(serialized[2].storage));
    std::memcpy(&uuid._data[12], &serialized[3].storage, sizeof(serialized[3].storage));
    return uuid;
}
inline Uuid DeserializeUuid(const SerializedUuid& src) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(src) == sizeof(Uuid));
    SerializedUuid dst = SerializeUuid(*reinterpret_cast<const Uuid*>(&src));
    return *reinterpret_cast<Uuid*>(&dst);
}
inline Uuid GenerateUuid() NN_NOEXCEPT
{
    auto src = util::GenerateUuid();
    NN_STATIC_ASSERT(sizeof(src) == sizeof(Uuid));
    SerializedUuid dst = SerializeUuid(*reinterpret_cast<const Uuid*>(&src));
    return *reinterpret_cast<Uuid*>(&dst);
}

}}}
