﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/account_ResultPrivate.h>

#include "account_PathUtil.h"
#include "account_TimeUtil.h"
#include "account_UuidUtil.h"

#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace detail {

class CacheUtil
{
private:
    static Result ReserveCacheFileUnsafe(Uuid* pOut, char* path, size_t pathSize, size_t size, const AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        Uuid cacheId = InvalidUuid;
        auto retryCount = 10; // TORIAEZU
        while (retryCount -- > 0)
        {
            auto uuid = GenerateUuid();

            auto r = storage.GetFileSystem().Create(PathUtil::GetCachePath(path, pathSize, uuid, storage.GetRootPath()), size);
            if (r.IsSuccess())
            {

                cacheId = uuid;
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
        }
        NN_RESULT_THROW_UNLESS(cacheId, ResultNotSupported());

        *pOut = cacheId;
        NN_RESULT_SUCCESS;
    }

public:
    template <int64_t ExpirationMargin>
    static bool IsExpiredInUptime(int64_t expiration) NN_NOEXCEPT
    {
        return expiration - ExpirationMargin < GetUptimeInSeconds();
    }

    static Result ReserveCacheFile(Uuid* pOut, size_t size, const AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        auto lock = storage.AcquireWriterLock();
        return ReserveCacheFileUnsafe(pOut, path, sizeof(path), size, storage);
    }

    static Result StoreCacheFile(Uuid* pOut, const void* buffer, size_t size, const AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        auto lock = storage.AcquireWriterLock();
        NN_RESULT_DO(ReserveCacheFileUnsafe(pOut, path, sizeof(path), size, storage));
        NN_RESULT_DO(storage.GetFileSystem().Write(path, buffer, size));
        NN_RESULT_SUCCESS;
    }

    static void DeleteCacheFile(const Uuid& cacheId, const AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        auto lock = storage.AcquireWriterLock();
        auto r = storage.GetFileSystem().Delete(
            PathUtil::GetCachePath(path, sizeof(path), cacheId, storage.GetRootPath()));
        NN_UNUSED(r); // 削除に失敗してもどうしようもないので不問
    }

    template <size_t MaxSize, typename MaxSizeErrorResultType>
    static Result LoadCacheFile(
        size_t* pOutActualSize, void* buffer, size_t bufferSize,
        const Uuid& cacheId, const AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        char path[128];
        PathUtil::GetCachePath(path, sizeof(path), cacheId, storage.GetRootPath());

        size_t size;
        NN_RESULT_DO(storage.GetFileSystem().GetSize(&size, path));
        NN_RESULT_THROW_UNLESS(size <= MaxSize, MaxSizeErrorResultType());

        if (!buffer)
        {
            // バッファの指定がなければサイズだけ返す
            *pOutActualSize = size;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_THROW_UNLESS(size <= bufferSize, ResultInsufficientBuffer());
        NN_RESULT_DO(storage.GetFileSystem().Read(pOutActualSize, buffer, bufferSize, path));
        NN_SDK_ASSERT(*pOutActualSize == size);
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namespace nn::account::detail
