﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*  @file
    @brief  バイトデータ周りのユーティリティ群
 */

#include <cctype>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/util/util_Base64.h>

namespace nn { namespace account { namespace detail {

#define NN_ACCOUNT_MAX(a, b) (((a) > (b))? (a): (b))

// 16進数文字の整数変換
inline uint8_t ConvertHexToInteger(char c) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(std::isxdigit(c));
    return std::isdigit(c)
        ? static_cast<uint8_t>(c - '0')
        : static_cast<uint8_t>(std::tolower(c) - 'a' + 10);
}

// 16進数文字列の整数変換
template <typename T>
inline T ExtractHexadecimal(const char* pStr, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT(length == sizeof(T) * 2);
    NN_UNUSED(length);
    T out = 0;
    for (auto i = 0u; i < sizeof(T) * 2; ++ i)
    {
        out = (out << 4 | ConvertHexToInteger(*(pStr ++)));
    }
    return out;
}

// ホスト-ネットワーク間のバイトオーダー変換
#if NN_BUILD_CONFIG_ENDIAN_SUPPORTS_LITTLE

inline uint64_t ConvertHostToNetwork48(uint64_t v) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((v & 0xFFFF000000000000) == 0);
    return 0
        | ((v & 0x00000000000000FFull) << 40)
        | ((v & 0x000000000000FF00ull) << 24)
        | ((v & 0x0000000000FF0000ull) << 8)
        | ((v & 0x00000000FF000000ull) >> 8)
        | ((v & 0x000000FF00000000ull) >> 24)
        | ((v & 0x0000FF0000000000ull) >> 40);
}
inline uint64_t ConvertHostToNetwork(uint64_t v) NN_NOEXCEPT
{
    return 0
        | ((v & 0x00000000000000FFull) << 56)
        | ((v & 0x000000000000FF00ull) << 40)
        | ((v & 0x0000000000FF0000ull) << 24)
        | ((v & 0x00000000FF000000ull) << 8)
        | ((v & 0x000000FF00000000ull) >> 8)
        | ((v & 0x0000FF0000000000ull) >> 24)
        | ((v & 0x00FF000000000000ull) >> 40)
        | ((v & 0xFF00000000000000ull) >> 56);
}
inline uint32_t ConvertHostToNetwork(uint32_t v) NN_NOEXCEPT
{
    return 0
        | ((v & 0x000000FF) << 24)
        | ((v & 0x0000FF00) << 8)
        | ((v & 0x00FF0000) >> 8)
        | ((v & 0xFF000000) >> 24);
}
inline uint16_t ConvertHostToNetwork(uint16_t v) NN_NOEXCEPT
{
    return 0
        | ((v & 0x00FF) << 8)
        | ((v & 0xFF00) >> 8);
}
inline uint64_t ConvertNetworkToHost48(uint64_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork48(v);
}
inline uint64_t ConvertNetworkToHost(uint64_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork(v);
}
inline uint32_t ConvertNetworkToHost(uint32_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork(v);
}
inline uint16_t ConvertNetworkToHost(uint16_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork(v);
}

#elif NN_BUILD_CONFIG_ENDIAN_SUPPORTS_BIG
inline uint64_t ConvertHostToNetwork48(uint64_t v) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((v & 0xFFFF00000000) == 0);
    return v;
}
inline uint64_t ConvertHostToNetwork(uint64_t v) NN_NOEXCEPT
{
    return v;
}
inline uint32_t ConvertHostToNetwork(uint32_t v) NN_NOEXCEPT
{
    return v;
}
inline uint16_t ConvertHostToNetwork(uint16_t v) NN_NOEXCEPT
{
    return v;
}
inline uint64_t ConvertNetworkToHost48(uint64_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork48(v);
}
inline uint64_t ConvertNetworkToHost(uint64_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork(v);
}
inline uint32_t ConvertNetworkToHost(uint32_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork(v);
}
inline uint16_t ConvertNetworkToHost(uint16_t v) NN_NOEXCEPT
{
    return ConvertHostToNetwork(v);
}

#else

#error "Unknown endian type specified"

#endif

inline const char* FindChar(const char* str, size_t strSize, char c) NN_NOEXCEPT
{
    size_t i;
    for (i = 0; i < strSize && str[i] != c && str[i] != '\0'; ++ i)
    {
    }
    return (i < strSize && str[i] == c)
        ? &str[i]
        : nullptr;
}

inline void GetEncodedSha256Hash(
    char* buffer, size_t bufferSize,
    const void* data, size_t dataSize, util::Base64::Mode mode) NN_NOEXCEPT
{
    crypto::Sha256Generator gen;
    uint8_t h[crypto::Sha256Generator::HashSize];
    gen.Initialize();
    gen.Update(data, dataSize);
    gen.GetHash(h, sizeof(h));

    auto s = util::Base64::ToBase64String(buffer, bufferSize, h, sizeof(h), mode);
    NN_SDK_ASSERT(s == util::Base64::Status_Success);
    NN_UNUSED(s);
    NN_SDK_ASSERT(strnlen(buffer, bufferSize) < bufferSize);
}

}}} // ~namespace nn::account::detail
