﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_BaasUserAdaptor.h"

#include "../detail/account_ByteUtil.h"
#include "../detail/account_CacheUtil.h"

#include <nn/account/baas/account_ResultForBaas.h>
#include <nn/account/account_Result.h>

#include <cstring>

namespace nn { namespace account { namespace baas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

/* --------------------------------------------------------------------------------------------
    ImageUploadAdaptor
*/

ImageUploadAdaptor::ImageUploadAdaptor(const char* expectedFormat, size_t length) NN_NOEXCEPT
    : m_UrlEntry("$.content.url")
    , m_FormatEntry("$.content.format")
    , m_Adapted(false)
{
    NN_SDK_ASSERT(length < sizeof(m_Format));
    NN_SDK_ASSERT(expectedFormat[length] == '\0');
    NN_UNUSED(length);
    std::strncpy(m_Format, expectedFormat, sizeof(m_Format));
}
Result ImageUploadAdaptor::AdaptImpl() NN_NOEXCEPT
{
    if (!(m_UrlEntry && m_FormatEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS ImageUpload Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_UrlEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_FormatEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }
    m_Adapted = true;
    NN_RESULT_SUCCESS;
}
bool ImageUploadAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_UrlEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_Url))
        {
            std::strncpy(m_Url, value, sizeof(m_Url));
            NN_SDK_ASSERT(strnlen(m_Url, sizeof(m_Url)) < sizeof(m_Url));
            m_Url[valueLength] = '\0';
            m_UrlEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_FormatEntry.CanAccept(jsonPath))
    {
        if (strncmp(m_Format, value, sizeof(m_Format)) == 0)
        {
            m_FormatEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}
void ImageUploadAdaptor::GetUrl(char* urlBuffer, size_t urlBufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Adapted);
    NN_SDK_ASSERT(urlBufferSize >= strnlen(m_Url, sizeof(m_Url)));
    std::strncpy(urlBuffer, m_Url, urlBufferSize);
}

/* --------------------------------------------------------------------------------------------
    UserResourceAdaptor
*/

UserResourceAdaptor::UserResourceAdaptor(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : UserResourceAdaptorWithBase<8, 32>("$", storage)
{
}

/* --------------------------------------------------------------------------------------------
    UserLinkAdaptor
**/

UserLinkAdaptor::UserLinkAdaptor() NN_NOEXCEPT
    : m_IdEntry("$.id")
    , m_NaIdEntry("$.links.nintendoAccount.id")
    , m_Adapted(false)
{
}
Result UserLinkAdaptor::AdaptImpl() NN_NOEXCEPT
{
    if (!(m_IdEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS UserResource failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NaIdEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }
    if (!m_NaIdEntry)
    {
        m_NaId = InvalidNintendoAccountId;
    }
    m_Adapted = true;
    NN_RESULT_SUCCESS;
}
bool UserLinkAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_IdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_Id.id = id;
                m_IdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NaIdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_NaId.id = id;
                m_NaIdEntry.MarkAccepted();
            }
        }
        return true;
    }
    return false;
}
}}} // ~namespace nn::account::baas
