﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasRegistrationDriver.h>

#include "account_BaasAuthorizationHeader.h"
#include "account_BaasRegistrationAdaptor.h"
#include "account_BaasResourceResolver.h"
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/account_Result.h>

#include <nn/util/util_ScopeExit.h>

namespace nn { namespace account { namespace baas {
namespace
{
const size_t StringBufferSizeMin = 128u;
NN_STATIC_ASSERT(StringBufferSizeMin >= BaasLoginPasswordLength);

nn::Result SetupRequestToRegister(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    const ClientAccessTokenCache& clientAccessTokenCache,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithClientAccessToken(authHeader, workBufferSize, clientAccessTokenCache));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(""));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForUserRegistry(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}
nn::Result SetupRequestToUnregister(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& id,
    const BaasCredential& credential,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, id));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpDelete());

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForDeviceAccount(url, sizeof(url), id, credential.loginId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

// バッファ定義
struct BasicBuffer
{
    union
    {
        char workBuffer[RequiredBufferSizeForAuthorizationHeader];
        struct
        {
            char stringBuffer[StringBufferSizeMin];
            char inputBuffer[detail::IoBufferSizeMin];
        } io;
    } u;
};
NN_STATIC_ASSERT(std::alignment_of<BasicBuffer>::value <= std::alignment_of<std::max_align_t>::value);
} // ~namespace nn::account::baas::<anonymous>

const size_t BaasRegistrationDriver::RequiredBufferSize = 8 * 1024u;
NN_STATIC_ASSERT(sizeof(BasicBuffer) <= BaasRegistrationDriver::RequiredBufferSize);

BaasRegistrationDriver::BaasRegistrationDriver(
    UserAccessTokenCache& userAccessTokenCache,
    UserIdTokenCache& userIdTokenCache,
    const ClientAccessTokenCache& clientAccessTokenCache) NN_NOEXCEPT
    : m_UserAccessTokenCache(userAccessTokenCache)
    , m_UserIdTokenCache(userIdTokenCache)
    , m_ClientAccessTokenCache(clientAccessTokenCache)
{
}

Result BaasRegistrationDriver::RegisterDeviceAccount(
    NetworkServiceAccountId* pOutId,
    BaasCredential* pOutCredential,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutCredential);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(BasicBuffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<BasicBuffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(BasicBuffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupRequestToRegister(
        stream, &headers, m_ClientAccessTokenCache,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 実行
    DeviceAccountRegistrationAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    *pOutId = adaptor.GetId();
    *pOutCredential = adaptor.GetCredential();
    NN_RESULT_SUCCESS;
}
Result BaasRegistrationDriver::UnregisterDeviceAccount(
    const NetworkServiceAccountId& id, const BaasCredential& credential,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(BasicBuffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<BasicBuffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(BasicBuffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupRequestToUnregister(
        stream, &headers, m_UserAccessTokenCache, id, credential,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 実行
    BaasDeleteAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    auto result = json::ImportJsonByRapidJson(adaptor, stream, pCancellable);
    NN_RESULT_THROW_UNLESS(
        result.IsSuccess() || json::ResultJsonEmpty::Includes(result), // 成功の場合 Empty が返る。
        result);
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::account::baas

