﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/baas/account_ResultForBaas.h>

#include "account_BaasProfileSynchronizer.h"
#include "../detail/account_CacheUtil.h"
#include "../nas/account_NasCredentialHolder.h"
#include "../profile/account_ProfileAdaptor.h"
#include <nn/account/nas/account_NasUserResourceCache.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/account/account_RuntimeResource.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace baas {

namespace
{
struct ReaderLock
{
    BaasUserInfoHolder& _lock;
    explicit ReaderLock(BaasUserInfoHolder& lock) NN_NOEXCEPT
        : _lock(lock)
    {
        _lock.lock_shared();
    }
    ~ReaderLock() NN_NOEXCEPT
    {
        _lock.unlock_shared();
    }
};
struct WriterLock
{
    BaasUserInfoHolder& _lock;
    explicit WriterLock(BaasUserInfoHolder& lock) NN_NOEXCEPT
        : _lock(lock)
    {
        _lock.lock();
    }
    ~WriterLock() NN_NOEXCEPT
    {
        _lock.unlock();
    }
};
} // ~namespace nn::account::baas::<anonymous>

BaasOperator::BaasOperator(
    ClientAccessTokenCache& clientAccessTokenCache,
    UserAccessTokenCache& userAccessTokenCache,
    UserIdTokenCache& userIdTokenCache,
    BaasUserInfoHolder& userInfoHolder,
    profile::ProfileStorage& profile,
    ndas::NdasOperator& ndasOperator,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_BaasLoginDriver(clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, ndasOperator, storage)
    , m_BaasRegistrationDriver(userAccessTokenCache, userIdTokenCache, clientAccessTokenCache)
    , m_BaasUserDriver(userAccessTokenCache, storage)
    , m_BaasChannelDriver(userAccessTokenCache)
    , m_UserInfoHolder(userInfoHolder)
    , m_ProfileStorage(profile)
    , m_NdasOperator(ndasOperator)
    , m_Storage(storage)
    , m_UserAccessTokenCache(userAccessTokenCache)
    , m_UserIdTokenCache(userIdTokenCache)
    , m_BaasUserResourceCache()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BaasUserResourceCache.Initialize(storage));
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BaasUserServiceEntryRequirementCache.Initialize(storage));
}

/* --------------------------------------------------------------------------------------------
    非通信系
*/
Result BaasOperator::GetUserInfo(
    NetworkServiceAccountId* pOutId, BaasCredential* pOutCredential, util::optional<NintendoAccountId> *ppOutNaId,
    const Uid& user) NN_NOEXCEPT
{
    NetworkServiceAccountId id;
    BaasCredential credential;
    util::optional<NintendoAccountId> pNaId;

    ReaderLock lock(m_UserInfoHolder);
    NN_RESULT_DO(m_UserInfoHolder.GetNetworkServiceAccountId(&id, user));
    NN_RESULT_DO(m_UserInfoHolder.GetCredential(&credential, user));

    NintendoAccountId naId;
    auto r = m_UserInfoHolder.GetLinkedNintendoAccountId(&naId, user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r), r);
    if (r.IsSuccess())
    {
        pNaId = naId;
    }

    *pOutId = id;
    *pOutCredential = credential;
    *ppOutNaId = pNaId;
    NN_RESULT_SUCCESS;
}

Result BaasOperator::HandleUserLoginResult(const Result result, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    NN_UNUSED(&user);
    // SIGLO-83428 で何もしないように変更
    return result;
}
Result BaasOperator::HandleUserLoginResult(const Result result, const Uid& user, const ncm::ApplicationId& appId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    NN_SDK_ASSERT_NOT_EQUAL(appId, ncm::ApplicationId::GetInvalidId());

    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(ResultBaasStatus403MembershipRequired)
    {
        m_BaasUserServiceEntryRequirementCache.Store(user, appId, BaasUserServiceEntryRequirement_Op2CommonLicense);
        NN_RESULT_THROW(m_BaasUserServiceEntryRequirementCache.CheckRequirement(user, appId));
    }
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_THROW(HandleUserLoginResult(result, user));
    }
    NN_RESULT_END_TRY

    m_BaasUserServiceEntryRequirementCache.Invalidate(user, appId);
    NN_RESULT_SUCCESS;
}

Result BaasOperator::DeleteForcibly(const Uid& user) NN_NOEXCEPT
{
    WriterLock lock(m_UserInfoHolder);

    // 資格情報のキャッシュをクリア
    m_BaasUserServiceEntryRequirementCache.Invalidate(user);

    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.Unregister(user));
    m_BaasUserResourceCache.Invalidate(user);
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
Result BaasOperator::IsRegistered(bool* pOut, const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    return m_UserInfoHolder.IsRegistered(pOut, user);
}
Result BaasOperator::CheckAvailability(const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    return m_UserInfoHolder.CheckAvailability(user);
}
Result BaasOperator::CheckAvailabilityWithUserServiceEntryRequirementCache(const Uid& user, const ncm::ApplicationId& appId) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    NN_RESULT_DO(m_UserInfoHolder.CheckAvailability(user));

    auto r = m_BaasUserServiceEntryRequirementCache.CheckRequirement(user, appId);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultResourceCacheUnavailable::Includes(r), r);
    NN_RESULT_SUCCESS;
}
Result BaasOperator::GetNetworkServiceAccountId(NetworkServiceAccountId* pOut, const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    return m_UserInfoHolder.GetNetworkServiceAccountId(pOut, user);
}

Result BaasOperator::LoadAccessTokenCache(
    size_t* pOutSizeActual, char* buffer, size_t bufferSize,
    const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    NetworkServiceAccountId id;
    NN_RESULT_DO(m_UserInfoHolder.GetNetworkServiceAccountId(&id, user));
    NN_RESULT_DO(m_BaasLoginDriver.LoadUserAccessTokenCache(pOutSizeActual, buffer, bufferSize, id));
    if (*pOutSizeActual < bufferSize)
    {
        buffer[*pOutSizeActual] = '\0';
    }
    NN_RESULT_SUCCESS;
}

Result BaasOperator::LoadIdTokenCache(
    size_t* pOutSizeActual, char* buffer, size_t bufferSize,
    const Uid& user, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    NetworkServiceAccountId nsaId;
    NN_RESULT_DO(m_UserInfoHolder.GetNetworkServiceAccountId(&nsaId, user));
    return LoadIdTokenCache(pOutSizeActual, buffer, bufferSize, nsaId, appInfo);
}

Result BaasOperator::LoadIdTokenCache(
    size_t* pOutSizeActual, char* buffer, size_t bufferSize,
    const NetworkServiceAccountId& nsaId, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
{
    NN_RESULT_DO(m_BaasLoginDriver.LoadUserIdTokenCache(pOutSizeActual, buffer, bufferSize, nsaId, appInfo));
    if (*pOutSizeActual < bufferSize)
    {
        buffer[*pOutSizeActual] = '\0';
    }
    NN_RESULT_SUCCESS;
}

void BaasOperator::InvalidateTokenCache(const Uid& uid) NN_NOEXCEPT
{
    NetworkServiceAccountId nsaId;
    auto r = GetNetworkServiceAccountId(&nsaId, uid);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN(
            "[account] GetNetworkServiceAccountId(%016llx_%016llx) failed with %03d-%04d\n",
            uid._data[0], uid._data[1], r.GetModule(), r.GetDescription());
        return;
    }
    m_UserAccessTokenCache.Invalidate(nsaId);
    m_UserIdTokenCache.InvalidateIfMatch(nsaId);
}

void BaasOperator::InvalidateTokenCache(const Uid& uid, const ncm::ApplicationId& appId) NN_NOEXCEPT
{
    NetworkServiceAccountId nsaId;
    auto r = GetNetworkServiceAccountId(&nsaId, uid);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN(
            "[account] GetNetworkServiceAccountId(%016llx_%016llx) failed with %03d-%04d\n",
            uid._data[0], uid._data[1], r.GetModule(), r.GetDescription());
        return;
    }
    m_UserIdTokenCache.InvalidateIfMatch(nsaId, appId);
}

Result BaasOperator::IsLinkedWithNintendoAccount(bool* pOut, const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    return m_UserInfoHolder.IsLinkedWithNintendoAccount(pOut, user);
}
Result BaasOperator::GetLinkedNintendoAccountId(NintendoAccountId* pOut, const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    return m_UserInfoHolder.GetLinkedNintendoAccountId(pOut, user);
}
Result BaasOperator::GetNetworkServiceAccountLoginId(uint64_t* pOut, const Uid& user) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    BaasCredential credential;
    NN_RESULT_DO(m_UserInfoHolder.GetCredential(&credential, user));
    *pOut = credential.loginId;
    NN_RESULT_SUCCESS;
}
Result BaasOperator::GetTimeSpanSinceLastSyncOfUserProfile(TimeSpan* pOut, const Uid& user) NN_NOEXCEPT
{
    BaasUserResource resource;
    NN_RESULT_DO(m_BaasUserResourceCache.Get(&resource, user));
    *pOut = (os::GetSystemTick().ToTimeSpan() - resource.timeStamp);
    NN_RESULT_SUCCESS;
}
void BaasOperator::InvalidateUserResourceCache(const Uid& user) NN_NOEXCEPT
{
    m_BaasUserResourceCache.Invalidate(user);
}

Result BaasOperator::GetUserServiceEntryRequirementCache(BaasUserServiceEntryRequirement* pOut, const Uid& user, const ncm::ApplicationId& appId) const NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    auto r = m_BaasUserServiceEntryRequirementCache.Get(pOut, user, appId);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultResourceCacheUnavailable::Includes(r), r);
    if (!r.IsSuccess())
    {
        NN_SDK_ASSERT(ResultResourceCacheUnavailable::Includes(r));
        *pOut = BaasUserServiceEntryRequirement_None;
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_SUCCESS;
}
void BaasOperator::InvalidateUserServiceEntryRequirementCache(const Uid& user, const ncm::ApplicationId& appId) NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    m_BaasUserServiceEntryRequirementCache.Invalidate(user, appId);
}
void BaasOperator::InvalidateUserServiceEntryRequirementCache(const Uid& user) NN_NOEXCEPT
{
    ReaderLock lock(m_UserInfoHolder);

    m_BaasUserServiceEntryRequirementCache.Invalidate(user);
}
void BaasOperator::SetUserServiceEntryRequirementCache(const Uid& user, const ncm::ApplicationId& appId, BaasUserServiceEntryRequirement requirement) NN_NOEXCEPT
{
    NN_SDK_ASSERT(requirement != BaasUserServiceEntryRequirement_None);

    ReaderLock lock(m_UserInfoHolder);

    m_BaasUserServiceEntryRequirementCache.Store(user, appId, requirement);
}

Result BaasOperator::DegradeNetworkServiceAccountUserState(const Uid& user, BaasUserState state) NN_NOEXCEPT
{
    WriterLock lock(m_UserInfoHolder);

    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.DegradeBaasUserState(user, state));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
Result BaasOperator::DegradeNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT
{
    WriterLock lock(m_UserInfoHolder);

    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.DegradeNintendoAccountLinkageState(user));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
Result BaasOperator::DegradeLinkedNintendoAccountUserState(const Uid& user, NintendoAccountUserState state) NN_NOEXCEPT
{
    WriterLock lock(m_UserInfoHolder);

    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.DegradeNintendoAccountUserState(user, state));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
Result BaasOperator::RecoveryLinkedNintendoAccountUserState(const Uid& user) NN_NOEXCEPT
{
    WriterLock lock(m_UserInfoHolder);

    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.RecoveryNintendoAccountUserState(user));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------------------------------------------
    通信系
*/

Result BaasOperator::EnsureUserAccessTokenCacheImpl(
    NetworkServiceAccountId* pOutId, BaasCredential* pOutCredential, util::optional<NintendoAccountId>* ppNaId,
    const Uid& user,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    NetworkServiceAccountId id;
    BaasCredential credential;
    util::optional<NintendoAccountId> pNaId;
    NN_RESULT_DO(GetUserInfo(&id, &credential, &pNaId, user));

    // public_user 権限のアクセストークン
    auto r = m_BaasLoginDriver.EnsureUserAccessTokenCache(
        id, credential, pNaId,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
    if (!r.IsSuccess())
    {
        // 以下 BaasUserInfo に変更が生じ得るためロックする
        WriterLock lock(m_UserInfoHolder);
        NetworkServiceAccountId id1;
        NN_RESULT_DO(m_UserInfoHolder.GetNetworkServiceAccountId(&id1, user));
        NN_RESULT_THROW_UNLESS(id == id1, r); // NSA ID が変化してたらそのまま返す
        NN_RESULT_DO(HandleUserLoginResult(r, user));
    }

    if (pOutId != nullptr)
    {
        *pOutId = id;
    }
    if (pOutCredential != nullptr)
    {
        *pOutCredential = credential;
    }
    if (ppNaId != nullptr)
    {
        *ppNaId = pNaId;
    }
    NN_RESULT_SUCCESS;
}
Result BaasOperator::EnsureUserIdTokenCacheImpl(
    NetworkServiceAccountId* pOutId, BaasCredential* pOutCredential, util::optional<NintendoAccountId>* ppNaId,
    const Uid& user, const detail::ApplicationInfo& appInfo,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // アプリケーション認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateApplication(appInfo, resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    NetworkServiceAccountId id;
    BaasCredential credential;
    util::optional<NintendoAccountId> pNaId;
    NN_RESULT_DO(GetUserInfo(&id, &credential, &pNaId, user));

    // public_user 権限のIDトークン for アプリケーション
    auto r = m_BaasLoginDriver.EnsureUserIdTokenCache(
        id, credential, pNaId, appInfo,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
    if (!r.IsSuccess())
    {
        // 以下 BaasUserInfo に変更が生じ得るためロックする
        WriterLock lock(m_UserInfoHolder);
        NetworkServiceAccountId id1;
        NN_RESULT_DO(m_UserInfoHolder.GetNetworkServiceAccountId(&id1, user));
        NN_RESULT_THROW_UNLESS(id == id1, r); // NSA ID が変化してたらそのまま返す
        NN_RESULT_DO(HandleUserLoginResult(r, user, appInfo.launchProperty.id));
    }

    if (pOutId != nullptr)
    {
        *pOutId = id;
    }
    if (pOutCredential != nullptr)
    {
        *pOutCredential = credential;
    }
    if (ppNaId != nullptr)
    {
        *ppNaId = pNaId;
    }
    NN_RESULT_SUCCESS;
}

Result BaasOperator::Register(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // DA の作成
    NetworkServiceAccountId nsaId;
    BaasCredential credential;
    NN_RESULT_DO(m_BaasRegistrationDriver.RegisterDeviceAccount(
        &nsaId, &credential, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // 適用
    WriterLock lock(m_UserInfoHolder);
    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.Register(user, nsaId, credential));
    m_BaasUserResourceCache.Invalidate(user);
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
Result BaasOperator::Import(
    const Uid& user,
    const NetworkServiceAccountId& id, const BaasCredential& credential,
    const NintendoAccountId& naId, const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    {
        WriterLock lock(m_UserInfoHolder);
        auto fsLock = m_Storage.AcquireWriterLock();
        NN_RESULT_DO(m_UserInfoHolder.Register(user, id, credential));
        m_BaasUserResourceCache.Invalidate(user);
        NN_RESULT_DO(m_UserInfoHolder.InitializeNintendoAccountLinkageState(
            user, naId, nasCredentialCache, codeVerifier, resource.buffer.address, resource.buffer.size));
        NN_RESULT_DO(m_UserInfoHolder.CompleteNintendoAccountLinkageState(user));
        NN_RESULT_DO(m_Storage.Commit());
    }

    // NX BaaS とのプロフィール同期
    auto r = SynchronizeUserProfile(user, resource, pCancellable);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultBaasStatus400InvalidRawContent::Includes(r), r);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN("[nn::account] SynchronizeUserProfile() failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
    }
    NN_RESULT_SUCCESS;
}

Result BaasOperator::Unregister(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    BaasCredential credential;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, &credential, nullptr, user, resource, pCancellable));

    // 一連の削除処理はアトミックに行う
    WriterLock lock(m_UserInfoHolder);

    // DA を BaaS から削除する
    NN_RESULT_DO(m_BaasRegistrationDriver.UnregisterDeviceAccount(
        id, credential,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // ローカルのログイン情報を削除
    return DeleteForcibly(user);
}

Result BaasOperator::EnsureAccessTokenCache(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, nullptr, nullptr, user, resource, pCancellable));
    NN_UNUSED(id);

#if !defined(NN_SDK_BUILD_RELEASE)
    bool cached;
    NN_RESULT_DO(m_BaasLoginDriver.IsUserAccessTokenCacheAvailable(&cached, id));
    NN_SDK_ASSERT(cached);
#endif
    NN_RESULT_SUCCESS;
}

Result BaasOperator::EnsureIdTokenCacheForApplication(const Uid& user, const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のIDトークン for アプリケーション
    NetworkServiceAccountId id;
    NN_RESULT_DO(EnsureUserIdTokenCacheImpl(&id, nullptr, nullptr, user, appInfo, resource, pCancellable));
    NN_UNUSED(id);

#if !defined(NN_SDK_BUILD_RELEASE)
    bool cached;
    NN_RESULT_DO(m_BaasLoginDriver.IsUserIdTokenCacheAvailable(&cached, id, appInfo));
    NN_SDK_ASSERT(cached);
#endif
    NN_RESULT_SUCCESS;
}
Result BaasOperator::EnsureIdTokenCacheForApplication(
    const NetworkServiceAccountId& nsaId, const BaasCredential& credential, const NintendoAccountId& naId, const detail::ApplicationInfo& appInfo,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // アプリケーション認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateApplication(appInfo, resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // public_user 権限のIDトークン for アプリケーション
    NN_RESULT_DO(m_BaasLoginDriver.EnsureUserIdTokenCache(
        nsaId, credential, naId, appInfo,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

#if !defined(NN_SDK_BUILD_RELEASE)
    bool cached;
    NN_RESULT_DO(m_BaasLoginDriver.IsUserIdTokenCacheAvailable(&cached, nsaId, appInfo));
    NN_SDK_ASSERT(cached);
#endif
    NN_RESULT_SUCCESS;
}
Result BaasOperator::UploadUserProfile(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, nullptr, nullptr, user, resource, pCancellable));

    // プロフィールのエクスポートとアップロード
    NN_RESULT_DO(ExportBaasUserProfile(
        m_ProfileStorage, m_BaasUserResourceCache,
        user, id, m_BaasUserDriver, m_Storage, resource, pCancellable));

    NN_RESULT_SUCCESS;
}
Result BaasOperator::SynchronizeUserProfile(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, nullptr, nullptr, user, resource, pCancellable));

    // プロフィールのダウンロードとインポートを試行
    NN_RESULT_DO(SynchronizeBaasUserProfile(
        m_ProfileStorage, m_BaasUserResourceCache, user, id, m_BaasUserDriver, m_Storage, resource, pCancellable));
    NN_RESULT_SUCCESS;
}
Result BaasOperator::RegisterNotificationToken(const Uid& user, const npns::NotificationToken& nt, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    BaasCredential credential;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, &credential, nullptr, user, resource, pCancellable));

    // NT を BaaS DA に登録する
    NN_RESULT_DO(m_BaasChannelDriver.PutChannel(id, credential.loginId, nt, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));
    NN_RESULT_SUCCESS;
}
Result BaasOperator::UnregisterNotificationToken(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    BaasCredential credential;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, &credential, nullptr, user, resource, pCancellable));

    // NT を BaaS DA から削除する
    NN_RESULT_DO(m_BaasChannelDriver.DeleteChannel(id, credential.loginId, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));
    NN_RESULT_SUCCESS;
}
Result BaasOperator::DownloadNotificationToken(npns::NotificationToken* pOut, const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    BaasCredential credential;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, &credential, nullptr, user, resource, pCancellable));

    // NT を BaaS DA から取得する
    NN_RESULT_DO(m_BaasChannelDriver.GetChannel(pOut, id, credential.loginId, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));
    NN_RESULT_SUCCESS;
}
Result BaasOperator::LinkWithNintendoAccount(
    bool* pOutIsNetworkServiceAccountReplaced,
    const Uid& user,
    const NintendoAccountId& naId, const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // ------------------------------------------------------------------------------------------------------------------------
    WriterLock lock(m_UserInfoHolder);

    // NX BaaS 認証情報
    NetworkServiceAccountId id;
    BaasCredential credential;
    NN_RESULT_DO(m_UserInfoHolder.GetNetworkServiceAccountId(&id, user));
    NN_RESULT_DO(m_UserInfoHolder.GetCredential(&credential, user));

    // 電断時のためにエラー状態に設定
    {
        auto fsLock = m_Storage.AcquireWriterLock();
        NN_RESULT_DO(m_UserInfoHolder.InitializeNintendoAccountLinkageState(
            user, naId, nasCredentialCache, codeVerifier, resource.buffer.address, resource.buffer.size));
        NN_RESULT_DO(m_Storage.Commit());
    }

    // federation の試行
    NetworkServiceAccountId loggedInAs;
    UserProfile profile;
    auto r = m_BaasLoginDriver.ExecuteFederationLoginWithNintendoAccount(
        &loggedInAs, &profile, credential, naId, nasCredentialCache.idTokenCacheId,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultBaasStatus400LinkedUserNotFound::Includes(r), r);
    if (!r.IsSuccess())
    {
        // NA に連携済みの NSA が存在しない
        NN_SDK_ASSERT(ResultBaasStatus400LinkedUserNotFound::Includes(r));

        // 新規に NSA を NA に連携する
        NN_RESULT_DO(HandleUserLoginResult(
            m_BaasLoginDriver.EnsureUserAccessTokenCache(
                id, credential, nullptr /* pNaId: この時点では連携していないため nullptr */,
                resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable),
            user));

        NN_RESULT_DO(m_BaasUserDriver.LinkWithNintendoAccount(
            id, naId, nasCredentialCache.idTokenCacheId,
            resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

        // NX BaaS にプロフィールをアップロード
        r = ExportBaasUserProfile(m_ProfileStorage, m_BaasUserResourceCache, user, id, m_BaasUserDriver, m_Storage, resource, pCancellable);
        NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultBaasStatus400InvalidRawContent::Includes(r), r);
        if (!r.IsSuccess())
        {
            NN_DETAIL_ACCOUNT_WARN("[nn::account] ExportBaasUserProfile() failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
        }

        // ユーザーアカウントの情報を更新
        auto fsLock = m_Storage.AcquireWriterLock();
        NN_RESULT_DO(m_UserInfoHolder.CompleteNintendoAccountLinkageState(user));
        NN_RESULT_DO(m_Storage.Commit());

        *pOutIsNetworkServiceAccountReplaced = false;
        NN_RESULT_SUCCESS;
    }

    // NX BaaS からインポート優先でプロフィール同期
    r = SynchronizeBaasUserProfile(
        m_ProfileStorage, m_BaasUserResourceCache, user, loggedInAs, profile,
        m_BaasUserDriver, m_Storage, resource, pCancellable);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultBaasStatus400InvalidRawContent::Includes(r), r);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN("[nn::account] SynchronizeBaasUserProfile() failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
    }

    // プロフィール同期の成功をもって連携完了とする
    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.UpdateNetworkServiceAccountId(user, loggedInAs));
    NN_RESULT_DO(m_UserInfoHolder.CompleteNintendoAccountLinkageState(user));
    NN_RESULT_DO(m_Storage.Commit());
    *pOutIsNetworkServiceAccountReplaced = (loggedInAs != id);
    NN_RESULT_SUCCESS;
}
Result BaasOperator::UpdateLinkageWithNintendoAccount(
    const Uid& user,
    const NintendoAccountId& naId, const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // Credential を持っている必要がある
    NN_RESULT_THROW_UNLESS(nas::NasCredentialHolder::HasCredential(naId, m_Storage), ResultNintendoAccountLinkageRequired());

    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // ------------------------------------------------------------------------------------------------------------------------
    // NX BaaS とのプロフィール同期
    auto r = SynchronizeUserProfile(user, resource, pCancellable);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultBaasStatus400InvalidRawContent::Includes(r), r);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN("[nn::account] SynchronizeUserProfile() failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
    }

    // プロフィール同期後に確定する
    WriterLock lock(m_UserInfoHolder);
    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.RecoveryNintendoAccountUserState(
        user, naId, nasCredentialCache, codeVerifier, resource.buffer.address, resource.buffer.size));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
Result BaasOperator::UnlinkNintendoAccount(
    const Uid& user,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // ------------------------------------------------------------------------------------------------------------------------
    // public_user 権限のアクセストークン
    NetworkServiceAccountId id;
    NN_RESULT_DO(EnsureUserAccessTokenCacheImpl(&id, nullptr, nullptr, user, resource, pCancellable));

    WriterLock lock(m_UserInfoHolder);

    // エラー状態に設定
    {
        auto fsLock = m_Storage.AcquireWriterLock();
        NN_RESULT_DO(m_UserInfoHolder.DegradeNintendoAccountLinkageState(user));
        NN_RESULT_DO(m_Storage.Commit());
    }

    // 紐づけ解除
    NN_RESULT_DO(m_BaasUserDriver.UnlinkFromNintendoAccount(
        id, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // エラー状態の更新
    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserInfoHolder.DismissNintendoAccountLinkageState(user));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}

Result BaasOperator::AcquireCredentialFromNintendoAccountIdToken(
    NetworkServiceAccountId* pOutNsaId, BaasCredential* pOutCredential, UserProfile* pOutProfile,
    const NintendoAccountId& naId, const detail::Uuid& nasIdTokenCacheId,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // デバイス認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForApplications(resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // DA 作成
    NetworkServiceAccountId nsaId;
    BaasCredential credential;
    NN_RESULT_DO(m_BaasRegistrationDriver.RegisterDeviceAccount(
        &nsaId, &credential,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            auto r = m_BaasRegistrationDriver.UnregisterDeviceAccount(
                nsaId, credential, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
            NN_DETAIL_ACCOUNT_WARN(
                "[account] UnregisterDeviceAccount() for temporal DA failed with %03d-%04d (0x%08llx)\n",
                r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
            NN_UNUSED(r);
        }
    };

    // federation の試行
    auto r = m_BaasLoginDriver.ExecuteFederationLoginWithNintendoAccount(
        pOutNsaId, pOutProfile, credential, naId, nasIdTokenCacheId,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
    NN_RESULT_THROW_UNLESS(false
        || r.IsSuccess()
        || ResultBaasStatus400LinkedUserNotFound::Includes(r) // NSA がない
        || ResultBaasDataNotExist::Includes(r), r); // NSA はあるが、 User リソースが空
    if (!r.IsSuccess())
    {
        NN_SDK_ASSERT(ResultBaasStatus400LinkedUserNotFound::Includes(r) || ResultBaasDataNotExist::Includes(r));
        NN_RESULT_THROW(ResultExternalNetworkServiceAccountNotExist());
    }
    *pOutCredential = credential;
    success = true;
    NN_RESULT_SUCCESS;
}

Result BaasOperator::EnsureIdTokenCacheForApplicationForGuest(
    baas::GuestUserProfile* pOut,
    const detail::ApplicationInfo& appInfo, const char (&nasIdToken)[detail::NasIdTokenSizeMax],
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // アプリケーション認証トークン
    NN_RESULT_DO(m_NdasOperator.AuthenticateApplication(appInfo, resource, pCancellable));

    // public_client 権限のアクセストークン
    NN_RESULT_DO(m_BaasLoginDriver.EnsureClientAccessTokenCache(
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // federation の試行
    auto r = m_BaasLoginDriver.ExecuteFederationLoginWithNintendoAccountForGuest(
        pOut, nasIdToken, appInfo,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
    NN_RESULT_THROW_UNLESS(false
        || r.IsSuccess()
        || ResultBaasStatus400LinkedUserNotFound::Includes(r) // NSA がない
        || ResultBaasDataNotExist::Includes(r), r); // NSA はあるが、 User リソースが空
    if (!r.IsSuccess())
    {
        NN_SDK_ASSERT(ResultBaasStatus400LinkedUserNotFound::Includes(r) || ResultBaasDataNotExist::Includes(r));
        NN_RESULT_THROW(ResultExternalNetworkServiceAccountNotExist());
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    bool cached;
    NN_RESULT_DO(m_BaasLoginDriver.IsUserIdTokenCacheAvailable(&cached, pOut->id, appInfo));
    NN_SDK_ASSERT(cached);
#endif
    NN_RESULT_SUCCESS;
}

}}} // ~namespace account::baas
