﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_ApiForMigrationService.h>
#include <nn/account/account_ExternalNetworkServiceAccountInfo.h>

#include <nn/account/baas/account_Interface.sfdl.h>

#include <utility>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_IServiceObject.h>

#include "detail/account_ShimLibraryUtility.h"

namespace nn {
namespace account {

namespace {

template <typename Interface>
inline Result GetNicknameImpl(Interface* ptr, Nickname* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    NN_RESULT_DO(ptr->GetNickname(sf::OutArray<char>(pOut->name, sizeof(pOut->name))));
    NN_SDK_ASSERT(strnlen(pOut->name, sizeof(pOut->name)) < sizeof(pOut->name));
    NN_RESULT_SUCCESS;
}
template <typename Interface>
inline Result GetProfileImageImpl(Interface* ptr, size_t* pOutSizeActual, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    NN_SDK_REQUIRES_NOT_NULL(pOutSizeActual);

    uint32_t size;
    NN_RESULT_DO(ptr->GetProfileImage(&size, sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize)));
    *pOutSizeActual = static_cast<size_t>(size);
    NN_RESULT_SUCCESS;
}
template <typename Interface>
inline Result GetNetworkServiceAccountIdImpl(Interface* ptr, NetworkServiceAccountId* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    NN_RESULT_DO(ptr->GetAccountId(pOut));
    NN_SDK_ASSERT(pOut->id != 0x00ull);
    NN_RESULT_SUCCESS;
}
template <typename Interface>
inline Result LoadNetworkServiceAccountIdTokenCacheImpl(Interface* ptr, size_t* pOutSizeActual, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    NN_SDK_REQUIRES_NOT_NULL(pOutSizeActual);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= NetworkServiceAccountIdTokenLengthMax);

    uint32_t size;
    NN_RESULT_DO(ptr->LoadIdTokenCache(&size, sf::OutBuffer(buffer, bufferSize)));
    *pOutSizeActual = static_cast<size_t>(size);
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::<anonymous>

/* ExternalNetworkServiceAccountInfo */

ExternalNetworkServiceAccountInfo::ExternalNetworkServiceAccountInfo() NN_NOEXCEPT
    : m_Ptr(nullptr)
{
}
ExternalNetworkServiceAccountInfo::ExternalNetworkServiceAccountInfo(baas::IGuestLoginRequest* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
}
ExternalNetworkServiceAccountInfo::ExternalNetworkServiceAccountInfo(ExternalNetworkServiceAccountInfo&& rhs) NN_NOEXCEPT
    : m_Ptr(rhs.m_Ptr)
{
    rhs.m_Ptr = nullptr;
}
ExternalNetworkServiceAccountInfo::~ExternalNetworkServiceAccountInfo() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}
ExternalNetworkServiceAccountInfo& ExternalNetworkServiceAccountInfo::operator=(ExternalNetworkServiceAccountInfo&& rhs) NN_NOEXCEPT
{
    ExternalNetworkServiceAccountInfo tmp(std::move(rhs));
    tmp.Swap(*this);
    return *this;
}
void ExternalNetworkServiceAccountInfo::Swap(ExternalNetworkServiceAccountInfo& rhs) NN_NOEXCEPT
{
    std::swap(m_Ptr, rhs.m_Ptr);
}
Result ExternalNetworkServiceAccountInfo::GetNickname(Nickname* pOut) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountInfo);

    return GetNicknameImpl<>(m_Ptr, pOut);
}
Result ExternalNetworkServiceAccountInfo::GetProfileImage(size_t* pOutSizeActual, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountInfo);

    return GetProfileImageImpl<>(m_Ptr, pOutSizeActual, buffer, bufferSize);
}
Result ExternalNetworkServiceAccountInfo::GetNetworkServiceAccountId(NetworkServiceAccountId* pOut) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountInfo);

    return GetNetworkServiceAccountIdImpl<>(m_Ptr, pOut);
}
Result ExternalNetworkServiceAccountInfo::LoadNetworkServiceAccountIdTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountInfo);

    return LoadNetworkServiceAccountIdTokenCacheImpl<>(m_Ptr, pOutSizeActual, buffer, bufferSize);
}

/* ExternalNetworkServiceAccountRegistrar */

ExternalNetworkServiceAccountRegistrar::ExternalNetworkServiceAccountRegistrar() NN_NOEXCEPT
    : m_Ptr(nullptr)
{
}
ExternalNetworkServiceAccountRegistrar::ExternalNetworkServiceAccountRegistrar(baas::IFloatingRegistrationRequest* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
}
ExternalNetworkServiceAccountRegistrar::ExternalNetworkServiceAccountRegistrar(ExternalNetworkServiceAccountRegistrar&& rhs) NN_NOEXCEPT
    : m_Ptr(rhs.m_Ptr)
{
    rhs.m_Ptr = nullptr;
}
ExternalNetworkServiceAccountRegistrar::~ExternalNetworkServiceAccountRegistrar() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}
ExternalNetworkServiceAccountRegistrar& ExternalNetworkServiceAccountRegistrar::operator=(ExternalNetworkServiceAccountRegistrar&& rhs) NN_NOEXCEPT
{
    ExternalNetworkServiceAccountRegistrar tmp(std::move(rhs));
    tmp.Swap(*this);
    return *this;
}
void ExternalNetworkServiceAccountRegistrar::Swap(ExternalNetworkServiceAccountRegistrar& rhs) NN_NOEXCEPT
{
    std::swap(m_Ptr, rhs.m_Ptr);
}
Result ExternalNetworkServiceAccountRegistrar::GetOAuthProcedureSessionId(SessionId* pOut) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    detail::Uuid sessionId;
    NN_RESULT_DO(m_Ptr->GetSessionId(&sessionId));
    std::memcpy(pOut, &sessionId, sizeof(sessionId));
    NN_RESULT_SUCCESS;
}
Result ExternalNetworkServiceAccountRegistrar::RegisterUser(Uid* pOutUid) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES_NOT_NULL(pOutUid);

    return m_Ptr->RegisterUser(pOutUid);
}
Result ExternalNetworkServiceAccountRegistrar::RegisterUserAs(const Uid& uid) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES(uid);

    return m_Ptr->RegisterUserWithUid(uid);
}
Result ExternalNetworkServiceAccountRegistrar::RegisterNetworkServiceAccountAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(m_Ptr->RegisterNetworkServiceAccountAsync(&asyncContextPtr));
    *pOutContext = AsyncContext(asyncContextPtr.Detach());
    NN_RESULT_SUCCESS;
}
Result ExternalNetworkServiceAccountRegistrar::RegisterNetworkServiceAccountAsync(AsyncContext* pOutContext, const Uid& uid) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES(uid);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(m_Ptr->RegisterNetworkServiceAccountWithUidAsync(&asyncContextPtr, uid));
    *pOutContext = AsyncContext(asyncContextPtr.Detach());
    NN_RESULT_SUCCESS;
}
Result ExternalNetworkServiceAccountRegistrar::GetNickname(Nickname* pOut) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    return GetNicknameImpl<>(m_Ptr, pOut);
}
Result ExternalNetworkServiceAccountRegistrar::GetProfileImage(size_t* pOutSizeActual, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    return GetProfileImageImpl<>(m_Ptr, pOutSizeActual, buffer, bufferSize);
}
Result ExternalNetworkServiceAccountRegistrar::GetNetworkServiceAccountId(NetworkServiceAccountId* pOut) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    return GetNetworkServiceAccountIdImpl<>(m_Ptr, pOut);
}
Result ExternalNetworkServiceAccountRegistrar::SetSystemProgramIdentification(const SystemProgramIdentification& identification) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    return m_Ptr->SetSystemProgramIdentification(identification, 0x00ull);
}
Result ExternalNetworkServiceAccountRegistrar::EnsureNetworkServiceAccountIdTokenCacheAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(m_Ptr->EnsureIdTokenCacheAsync(&asyncContextPtr));
    *pOutContext = AsyncContext(asyncContextPtr.Detach());
    NN_RESULT_SUCCESS;
}
Result ExternalNetworkServiceAccountRegistrar::LoadNetworkServiceAccountIdTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ExternalNetworkServiceAccountRegistrar);
    return LoadNetworkServiceAccountIdTokenCacheImpl<>(m_Ptr, pOutSizeActual, buffer, bufferSize);
}

} // ~namespace nn::account
}
