﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/result/result_HandlingUtility.h>
#include <nnd/tmp451/tmp451.h>

#include "tmp451_BusConfig.h"
#include "tmp451_Debug.h"
#include "tmp451_InternalFunction.h"
#include "tmp451_Specification.h"
#include "tmp451_SettingsMeasurment.h"

namespace nnd {
namespace tmp451 {
namespace detail {

nn::Result GetRemoteDigitalFilterControl(DigitalFilter* filter, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t data;

    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::DigitalFilterControl));

    switch (data & 0x03)
    {
    case 0x0:
        {
            *filter = DigitalFilter::FilterOff;
        }
        break;
    case 0x1:
        {
            *filter = DigitalFilter::Filter4Results;
        }
        break;
    case 0x2:
        {
            *filter = DigitalFilter::Filter8Results;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result SetRemoteDigitalFilterControl(DigitalFilter filter, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t data;

    switch (filter)
    {
    case DigitalFilter::FilterOff:
        {
            data = 0x0;
        }
        break;
    case DigitalFilter::Filter4Results:
        {
            data = 0x1;
        }
        break;
    case DigitalFilter::Filter8Results:
        {
            data = 0x2;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::DigitalFilterControl),
        data,
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;
}

nn::Result GetRemoteTemperatureOffset(Temperature* offset, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    TemperatureByteData data;

    MeasurementRange range;
    NN_RESULT_DO(GetMeasurementRange(&range));

    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::RemoteOffsetInt));
    NN_RESULT_DO(ReadRegister(&data.decimalByte, bus, RegRead::RemoteOffsetDec));
    *offset = ConvertByteData2Temperature(data, MeasurementRange::Standard);

    NN_RESULT_SUCCESS;
}

nn::Result SetRemoteTemperatureOffset(Temperature offset, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    TemperatureByteData data = ConvertTemperature2ByteData(offset, MeasurementRange::Standard);

    // 整数部
    uint8_t cmdSet[2] = { 0, 0 };
    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteOffsetInt);
    cmdSet[1] = data.integerByte;
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    // 小数部
    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteOffsetDec);
    cmdSet[1] = data.decimalByte;
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    NN_RESULT_SUCCESS;
}

nn::Result GetRemoteEtaFactorCorrectionValueByN(int8_t* n, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t data;

    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::EtaFactorCorrection));
    *n = static_cast<int8_t>(data & 0xF);
    NN_RESULT_SUCCESS;
}

nn::Result SetRemoteEtaFactorCorrectionValueByN(int8_t n, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::EtaFactorCorrection),
        static_cast<uint8_t>(n),
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;
}

} // detail
} // tmp451
} // nnd
