﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#pragma once

#include <nn/nn_Common.h>

#include <nn/dd.h>

#include <nn/util/util_BitPack.h>

#include "i2cTegra_DdUtil.h"

namespace nnd { namespace i2c { namespace tegra { namespace detail {

/**
 * @brief レジスタのビット定義 ([Ref1] 31.5)
 */
struct I2cRegister
{
    // 必要なものだけ抜粋して定義
    // [31.5.1]  cnfg
    typedef nn::util::BitPack32::Field< 12,  3, int  > Cnfg_Debounce;          // Debounce period for SDA/SCL lines
    typedef nn::util::BitPack32::Field< 10,  1, bool > Cnfg_PacketModeEn;      // Write "1" to initiate transfer in packet mode

    // [31.5.2]  cmdAddr0
    // [31.5.3]  cmdAddr1
    // [31.5.4]  cmdData1
    // [31.5.5]  cmdData2
    // [31.5.6]  status
    // [31.5.7]  slCnfg
    // [31.5.8]  slRcvd
    // [31.5.9]  slStatus
    // [31.5.10] slAddr1
    // [31.5.11] slAddr2
    // [31.5.12] tlowSext
    // [31.5.13] slDelayCount
    // [31.5.14] slIntMask
    // [31.5.15] slIntSource
    // [31.5.16] slIntSet
    // [31.5.17] txPacketFifo
    typedef nn::util::BitPack32::Field<  0, 32,  int > TxPacketFifo_TxPacket;

    // [31.5.18] rxFifo
    typedef nn::util::BitPack32::Field<  0, 32,  int > RxFifo_RdData;

    // [31.5.19] packetTransferStatus
    typedef nn::util::BitPack32::Field< 24,  1, int  > PacketTransferStatus_TransferComplete;
    typedef nn::util::BitPack32::Field< 16,  8, int  > PacketTransferStatus_TransferPktId;
    typedef nn::util::BitPack32::Field<  4, 12, int  > PacketTransferStatus_TransferByteNum;
    typedef nn::util::BitPack32::Field<  3,  1, bool > PacketTransferStatus_NoAckForAddr;
    typedef nn::util::BitPack32::Field<  2,  1, bool > PacketTransferStatus_NoAckForData;
    typedef nn::util::BitPack32::Field<  1,  1, bool > PacketTransferStatus_ArbLost;
    typedef nn::util::BitPack32::Field<  0,  1, bool > PacketTransferStatus_ControllerBusy;

    // [31.5.20] fifoControl
    typedef nn::util::BitPack32::Field<  5,  3, int  > FifoControl_TxFifoTrig;
    typedef nn::util::BitPack32::Field<  2,  3, int  > FifoControl_RxFifoTrig;
    typedef nn::util::BitPack32::Field<  1,  1, bool > FifoControl_TxFifoFlush;
    typedef nn::util::BitPack32::Field<  0,  1, bool > FifoControl_RxFifoFlush;

    // [31.5.21] fifoStatus
    typedef nn::util::BitPack32::Field< 25,  1, bool > FifoStatus_SlvXferErrReason;
    typedef nn::util::BitPack32::Field< 20,  4, int  > FifoStatus_SlvTxFifoEmptyCnt;
    typedef nn::util::BitPack32::Field< 16,  4, int  > FifoStatus_SlvRxFifoFullCnt;
    typedef nn::util::BitPack32::Field<  4,  4, int  > FifoStatus_TxFifoEmptyCnt;
    typedef nn::util::BitPack32::Field<  0,  4, int  > FifoStatus_RxFifoFullCnt;

    // [31.5.22] interruptMaskRegister
    typedef nn::util::BitPack32::Field< 11,  1, bool > InterruptMask_BusClearDoneIntEn;
    typedef nn::util::BitPack32::Field<  7,  1, bool > InterruptMask_PacketXferCompleteIntEn;
    typedef nn::util::BitPack32::Field<  6,  1, bool > InterruptMask_AllPacketsXferCompleteIntEn;
    typedef nn::util::BitPack32::Field<  5,  1, bool > InterruptMask_TfifoOvfIntEn;
    typedef nn::util::BitPack32::Field<  4,  1, bool > InterruptMask_RfifoUnfIntEn;
    typedef nn::util::BitPack32::Field<  3,  1, bool > InterruptMask_NoAckIntEn;
    typedef nn::util::BitPack32::Field<  2,  1, bool > InterruptMask_ArbLostIntEn;
    typedef nn::util::BitPack32::Field<  1,  1, bool > InterruptMask_TfifoDataReq;
    typedef nn::util::BitPack32::Field<  0,  1, bool > InterruptMask_RfifoDataReq;


    // [31.5.23] interruptStatusRegister
    typedef nn::util::BitPack32::Field<  7,  1, bool > InterruptStatus_PacketXferComplete;
    typedef nn::util::BitPack32::Field<  6,  1, bool > InterruptStatus_AllPacketsXferComplete;
    typedef nn::util::BitPack32::Field<  5,  1, bool > InterruptStatus_TfifoOvf;
    typedef nn::util::BitPack32::Field<  4,  1, bool > InterruptStatus_RfifoOvf;
    typedef nn::util::BitPack32::Field<  3,  1, bool > InterruptStatus_NoAck;
    typedef nn::util::BitPack32::Field<  2,  1, bool > InterruptStatus_ArbLost;
    typedef nn::util::BitPack32::Field<  1,  1, bool > InterruptStatus_TfifoDataReq;
    typedef nn::util::BitPack32::Field<  0,  1, bool > InterruptStatus_RfifoDataReq;

    // [31.5.24] clkDivisorRegister
    typedef nn::util::BitPack32::Field< 16, 16, int  > ClkDivisorRegister_StdFastMode; // Divisor for standard/fast mode
    typedef nn::util::BitPack32::Field<  0, 16, int  > ClkDivisorRegister_HsMode;      // Divisor for high speed mode

    // [31.5.25] interruptSourceRegister
    // [31.5.26] interruptSetRegister
    // [31.5.27] slvTxPacketFifo
    // [31.5.28] slvRxFifo
    // [31.5.29] slvPacketStatus
    // [31.5.30] busClearConfig
    typedef nn::util::BitPack32::Field< 16,  8, int  > BusClearConfig_BcSclkThreshold;
    typedef nn::util::BitPack32::Field<  2,  1, bool > BusClearConfig_BcStopCond;
    typedef nn::util::BitPack32::Field<  1,  1, bool > BusClearConfig_BcTerminate;
    typedef nn::util::BitPack32::Field<  0,  1, bool > BusClearConfig_BcEnable;

    // [31.5.31] busClearStatus
    typedef nn::util::BitPack32::Field<  0,  1, bool > BusClearStatus_BcStatus;

    // [31.5.32] configLoad
    typedef nn::util::BitPack32::Field<  0,  1, bool > ConfigLoad_MstrConfigLoad;

    // [31.5.33] interfaceTiming_0
    typedef nn::util::BitPack32::Field<  0, 14, int  > InterfaceTiming_0;
    typedef nn::util::BitPack32::Field<  8,  6, int  > InterfaceTiming_0_Thigh;
    typedef nn::util::BitPack32::Field<  0,  6, int  > InterfaceTiming_0_Tlow;

    // [31.5.34] interfaceTiming_1
    // [31.5.35] hsInterfaceTiming_0
    typedef nn::util::BitPack32::Field<  0, 14, int  > HsInterfaceTiming_0;
    typedef nn::util::BitPack32::Field<  8,  6, int  > HsInterfaceTiming_0_Thigh;
    typedef nn::util::BitPack32::Field<  0,  6, int  > HsInterfaceTiming_0_Tlow;

    // [31.5.36] hsInterfaceTiming_1
};


/**
 * @brief Packet のビット定義 ( [Ref1] Table 133 - Table 136 )
 */
struct I2cPacket
{
    // [Table 133]
    typedef nn::util::BitPack32::Field< 28,  2, int  > Generic0_ProtHdrSz;
    typedef nn::util::BitPack32::Field< 16,  8, int  > Generic0_PktId;
    typedef nn::util::BitPack32::Field< 12,  4, int  > Generic0_ControllerId;
    typedef nn::util::BitPack32::Field<  4,  4, int  > Generic0_Protocol;
    typedef nn::util::BitPack32::Field<  0,  3, int  > Generic0_PktType;

    // [Table 134]
    typedef nn::util::BitPack32::Field<  0, 12, int  > Generic1_PayloadSize;

    // [Table 135]
    typedef nn::util::BitPack32::Field< 25,  1, bool > Transmit_RespPktFreq;
    typedef nn::util::BitPack32::Field< 24,  1, bool > Transmit_RespPktEnable;
    typedef nn::util::BitPack32::Field< 22,  1, bool > Transmit_HsMode;
    typedef nn::util::BitPack32::Field< 21,  1, bool > Transmit_ContinueOnNack;
    typedef nn::util::BitPack32::Field< 20,  1, bool > Transmit_SendStartByte;
    typedef nn::util::BitPack32::Field< 19,  1, bool > Transmit_ReadWrite;
    typedef nn::util::BitPack32::Field< 18,  1, bool > Transmit_AddressMode;
    typedef nn::util::BitPack32::Field< 17,  1, bool > Transmit_Ie;
    typedef nn::util::BitPack32::Field< 16,  1, bool > Transmit_RepeatStartStop;
    typedef nn::util::BitPack32::Field< 15,  1, bool > Transmit_ContinueXfer;
    typedef nn::util::BitPack32::Field< 12,  3, int  > Transmit_HsMasterAddr;
    typedef nn::util::BitPack32::Field<  0, 10, int  > Transmit_SlaveAddr;

    // [Table 136]
    typedef nn::util::BitPack32::Field< 26,  1, bool > Response_RfifoOvf;
    typedef nn::util::BitPack32::Field< 25,  1, bool > Response_TfifoOvf;
    typedef nn::util::BitPack32::Field< 24,  1, bool > Response_TransferComplete;
    typedef nn::util::BitPack32::Field< 16,  1, bool > Response_TransferPktId;
    typedef nn::util::BitPack32::Field<  4, 12, int  > Response_TransferByteNum;
    typedef nn::util::BitPack32::Field<  3,  1, bool > Response_NoAckForAddr;
    typedef nn::util::BitPack32::Field<  2,  1, bool > Response_NoAckForData;
    typedef nn::util::BitPack32::Field<  1,  1, bool > Response_ArbLost;
    typedef nn::util::BitPack32::Field<  0,  1, bool > Response_ControllerBusy;
};

// @brief ベースレジスタからのオフセット参照のための構造体([Ref1] 31.5)
struct I2cRegTable
{
    volatile uint32_t cnfg;                      // [31.5.1]  0x00 I2C Controller Configuration Register (Master)
    volatile uint32_t cmdAddr0;                  // [31.5.2]  0x04 I2C Slave-1 Address
    volatile uint32_t cmdAddr1;                  // [31.5.3]  0x08 I2C Slave-2 Address
    volatile uint32_t cmdData1;                  // [31.5.4]  0x0c I2C Controller Data1: Transmit/Receive
    volatile uint32_t cmdData2;                  // [31.5.5]  0x10 I2C Controller Data2: Transmit/Receive
                      NN_PADDING4;               //           0x14
                      NN_PADDING4;               //           0x18
    volatile uint32_t status;                    // [31.5.6]  0x1c I2C Controller Status (Master)
    volatile uint32_t slCnfg;                    // [31.5.7]  0x20 I2C Controller Configuration (Slave)
    volatile uint32_t slRcvd;                    // [31.5.8]  0x24 I2C Controller Slave Receive/Transmit Data (Slave)
    volatile uint32_t slStatus;                  // [31.5.9]  0x28 I2C Controller Slave Status (Slave)
    volatile uint32_t slAddr1;                   // [31.5.10] 0x2c I2C Controller Slave Address 1 Register (Slave)
    volatile uint32_t slAddr2;                   // [31.5.11] 0x30 I2C Controller Slave Address 2 Register (Slave)
    volatile uint32_t tlowSext;                  // [31.5.12] 0x34 I2C Controller SMBUS Timeout Thresholds
                      NN_PADDING4;               //           0x38
    volatile uint32_t slDelayCount;              // [31.5.13] 0x3c I2C Slave Controller Delay Count
    volatile uint32_t slIntMask;                 // [31.5.14] 0x40 I2C Controller Slave Mask (Slave)
    volatile uint32_t slIntSource;               // [31.5.15] 0x44 I2C Controller Slave Source (Slave)
    volatile uint32_t slIntSet;                  // [31.5.16] 0x48 I2C Controller Slave Setting (Slave)
                      NN_PADDING4;               //           0x4c
    volatile uint32_t txPacketFifo;              // [31.5.17] 0x50 I2C Controller packet FIFO
    volatile uint32_t rxFifo;                    // [31.5.18] 0x54
    volatile uint32_t packetTransferStatus;      // [31.5.19] 0x58
    volatile uint32_t fifoControl;               // [31.5.20] 0x5c
    volatile uint32_t fifoStatus;                // [31.5.21] 0x60
    volatile uint32_t interruptMaskRegister;     // [31.5.22] 0x64
    volatile uint32_t interruptStatusRegister;   // [31.5.23] 0x68
    volatile uint32_t clkDivisorRegister;        // [31.5.24] 0x6c
    volatile uint32_t interruptSourceRegister;   // [31.5.25] 0x70
    volatile uint32_t interruptSetRegister;      // [31.5.26] 0x74
    volatile uint32_t slvTxPacketFifo;           // [31.5.27] 0x78
    volatile uint32_t slvRxFifo;                 // [31.5.28] 0x7c
    volatile uint32_t slvPacketStatus;           // [31.5.29] 0x80
    volatile uint32_t busClearConfig;            // [31.5.30] 0x84
    volatile uint32_t busClearStatus;            // [31.5.31] 0x88
    volatile uint32_t configLoad;                // [31.5.32] 0x8c
                      NN_PADDING4;               //           0x90
    volatile uint32_t interfaceTiming_0;         // [31.5.33] 0x94
    volatile uint32_t interfaceTiming_1;         // [31.5.34] 0x98
    volatile uint32_t hsInterfaceTiming_0;       // [31.5.35] 0x9c
    volatile uint32_t hsInterfaceTiming_1;       // [31.5.36] 0xa0
};

/**
 * @brief 指定したレジスタ空間のベースアドレスをもとに、指定したバスに対応するレジスタセット構造体へのポインタを返す
 */
NN_FORCEINLINE I2cRegTable* GetI2cReg(uint32_t address, uint32_t length) NN_NOEXCEPT
{
    // IO マッピングを取得する
    return reinterpret_cast<I2cRegTable*>(GetVirtualAddress(address, length));
}

}}}} // nnd::i2c::tegra::detail
