﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       タッチコントローラ ftm3bd56 のデバイス固有の定義です。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/i2c/i2c.h>
#include <nnd/ftm/ftm.h>
#include "ftm_Driver-3bd56.h"
#include "ftm_Util.h"

namespace nnd { namespace ftm { namespace detail {

namespace {

// Events ID のテーブル
struct EventIdSet
{
    EventId id;
    uint8_t val;
    uint8_t mask;   // タッチ系イベントIDのサイズ違い対応で使用
};
const EventIdSet EventIdTable[] =
{
    {EventId::NoEvents,                 0x00,   0xFF},
    {EventId::TouchEnter,               0x03,   0x0F},  // 下 4 bit のみ
    {EventId::TouchLeave,               0x04,   0x0F},  // 下 4 bit のみ
    {EventId::TouchMotion,              0x05,   0x0F},  // 下 4 bit のみ
    {EventId::Error,                    0x0F,   0xFF},
    {EventId::ControllerReady,          0x10,   0xFF},
    {EventId::SleepOutContollerReady,   0x11,   0xFF},
    {EventId::Status,                   0x16,   0xFF},
    {EventId::Inform,                   0x20,   0xFF},
    // {EventId::Others,},   未定義イベントには数字を割り当てない
};

// Orientation のテーブル
struct OrientationSet
{
    int32_t     milliDeg;
    uint8_t     val;
};
const OrientationSet OrientationTable[] =
{
    {      0, 0x00},
    {  45000, 0x01},
    {  90000, 0x02},
    { -45000, 0x03},
    {  22500, 0x04},
    {  67500, 0x05},
    { -22500, 0x06},
    { -67500, 0x07},
};

// Status のテーブル
struct StatusSet
{
    StatusType type;
    uint8_t    val;
};
const StatusSet StatusTable[] =
{
    {StatusType::AutoTuneMutual,                      0x01},
    {StatusType::FlashWriteConfig,                    0x03},
    {StatusType::FlashWriteNodeCompensationMemory,    0x04},
    {StatusType::ForceCalSelfAndMutual,               0x05},
    //{StatusType::Others,},   未定義イベントには数字を割り当てない
};

// FW Commands
enum class FwCommand : uint8_t
{
    ReadStatus                                             = 0x84,
    ReadOneEvent                                           = 0x85,
    ReadAllEvent                                           = 0x86,
    ReadLatestEvent                                        = 0x87,
    SleepIn                                                = 0x90,
    SleepOut                                               = 0x91,
    ScreenSenseOff                                         = 0x92,
    ScreenSenseOn                                          = 0x93,
    KeyOff                                                 = 0x9A,
    KeyOn                                                  = 0x9B,
    GloveOff                                               = 0x9E,
    GloveOn                                                = 0x9F,
    SystemReset                                            = 0xA0,
    ClearEventStackOrFlushBuffer                           = 0xA1,
    ForceCalibration                                       = 0xA2,
    MutualSenseAutoTune                                    = 0xA3,
    SelfSenseAutoTune                                      = 0xA4,
    ItoCheck                                               = 0xA7,
    InternalAndExternalReleaseInformation                  = 0xAA,
    SwWriteConfigurationFileToFlash                        = 0xFB,
    SwWriteRegisterFileToFlashInfoBlockForNodeCompensation = 0xFC,
};

// HW Commands
const uint8_t HwCommandReadInformation[]  = {0xB6, 0x00, 0x07};
const uint8_t HwCommandSystemReset[]      = {0xB6, 0x00, 0x23, 0x01}; // write
const uint8_t HwCommandInterruptOn[]      = {0xB6, 0x00, 0x1C, 0x41}; // write

// 取得可能な最大点数
const uint8_t MaxTouchNumber = 10;

// FIFO の情報
const uint8_t MaxEventReportCount = 32;
const size_t  EventReportByteSize = 8;

const uint32_t I2cRetryCountMax = 3; // リトライの最大回数
const uint32_t I2cAccessRetryIntervalMs = 5; // リトライ時のアクセス間隔
const uint32_t I2cCommandMaxDataBytes = 255; // Command List で受信可能な最大サイズ [byte]
const uint32_t CommandListMaxReceiveEvents = I2cCommandMaxDataBytes / EventReportByteSize;

} // namespace

Ftm3bd56Controller::Ftm3bd56Controller() NN_NOEXCEPT
    : IController()
{
    m_MaxTouchNumber = static_cast<int>(MaxTouchNumber);
    m_MaxEventReportCount = static_cast<int>(MaxEventReportCount);
    m_EventReportByteSize = static_cast<size_t>(EventReportByteSize);
}

::nn::Result Ftm3bd56Controller::ResetDevice() const NN_NOEXCEPT
{
    // ハードバグ回避のためのワークアラウンド
    ::nn::os::SleepThread(::nn::TimeSpan::FromSeconds(5));

    NN_RESULT_DO(this->WriteRegister(HwCommandSystemReset, sizeof(HwCommandSystemReset)));
    NN_RESULT_DO(this->WaitForControllerReady());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::SleepInDevice() const NN_NOEXCEPT
{
    auto cmd = FwCommand::SleepIn;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::SleepOutDevice() const NN_NOEXCEPT
{
    auto cmd = FwCommand::SleepOut;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::BindInterrupt() const NN_NOEXCEPT
{
    NN_RESULT_DO(this->WriteRegister(HwCommandInterruptOn, sizeof(HwCommandInterruptOn)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::ActivateSensing() const NN_NOEXCEPT
{
    auto cmd = FwCommand::ScreenSenseOn;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::DeactivateSensing() const NN_NOEXCEPT
{
    auto cmd = FwCommand::ScreenSenseOff;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::ReadInformation(char* pOutInformation, size_t size) const NN_NOEXCEPT
{
    const size_t InformationSize = 7;
    char information[InformationSize];

    // 先頭にダミーが返るので、最低 2 バイト読まないと意味のある値は取得できない
    NN_ASSERT_MINMAX(size, static_cast<size_t>(2), InformationSize);

    NN_RESULT_DO(this->ReadRegister(information, InformationSize, HwCommandReadInformation, sizeof(HwCommandReadInformation)));

    ::std::memcpy(pOutInformation, information, size);

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::ReadLeftEventCount(uint32_t* pOutLeftCount) const NN_NOEXCEPT
{
    // ftm4cd60d との互換のために用意する API
    // 効率は悪くなるが最大数を返しておけば動作はする
    // ftm3bd56 がすでに非サポートのためこの対処で凌ぐ
    *pOutLeftCount = MaxEventReportCount;

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::ReadEventReports(char* pOutReadData, uint32_t* pOutReadCount, bool* pOutIsOverflow, uint32_t readCount) const NN_NOEXCEPT
{
    uint32_t leftCount;
    NN_RESULT_DO(this->ReadEventReports(pOutReadData, pOutReadCount, &leftCount, pOutIsOverflow, readCount));

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm3bd56Controller::ReadEventReports(char* pOutReadData, uint32_t* pOutReadCount, uint32_t* pOutLeftCount, bool* pOutIsOverflow, uint32_t readCount) const NN_NOEXCEPT
{
    // 一度に読み出し可能な上限数を超えていないかの確認
    NN_RESULT_THROW_UNLESS(readCount <= static_cast<uint32_t>(MaxEventReportCount), ResultInvalidArgument());

    size_t offset = 0;
    uint32_t leftCount = readCount;
    while (leftCount > 0)
    {
        uint32_t actReadCount = (leftCount < CommandListMaxReceiveEvents) ? leftCount : CommandListMaxReceiveEvents;
        size_t readByteSize = actReadCount * EventReportByteSize;
        auto cmd = static_cast<uint8_t>((actReadCount == 1) ? FwCommand::ReadOneEvent : FwCommand::ReadAllEvent);

        uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
        ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
        ::std::memset(commandList, 0, sizeof(commandList));

        commandListFormatter.EnqueueSendCommand(I2cTransStart, &cmd, sizeof(cmd));
        commandListFormatter.EnqueueReceiveCommand(I2cTransStop, readByteSize);
        NN_RESULT_DO(::nn::i2c::ExecuteCommandList(pOutReadData + reinterpret_cast<uintptr_t>(offset), readByteSize, m_Session,
                                                   commandList, commandListFormatter.GetCurrentLength()));

        offset += actReadCount * EventReportByteSize; // 受信バッファに加えるオフセットをすすめる
        leftCount -= actReadCount; // 残りイベント数更新
    }

    // 一つもイベントを取得しなかった時には残数やオーバーフローの発生状況は取得できない
    if (readCount != 0)
    {
        // FIFO に残っているイベントの数を取得
        // 最も新しいイベントレポートの byte7 を確認する
        *pOutLeftCount = (pOutReadData[readCount * EventReportByteSize - 1] & (MaxEventReportCount - 1));

        // オーバーフローが発生したかどうか
        // 最も古いイベントレポートの byte7 を確認する
        *pOutIsOverflow = (((pOutReadData[EventReportByteSize - 1] >> 5) & 0x01) != 0) ? true : false;
    }

    *pOutReadCount = readCount;

    NN_RESULT_SUCCESS;
}

void Ftm3bd56Controller::ParseEventReports(EventReport* pOutEventReport, const char* pRawData, uint32_t parseCount) const NN_NOEXCEPT
{
    // 取得したバイトデータ配列を EventReport 形式に変換。
    for (uint32_t event = 0; event < parseCount; event++)
    {
        // Event ID の変換
        uint8_t eId = pRawData[event * EventReportByteSize];
        pOutEventReport[event].eventId = this->ConvertEventValToEnum(eId);

        // Event ID 別コンテンツの変換
        switch (pOutEventReport[event].eventId)
        {
        // タッチ
        case EventId::TouchEnter:
        case EventId::TouchLeave:
        case EventId::TouchMotion:
            {
                pOutEventReport[event].content.touchReport = this->ParseTouchContent(&pRawData[event * EventReportByteSize]);
            }
            break;
        // ステータス
        case EventId::Status:
            {
                pOutEventReport[event].content.statusReport = this->ParseStatusContent(&pRawData[event * EventReportByteSize]);
            }
            break;
        // GPIO ステート (コントローラに繋がるパネルの種類判別用)
        case EventId::Inform:
            {
                pOutEventReport[event].content.gpioReport = this->ParseGpioStateContent(&pRawData[event * EventReportByteSize]);
            }
            break;
        // コンテンツを持たないその他イベント
        case EventId::Error:
        case EventId::NoEvents:
        case EventId::ControllerReady:
        case EventId::SleepOutContollerReady:
        case EventId::Others:
            {
                // EventID 以外データは無いので、何もしない。
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

::nn::Result Ftm3bd56Controller::WaitForControllerReady() const NN_NOEXCEPT
{
    char data[MaxEventReportCount * EventReportByteSize];
    uint32_t outReadCount;
    uint32_t leftCount;
    bool isOverflow;

    ::nn::os::Tick tick = ::nn::os::GetSystemTick();
    do {
        ::nn::TimeSpan timeSpan = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick() - tick);

        // 仕様上 10ms 以内に Ready が返ってくるが、オシレータがトリミングされてない場合はその限りでない
        // 納品された ftm3bd56 はトリミングされていないため、相当な余裕をみて 1000ms 以内には Ready になると考える
        if (timeSpan.GetMilliSeconds() > 1000)
        {
            // Ready イベントが返ってきているものの FIFO の読みが到達していない可能性があるので
            // 1000ms 経ったタイミングで FIFO の中身をすべて読み出す
            NN_RESULT_DO(this->ReadEventReports(data, &outReadCount, &leftCount, &isOverflow, 1));
            if (this->ConvertEventValToEnum(data[0]) == EventId::ControllerReady)
            {
                NN_RESULT_SUCCESS;
            }

            uint32_t readCount = leftCount;
            NN_RESULT_DO(this->ReadEventReports(data, &outReadCount, &leftCount, &isOverflow, readCount));
            for (uint8_t event = 0; event < readCount; event++)
            {
                if (this->ConvertEventValToEnum(data[event * EventReportByteSize]) == EventId::ControllerReady)
                {
                    NN_RESULT_SUCCESS;
                }
            }

            // また FW によっては Ready イベントが返らないことがあるためエラーは返さない
            //NN_RESULT_THROW(ResultTimeoutReady());
            break;
        }

        NN_RESULT_DO(this->ReadEventReports(data, &outReadCount, &leftCount, &isOverflow, 1));

    } while (this->ConvertEventValToEnum(data[0]) != EventId::ControllerReady);

    NN_RESULT_SUCCESS;
}

EventId Ftm3bd56Controller::ConvertEventValToEnum(uint8_t idVal) const NN_NOEXCEPT
{
    for (const auto& event : EventIdTable)
    {
        if (event.val == (idVal & event.mask))
        {
            return event.id;
        }
    }
    return EventId::Others;
}

TouchEventReport Ftm3bd56Controller::ParseTouchContent(const char* pRawData) const NN_NOEXCEPT
{
    TouchEventReport touchContent;

    touchContent.touchId = (pRawData[0] >> 4) & 0x0F;
    touchContent.x = ((pRawData[1] << 4) & 0xFF0) | ((pRawData[3] >> 4) & 0x0F);
    touchContent.y = ((pRawData[2] << 4) & 0xFF0) | (pRawData[3] & 0x0F);

    uint16_t z_coor = ((pRawData[5] << 8) & 0xFF00) | (pRawData[4] & 0x00FF);
    uint16_t minor = pRawData[6] & 0x3F;
    touchContent.major = (z_coor * 63) / (63 + minor) ;
    touchContent.minor = (z_coor * minor) / (63 + minor) ;

    uint8_t ori = ((pRawData[6] >> 6) & 0x03) | (pRawData[7] >> 5 & 0x04);
    touchContent.milliDegree = this->ConvertOrientationValToEnum(ori);

    return touchContent;
}

StatusType Ftm3bd56Controller::ParseStatusContent(const char* pRawData) const NN_NOEXCEPT
{
    return this->ConvertStatusValToEnum(pRawData[1]);
}

GpioInformEventReport Ftm3bd56Controller::ParseGpioStateContent(const char* pRawData) const NN_NOEXCEPT
{
    GpioInformEventReport gpioContent;
    gpioContent.gpio0 = pRawData[2];
    gpioContent.gpio1 = pRawData[3];
    gpioContent.gpio2 = pRawData[4];

    return gpioContent;
}

int32_t Ftm3bd56Controller::ConvertOrientationValToEnum(uint8_t orientationVal) const NN_NOEXCEPT
{
    for (const auto& orientation : OrientationTable)
    {
        if (orientation.val == orientationVal)
        {
            return orientation.milliDeg;
        }
    }
    return 0; // 未定義の値の場合、0 を返す
}

StatusType Ftm3bd56Controller::ConvertStatusValToEnum(uint8_t statusVal) const NN_NOEXCEPT
{
    for (const auto& status : StatusTable)
    {
        if (status.val == statusVal)
        {
            return status.type;
        }
    }
    return StatusType::Others;
}

::nn::Result Ftm3bd56Controller::WriteRegister(const void* pInData, size_t dataBytes) const NN_NOEXCEPT
{
    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        result = ::nn::i2c::Send(m_Session, pInData, dataBytes, I2cTransStartStop);

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));
            continue;
        }

        break;
    }

    return result;
}

::nn::Result Ftm3bd56Controller::ReadRegister(
    void* pBuffer, size_t receiveDataBytes, const nn::Bit8* pInData, size_t sendDataBytes) const NN_NOEXCEPT
{
    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
        ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
        ::std::memset(commandList, 0, sizeof(commandList));

        commandListFormatter.EnqueueSendCommand(I2cTransStart, pInData, sendDataBytes);
        commandListFormatter.EnqueueReceiveCommand(I2cTransStop, receiveDataBytes);

        result = ::nn::i2c::ExecuteCommandList(
            pBuffer, receiveDataBytes, m_Session, commandList, commandListFormatter.GetCurrentLength());

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));
            continue;
        }

        break;
    }

    return result;
}

}}} // namespace nnd::ftm::detail
