﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief       tmp451 ライブラリの温度計測の有効化と読み取り専用情報の取得の API リストです。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_Types.h"

namespace nnd {
namespace tmp451 {

//! @name 温度計測の有効化と読み取り専用情報の取得
//! @{

/**
 * @brief       現在の動作モードを取得します。
 *
 * @return      現在の動作モードを @ref MeasurementMode で返します。
 * @retval      Continuous  Continuous モード
 * @retval      OneShot     OneShot モード
 * @retval      Standby     Standby モード
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @detail      現在の動作モードを取得します。@n
 *              @ref nnd::tmp451::Initialize() による初期化直後は Standby を返します。 @n
 */
MeasurementMode GetMeasurementMode() NN_NOEXCEPT;

/**
 * @brief       指定する動作モードに設定します。
 *
 * @param[in]   mode    開始する動作モード
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                     指定する動作モードへの遷移に成功しました。
 * @retval      nn::ResultInvalidState
 *                     不正なステートであるため、処理を実行出来ませんでした。
 *                     設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         初期化済み状態である必要があります。 @n
 *              また、本関数により各動作モードに遷移させるためには、
 *              本関数を実行する直前の状態が以下に示す状態である必要があります。
 *              @li OneShot に遷移可能な状態: Standby 状態
 *              @li Continuous に遷移可能な状態: Standby 状態
 *              @li Standby に遷移可能な状態: Continuous 状態
 *
 * @post        設定するモードに遷移します。
 *
 * @detail      指定する温度計測モードで温度計測を開始します。
 */
nn::Result SetMeasurementMode(MeasurementMode mode) NN_NOEXCEPT;

/**
 * @brief       指定する地点の温度を読み取ります。
 *
 * @param[out]  pOutTemperatute 温度データ
 * @param[in]   location        取得したい計測地点
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @detail      指定する温度計測地点の温度値を取得します。 @n
 *              有効な値を取得するためには @ref nnd::tmp451::StartMeasurement()
 *              による温度計測をあらかじめ行っておく必要があります。
 */
nn::Result ReadTemperature(Temperature* pOutTemperatute, Location location) NN_NOEXCEPT;

/**
 * @brief       ステータスを読み取ります。
 *
 * @param[out]  pOutStatus  ステータス
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @detail      デバイスのステータスを取得します。
 */
nn::Result ReadStatus(Status* pOutStatus) NN_NOEXCEPT;

/**
 * @brief       製造IDを読み取ります。
 *
 * @param[out]  pOutStatus  ステータス
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @detail      デバイスの製造IDを取得します。期待される値は 0x55 です。
 */
nn::Result ReadManufacturerId(uint8_t* pOutId) NN_NOEXCEPT;

//! @}



} // tmp451
} // nnd
