﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ftm ライブラリの検査工程開発者向けの API リストです。
 */

#pragma once

#include <nnd/ftm/ftm_Types.h>
#include <nnd/ftm/ftm_TypesFab.h>

namespace nnd {
namespace ftm {

//! @name 検査処理
//! @{

/**
 * @brief       ITO テストを実施します。
 *
 * @param[out]  pOutItoEventReport    ITO テストの結果
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutItoTest ITO テストがタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::ActivateSensing() によりセンシング有効な状態である必要があります。
 *
 * @details     ITO テストを実施して解析結果をレポートします。
 */
nn::Result RunItoTest(ItoEventReport* pOutItoEventReport) NN_NOEXCEPT;

//! @}

//! @name 内部データ読み出し処理
//! @{

/**
 * @brief       相互補償データを読み出します。
 *
 * @param[out]  pOutCompensationData    読み出した相互補償データ
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutCompensation Compensation データ待ちがタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @post        FTM 内部の FIFO に蓄積された EventReport は全て破棄され、センシングが無効化されます。
 *              センシングを有効化するためには、
 *              @ref nnd::ftm::SleepOutDevice() によるアクティブ状態への遷移と、
 *              @ref nnd::ftm::ActivateSensing() によるセンシングの有効化が必要です。
 *
 * @details     FTM 内部の相互補償データを読み出します。
 */
nn::Result ReadMutualCompensationData(MutualCompensationData* pOutCompensationData) NN_NOEXCEPT;

/**
 * @brief       自己補償データを読み出します。
 *
 * @param[out]  pOutCompensationData    読み出した自己補償データ
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutCompensation Compensation データ待ちがタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @post        FTM 内部の FIFO に蓄積された EventReport は全て破棄され、センシングが無効化されます。
 *              センシングを有効化するためには、
 *              @ref nnd::ftm::SleepOutDevice() によるアクティブ状態への遷移と、
 *              @ref nnd::ftm::ActivateSensing() によるセンシングの有効化が必要です。
 *
 * @details     FTM 内部の自己補償データを読み出します。
 */
nn::Result ReadSelfCompensationData(SelfCompensationData* pOutCompensationData) NN_NOEXCEPT;

/**
 * @brief       生の相互データを読み出します。
 *
 * @param[out]  pOutRawData    読み出した生の相互データ
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @post        FTM 内部の FIFO に蓄積された EventReport は全て破棄され、センシングが無効化されます。
 *              センシングを有効化するためには、
 *              @ref nnd::ftm::SleepOutDevice() によるアクティブ状態への遷移と、
 *              @ref nnd::ftm::ActivateSensing() によるセンシングの有効化が必要です。
 *
 * @details     FTM 内部の生の相互データを読み出します。
 */
nn::Result ReadMutualRawData(MutualRawData* pOutRawData) NN_NOEXCEPT;

/**
 * @brief       生の自己データを読み出します。
 *
 * @param[out]  pOutRawData    読み出した生の自己データ
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @post        FTM 内部の FIFO に蓄積された EventReport は全て破棄され、センシングが無効化されます。
 *              センシングを有効化するためには、
 *              @ref nnd::ftm::SleepOutDevice() によるアクティブ状態への遷移と、
 *              @ref nnd::ftm::ActivateSensing() によるセンシングの有効化が必要です。
 *
 * @details     FTM 内部の生の自己データを読み出します。
 */
nn::Result ReadSelfRawData(SelfRawData* pOutRawData) NN_NOEXCEPT;

//! @}


} // ftm
} // nnd
