﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   Standard PCI definitions (C language)
 */

enum nnpcieStandardConfigSpace
{
    nnpcieSTD_HEADER_SIZEOF           = 64,
    nnpcieCFG_SPACE_SIZE              = 256,
    nnpcieCFG_SPACE_EXP_SIZE          = 4096,

    nnpcieVENDOR_ID                   = 0x00,      // 16 bits
    nnpcieDEVICE_ID                   = 0x02,      // 16 bits
    nnpcieCOMMAND                     = 0x04,      // 16 bits
    nnpcieCOMMAND_IO                  = 0x1,       // Enable response in I/O space
    nnpcieCOMMAND_MEMORY              = 0x2,       // Enable response in Memory space
    nnpcieCOMMAND_MASTER              = 0x4,       // Enable bus mastering
    nnpcieCOMMAND_SPECIAL             = 0x8,       // Enable response to special cycles
    nnpcieCOMMAND_INVALIDATE          = 0x10,      // Use memory write and invalidate
    nnpcieCOMMAND_VGA_PALETTE         = 0x20,      // Enable palette snooping
    nnpcieCOMMAND_PARITY              = 0x40,      // Enable parity checking
    nnpcieCOMMAND_WAIT                = 0x80,      // Enable address/data stepping
    nnpcieCOMMAND_SERR                = 0x100,     // Enable SERR
    nnpcieCOMMAND_FAST_BACK           = 0x200,     // Enable back-to-back writes
    nnpcieCOMMAND_INTX_DISABLE        = 0x400,     // INTx Emulation Disable
    nnpcieSTATUS                      = 0x06,      // 16 bits
    nnpcieSTATUS_INTERRUPT            = 0x08,      // Interrupt status
    nnpcieSTATUS_CAP_LIST             = 0x10,      // Support Capability List
    nnpcieSTATUS_66MHZ                = 0x20,      // Support 66 MHz PCI 2.1 bus
    nnpcieSTATUS_UDF                  = 0x40,      // Support User Definable Features [obsolete]
    nnpcieSTATUS_FAST_BACK            = 0x80,      // Accept fast-back to back
    nnpcieSTATUS_PARITY               = 0x100,     // Detected parity error
    nnpcieSTATUS_DEVSEL_MASK          = 0x600,     // DEVSEL timing
    nnpcieSTATUS_DEVSEL_FAST          = 0x000,
    nnpcieSTATUS_DEVSEL_MEDIUM        = 0x200,
    nnpcieSTATUS_DEVSEL_SLOW          = 0x400,
    nnpcieSTATUS_SIG_TARGET_ABORT     = 0x800,     // Set on target abort
    nnpcieSTATUS_REC_TARGET_ABORT     = 0x1000,    // Master ack of "
    nnpcieSTATUS_REC_MASTER_ABORT     = 0x2000,    // Set on master abort
    nnpcieSTATUS_SIG_SYSTEM_ERROR     = 0x4000,    // Set when we drive SERR
    nnpcieSTATUS_DETECTED_PARITY      = 0x8000,    // Set on parity error
    nnpcieCLASS_REVISION              = 0x08,      // High 24 bits are class, low 8 revision
    nnpcieREVISION_ID                 = 0x08,      // Revision ID
    nnpcieCLASS_PROG                  = 0x09,      // Reg. Level Programming Interface
    nnpcieCLASS_DEVICE                = 0x0a,      // Device class
    nnpcieCACHE_LINE_SIZE             = 0x0c,      // 8-bit
    nnpcieLATENCY_TIMER               = 0x0d,      // 8-bit
    nnpcieHEADER_TYPE                 = 0x0e,      // 8-bit
    nnpcieHEADER_TYPE_NORMAL          = 0,
    nnpcieHEADER_TYPE_BRIDGE          = 1,
    nnpcieHEADER_TYPE_CARDBUS         = 2,
    nnpcieBIST                        = 0x0f,      // 8-bit
    nnpcieBIST_CODE_MASK              = 0x0f,      // Return result
    nnpcieBIST_START                  = 0x40,      // 1 to start BIST, 2 secs or less
    nnpcieBIST_CAPABLE                = 0x80,      // 1 if BIST capable
    nnpcieMAX_BASE_ADDRESSES          = 6,
    nnpcieBASE_ADDRESS_0              = 0x10,      // 32-bit
    nnpcieBASE_ADDRESS_1              = 0x14,      // 32 bits [htype 0,1 only]
    nnpcieBASE_ADDRESS_2              = 0x18,      // 32 bits [htype 0 only]
    nnpcieBASE_ADDRESS_3              = 0x1c,      // 32-bit
    nnpcieBASE_ADDRESS_4              = 0x20,      // 32-bit
    nnpcieBASE_ADDRESS_5              = 0x24,      // 32-bit
    nnpcieBASE_ADDRESS_SPACE          = 0x01,      // 0 = memory, 1 = I/O
    nnpcieBASE_ADDRESS_SPACE_IO       = 0x01,
    nnpcieBASE_ADDRESS_SPACE_MEMORY   = 0x00,
    nnpcieBASE_ADDRESS_MEM_TYPE_MASK  = 0x06,
    nnpcieBASE_ADDRESS_MEM_TYPE_32    = 0x00,      // 32 bit address
    nnpcieBASE_ADDRESS_MEM_TYPE_1M    = 0x02,      // Below 1M [obsolete]
    nnpcieBASE_ADDRESS_MEM_TYPE_64    = 0x04,      // 64 bit address
    nnpcieBASE_ADDRESS_MEM_PREFETCH   = 0x08,      // prefetchable?
    nnpcieBASE_ADDRESS_MEM_MASK       = ( ~0x0fUL),
    nnpcieBASE_ADDRESS_IO_MASK        = ( ~0x03UL),
};

// Typical PCI devices are of type '0'
enum nnpcieHeaderType0ConfigSpace
{
    nnpcieHTYPE_0_MAX_BASE_ADDRESSES  = 6,
    nnpcieCARDBUS_CIS                 = 0x28,
    nnpcieSUBSYSTEM_VENDOR_ID         = 0x2c,
    nnpcieSUBSYSTEM_ID                = 0x2e,
    nnpcieROM_ADDRESS                 = 0x30,    // Bits 31..11 are address, 10..1 reserved
    nnpcieROM_ADDRESS_ENABLE          = 0x01,
    nnpcieROM_ADDRESS_MASK            = ( ~0x7ffUL),
    nnpcieCAPABILITY_LIST             = 0x34,    // Offset of first capability list entry
    nnpcieINTERRUPT_LINE              = 0x3c,    // 8-bit
    nnpcieINTERRUPT_PIN               = 0x3d,    // 8-bit
    nnpcieINTERRUPT_PIN_INTA          = 1,
    nnpcieINTERRUPT_PIN_INTB          = 2,
    nnpcieINTERRUPT_PIN_INTC          = 3,
    nnpcieINTERRUPT_PIN_INTD          = 4,
    nnpcieMIN_GNT                     = 0x3e,    // 8-bit
    nnpcieMAX_LAT                     = 0x3f,    // 8-bit
};

// PCI-to-PCI bridges are of type '1'
enum nnpcieHeaderType1ConfigSpace
{
    nnpcieHTYPE_1_MAX_BASE_ADDRESSES  = 2,
    nnpciePRIMARY_BUS                 = 0x18,      // Primary bus number
    nnpcieSECONDARY_BUS               = 0x19,      // Secondary bus number
    nnpcieSUBORDINATE_BUS             = 0x1a,      // Highest bus number behind the bridge
    nnpcieSEC_LATENCY_TIMER           = 0x1b,      // Latency timer for secondary interface
    nnpcieIO_BASE                     = 0x1c,      // I/O range behind the bridge
    nnpcieIO_LIMIT                    = 0x1d,
    nnpcieIO_RANGE_TYPE_MASK          = 0x0fUL,    // I/O bridging type
    nnpcieIO_RANGE_TYPE_16            = 0x00,
    nnpcieIO_RANGE_TYPE_32            = 0x01,
    nnpcieIO_RANGE_MASK               = ( ~0x0fUL), // Standard 4K I/O windows
    nnpcieIO_1K_RANGE_MASK            = ( ~0x03UL), // Intel 1K I/O windows
    nnpcieSEC_STATUS                  = 0x1e,      // Secondary status register, only bit 14 used
    nnpcieMEMORY_BASE                 = 0x20,      // Memory range behind
    nnpcieMEMORY_LIMIT                = 0x22,
    nnpcieMEMORY_RANGE_TYPE_MASK      = 0x0fUL,
    nnpcieMEMORY_RANGE_MASK           = ( ~0x0fUL),
    nnpciePREF_MEMORY_BASE            = 0x24,      // Prefetchable memory range behind
    nnpciePREF_MEMORY_LIMIT           = 0x26,
    nnpciePREF_RANGE_TYPE_MASK        = 0x0fUL,
    nnpciePREF_RANGE_TYPE_32          = 0x00,
    nnpciePREF_RANGE_TYPE_64          = 0x01,
    nnpciePREF_RANGE_MASK             = ( ~0x0fUL),
    nnpciePREF_BASE_UPPER32           = 0x28,      // Upper half of prefetchable memory range
    nnpciePREF_LIMIT_UPPER32          = 0x2c,
    nnpcieIO_BASE_UPPER16             = 0x30,      // Upper half of I/O addresses
    nnpcieIO_LIMIT_UPPER16            = 0x32,
    // 0x34 same as for htype 0
    // 0x35-0x3b is reserved
    nnpcieROM_ADDRESS1                = 0x38,      // Same as nnpcieROM_ADDRESS, but for htype 1
                                                  // 0x3c-0x3d are same as for htype 0
    nnpcieBRIDGE_CONTROL              = 0x3e,
    nnpcieBRIDGE_CTL_PARITY           = 0x01,      // Enable parity detection on secondary interface
    nnpcieBRIDGE_CTL_SERR             = 0x02,      // The same for SERR forwarding
    nnpcieBRIDGE_CTL_ISA              = 0x04,      // Enable ISA mode
    nnpcieBRIDGE_CTL_VGA              = 0x08,      // Forward VGA addresses
    nnpcieBRIDGE_CTL_MASTER_ABORT     = 0x20,      // Report master aborts
    nnpcieBRIDGE_CTL_BUS_RESET        = 0x40,      // Secondary bus reset
    nnpcieBRIDGE_CTL_FAST_BACK        = 0x80,      // Fast Back2Back enabled on secondary interface
};

// CardBus bridges are of type '2'
enum nnpcieHeaderType2ConfigSpace
{
    nnpcieCB_CAPABILITY_LIST          = 0x14,
    // 0x15 reserved
    nnpcieCB_SEC_STATUS               = 0x16,    // Secondary status
    nnpcieCB_PRIMARY_BUS              = 0x18,    // PCI bus number
    nnpcieCB_CARD_BUS                 = 0x19,    // CardBus bus number
    nnpcieCB_SUBORDINATE_BUS          = 0x1a,    // Subordinate bus number
    nnpcieCB_LATENCY_TIMER            = 0x1b,    // CardBus latency timer
    nnpcieCB_MEMORY_BASE_0            = 0x1c,
    nnpcieCB_MEMORY_LIMIT_0           = 0x20,
    nnpcieCB_MEMORY_BASE_1            = 0x24,
    nnpcieCB_MEMORY_LIMIT_1           = 0x28,
    nnpcieCB_IO_BASE_0                = 0x2c,
    nnpcieCB_IO_BASE_0_HI             = 0x2e,
    nnpcieCB_IO_LIMIT_0               = 0x30,
    nnpcieCB_IO_LIMIT_0_HI            = 0x32,
    nnpcieCB_IO_BASE_1                = 0x34,
    nnpcieCB_IO_BASE_1_HI             = 0x36,
    nnpcieCB_IO_LIMIT_1               = 0x38,
    nnpcieCB_IO_LIMIT_1_HI            = 0x3a,
    nnpcieCB_IO_RANGE_MASK            = ~0x03,
    // 0x3c-0x3d are same as for htype 0
    nnpcieCB_BRIDGE_CONTROL           = 0x3e,
    nnpcieCB_BRIDGE_CTL_PARITY        = 0x01,   // Similar to standard bridge control register
    nnpcieCB_BRIDGE_CTL_SERR          = 0x02,
    nnpcieCB_BRIDGE_CTL_ISA           = 0x04,
    nnpcieCB_BRIDGE_CTL_VGA           = 0x08,
    nnpcieCB_BRIDGE_CTL_MASTER_ABORT  = 0x20,
    nnpcieCB_BRIDGE_CTL_CB_RESET      = 0x40,    // CardBus reset
    nnpcieCB_BRIDGE_CTL_16BIT_INT     = 0x80,    // Enable interrupt for 16-bit cards
    nnpcieCB_BRIDGE_CTL_PREFETCH_MEM0 = 0x100,   // Prefetch enable for both memory regions
    nnpcieCB_BRIDGE_CTL_PREFETCH_MEM1 = 0x200,
    nnpcieCB_BRIDGE_CTL_POST_WRITES   = 0x400,
    nnpcieCB_SUBSYSTEM_VENDOR_ID      = 0x40,
    nnpcieCB_SUBSYSTEM_ID             = 0x42,
    nnpcieCB_LEGACY_MODE_BASE         = 0x44,    // 16-bit PC Card legacy mode base address (ExCa)
                                                // 0x48-0x7f reserved
};

// Capability lists
enum nnpcieCapabilityLists
{
    nnpcieCAP_LIST_ID         = 0,       // Capability ID
    nnpcieCAP_ID_PM           = 0x01,    // Power Management
    nnpcieCAP_ID_AGP          = 0x02,    // Accelerated Graphics Port
    nnpcieCAP_ID_VPD          = 0x03,    // Vital Product Data
    nnpcieCAP_ID_SLOTID       = 0x04,    // Slot Identification
    nnpcieCAP_ID_MSI          = 0x05,    // Message Signalled Interrupts
    nnpcieCAP_ID_CHSWP        = 0x06,    // CompactPCI HotSwap
    nnpcieCAP_ID_PCIX         = 0x07,    // PCI-X
    nnpcieCAP_ID_HT           = 0x08,    // HyperTransport
    nnpcieCAP_ID_VNDR         = 0x09,    // Vendor-Specific
    nnpcieCAP_ID_DBG          = 0x0A,    // Debug port
    nnpcieCAP_ID_CCRC         = 0x0B,    // CompactPCI Central Resource Control
    nnpcieCAP_ID_SHPC         = 0x0C,    // PCI Standard Hot-Plug Controller
    nnpcieCAP_ID_SSVID        = 0x0D,    // Bridge subsystem vendor/device ID
    nnpcieCAP_ID_AGP3         = 0x0E,    // AGP Target PCI-PCI bridge
    nnpcieCAP_ID_SECDEV       = 0x0F,    // Secure Device
    nnpcieCAP_ID_EXP          = 0x10,    // PCI Express
    nnpcieCAP_ID_MSIX         = 0x11,    // MSI-X
    nnpcieCAP_ID_SATA         = 0x12,    // SATA Data/Index Conf.
    nnpcieCAP_ID_AF           = 0x13,    // PCI Advanced Features
    nnpcieCAP_ID_MAX          = nnpcieCAP_ID_AF,
    nnpcieCAP_LIST_NEXT       = 1,       // Next capability in the list
    nnpcieCAP_FLAGS           = 2,       // Capability defined flags (16 bits)
    nnpcieCAP_SIZEOF          = 4,

    nnpcieFIND_CAPABILITY_LIMIT = 48
};

// Power Management
enum nnpciePowerManagement
{
    nnpciePM_PMC                  = 2,       // PM Capabilities Register
    nnpciePM_CAP_VER_MASK         = 0x0007,  // Version
    nnpciePM_CAP_PME_CLOCK        = 0x0008,  // PME clock required
    nnpciePM_CAP_RESERVED         = 0x0010,  // Reserved field
    nnpciePM_CAP_DSI              = 0x0020,  // Device specific initialization
    nnpciePM_CAP_AUX_POWER        = 0x01C0,  // Auxiliary power support mask
    nnpciePM_CAP_D1               = 0x0200,  // D1 power state support
    nnpciePM_CAP_D2               = 0x0400,  // D2 power state support
    nnpciePM_CAP_PME              = 0x0800,  // PME pin supported
    nnpciePM_CAP_PME_MASK         = 0xF800,  // PME Mask of all supported states
    nnpciePM_CAP_PME_D0           = 0x0800,  // PME# from D0
    nnpciePM_CAP_PME_D1           = 0x1000,  // PME# from D1
    nnpciePM_CAP_PME_D2           = 0x2000,  // PME# from D2
    nnpciePM_CAP_PME_D3           = 0x4000,  // PME# from D3 (hot)
    nnpciePM_CAP_PME_D3cold       = 0x8000,  // PME# from D3 (cold)
    nnpciePM_CAP_PME_SHIFT        = 11,      // Start of the PME Mask in PMC
    nnpciePM_CTRL                 = 4,       // PM control and status register
    nnpciePM_CTRL_STATE_MASK      = 0x0003,  // Current power state (D0 to D3)
    nnpciePM_CTRL_NO_SOFT_RESET   = 0x0008,  // No reset for D3hot->D0
    nnpciePM_CTRL_PME_ENABLE      = 0x0100,  // PME pin enable
    nnpciePM_CTRL_DATA_SEL_MASK   = 0x1e00,  // Data select (??)
    nnpciePM_CTRL_DATA_SCALE_MASK = 0x6000,  // Data scale (??)
    nnpciePM_CTRL_PME_STATUS      = 0x8000,  // PME pin status
    nnpciePM_PPB_EXTENSIONS       = 6,       // PPB support extensions (??)
    nnpciePM_PPB_B2_B3            = 0x40,    // Stop clock when in D3hot (??)
    nnpciePM_BPCC_ENABLE          = 0x80,    // Bus power/clock control enable (??)
    nnpciePM_DATA_REGISTER        = 7,
    nnpciePM_SIZEOF               = 8,
};

enum nnpcieAGP
{
    nnpcieAGP_VERSION             = 2,            // BCD version number
    nnpcieAGP_RFU                 = 3,            // Rest of capability flags
    nnpcieAGP_STATUS              = 4,            // Status register
    nnpcieAGP_STATUS_RQ_MASK      = 0xff000000,   // Maximum number of requests - 1
    nnpcieAGP_STATUS_SBA          = 0x0200,       // Sideband addressing supported
    nnpcieAGP_STATUS_64BIT        = 0x0020,       // 64-bit addressing supported
    nnpcieAGP_STATUS_FW           = 0x0010,       // FW transfers supported
    nnpcieAGP_STATUS_RATE4        = 0x0004,       // 4x transfer rate supported
    nnpcieAGP_STATUS_RATE2        = 0x0002,       // 2x transfer rate supported
    nnpcieAGP_STATUS_RATE1        = 0x0001,       // 1x transfer rate supported
    nnpcieAGP_COMMAND             = 8,            // Control register
    nnpcieAGP_COMMAND_RQ_MASK     = 0xff000000,   // Master: Maximum number of requests
    nnpcieAGP_COMMAND_SBA         = 0x0200,       // Sideband addressing enabled
    nnpcieAGP_COMMAND_AGP         = 0x0100,       // Allow processing of AGP transactions
    nnpcieAGP_COMMAND_64BIT       = 0x0020,       // Allow processing of 64-bit addresses
    nnpcieAGP_COMMAND_FW          = 0x0010,       // Force FW transfers
    nnpcieAGP_COMMAND_RATE4       = 0x0004,       // Use 4x rate
    nnpcieAGP_COMMAND_RATE2       = 0x0002,       // Use 2x rate
    nnpcieAGP_COMMAND_RATE1       = 0x0001,       // Use 1x rate
    nnpcieAGP_SIZEOF              = 12,
};

enum nnpcieVitalProductData
{
    nnpcieVPD_ADDR                = 2,       // Address to access (15 bits!)
    nnpcieVPD_ADDR_MASK           = 0x7fff,  // Address mask
    nnpcieVPD_ADDR_F              = 0x8000,  // Write 0, 1 indicates completion
    nnpcieVPD_DATA                = 4,       // 32-bits of data returned here
    nnpcieCAP_VPD_SIZEOF          = 8,
};

enum nnpcieSlotIdentification
{
    nnpcieSID_ESR                 = 2,       // Expansion Slot Register
    nnpcieSID_ESR_NSLOTS          = 0x1f,    // Number of expansion slots available
    nnpcieSID_ESR_FIC             = 0x20,    // First In Chassis Flag
    nnpcieSID_CHASSIS_NR          = 3,       // Chassis Number
};

enum nnpcieMSI
{
    nnpcieMSI_FLAGS               = 2,       // Message Control
    nnpcieMSI_FLAGS_ENABLE        = 0x0001,  // MSI feature enabled
    nnpcieMSI_FLAGS_QMASK         = 0x000e,  // Maximum queue size available
    nnpcieMSI_FLAGS_QSIZE         = 0x0070,  // Message queue size configured
    nnpcieMSI_FLAGS_QSIZE_SHIFT   = 4,       // Message queue size bit position
    nnpcieMSI_FLAGS_64BIT         = 0x0080,  // 64-bit addresses allowed
    nnpcieMSI_FLAGS_MASKBIT       = 0x0100,  // Per-vector masking capable
    nnpcieMSI_RFU                 = 3,       // Rest of capability flags
    nnpcieMSI_ADDRESS_LO          = 4,       // Lower 32 bits
    nnpcieMSI_ADDRESS_HI          = 8,       // Upper 32 bits (if nnpcieMSI_FLAGS_64BIT set)
    nnpcieMSI_DATA_32             = 8,       // 16 bits of data for 32-bit devices
    nnpcieMSI_MASK_32             = 12,      // Mask bits register for 32-bit devices
    nnpcieMSI_PENDING_32          = 16,      // Pending intrs for 32-bit devices
    nnpcieMSI_DATA_64             = 12,      // 16 bits of data for 64-bit devices
    nnpcieMSI_MASK_64             = 16,      // Mask bits register for 64-bit devices
    nnpcieMSI_PENDING_64          = 20,      // Pending intrs for 64-bit devices
};

enum nnpcieMSIX
{
    nnpcieMSIX_FLAGS              = 2,            // Message Control
    nnpcieMSIX_FLAGS_QSIZE        = 0x07FF,       // Table size
    nnpcieMSIX_FLAGS_MASKALL      = 0x4000,       // Mask all vectors for this function
    nnpcieMSIX_FLAGS_ENABLE       = 0x8000,       // MSI-X enable
    nnpcieMSIX_TABLE              = 4,            // Table offset
    nnpcieMSIX_TABLE_BIR          = 0x00000007,   // BAR index
    nnpcieMSIX_TABLE_OFFSET       = 0xfffffff8,   // Offset into specified BAR
    nnpcieMSIX_PBA                = 8,            // Pending Bit Array offset
    nnpcieMSIX_PBA_BIR            = 0x00000007,   // BAR index
    nnpcieMSIX_PBA_OFFSET         = 0xfffffff8,   // Offset into specified BAR
    nnpcieCAP_MSIX_SIZEOF         = 12,           // size of MSIX registers
    nnpcieMSIX_ENTRY_SIZE         = 16,
    nnpcieMSIX_ENTRY_LOWER_ADDR   = 0,
    nnpcieMSIX_ENTRY_UPPER_ADDR   = 4,
    nnpcieMSIX_ENTRY_DATA         = 8,
    nnpcieMSIX_ENTRY_VECTOR_CTRL  = 12,
    nnpcieMSIX_ENTRY_CTRL_MASKBIT = 1,
};

enum nnpcieCompactPCIHotswap
{
    nnpcieCHSWP_CSR       = 2,       // Control and Status Register
    nnpcieCHSWP_DHA       = 0x01,    // Device Hiding Arm
    nnpcieCHSWP_EIM       = 0x02,    // ENUM# Signal Mask
    nnpcieCHSWP_PIE       = 0x04,    // Pending Insert or Extract
    nnpcieCHSWP_LOO       = 0x08,    // LED On / Off
    nnpcieCHSWP_PI        = 0x30,    // Programming Interface
    nnpcieCHSWP_EXT       = 0x40,    // ENUM# status - extraction
    nnpcieCHSWP_INS       = 0x80,    // ENUM# status - insertion
};

enum nnpcieAdvancedFeature
{
    nnpcieAF_LENGTH       = 2,
    nnpcieAF_CAP          = 3,
    nnpcieAF_CAP_TP       = 0x01,
    nnpcieAF_CAP_FLR      = 0x02,
    nnpcieAF_CTRL         = 4,
    nnpcieAF_CTRL_FLR     = 0x01,
    nnpcieAF_STATUS       = 5,
    nnpcieAF_STATUS_TP    = 0x01,
    nnpcieCAP_AF_SIZEOF   = 6,
};

// Type 0 (non-bridge) devices
enum nnpcieXType0Registers
{
    nnpcieX_CMD               = 2,                       // Modes & Features
    nnpcieX_CMD_DPERR_E       = 0x0001,                  // Data Parity Error Recovery Enable
    nnpcieX_CMD_ERO           = 0x0002,                  // Enable Relaxed Ordering
    nnpcieX_CMD_READ_512      = 0x0000,                  // 512 byte maximum read byte count
    nnpcieX_CMD_READ_1K       = 0x0004,                  // 1Kbyte maximum read byte count
    nnpcieX_CMD_READ_2K       = 0x0008,                  // 2Kbyte maximum read byte count
    nnpcieX_CMD_READ_4K       = 0x000c,                  // 4Kbyte maximum read byte count
    nnpcieX_CMD_MAX_READ      = 0x000c,                  // Max Memory Read Byte Count
    nnpcieX_CMD_SPLIT_1       = 0x0000,                  // Max 1
    nnpcieX_CMD_SPLIT_2       = 0x0010,                  // Max 2
    nnpcieX_CMD_SPLIT_3       = 0x0020,                  // Max 3
    nnpcieX_CMD_SPLIT_4       = 0x0030,                  // Max 4
    nnpcieX_CMD_SPLIT_8       = 0x0040,                  // Max 8
    nnpcieX_CMD_SPLIT_12      = 0x0050,                  // Max 12
    nnpcieX_CMD_SPLIT_16      = 0x0060,                  // Max 16
    nnpcieX_CMD_SPLIT_32      = 0x0070,                  // Max 32
    nnpcieX_CMD_MAX_SPLIT     = 0x0070,                  // Max Outstanding Split Transactions
    nnpcieX_STATUS            = 4,                       // PCI-X capabilities
    nnpcieX_STATUS_DEVFN      = 0x000000ff,              // A copy of devfn
    nnpcieX_STATUS_BUS        = 0x0000ff00,              // A copy of bus nr
    nnpcieX_STATUS_64BIT      = 0x00010000,              // 64-bit device
    nnpcieX_STATUS_133MHZ     = 0x00020000,              // 133 MHz capable
    nnpcieX_STATUS_SPL_DISC   = 0x00040000,              // Split Completion Discarded
    nnpcieX_STATUS_UNX_SPL    = 0x00080000,              // Unexpected Split Completion
    nnpcieX_STATUS_COMPLEX    = 0x00100000,              // Device Complexity
    nnpcieX_STATUS_MAX_READ   = 0x00600000,              // Designed Max Memory Read Count
    nnpcieX_STATUS_MAX_SPLIT  = 0x03800000,              // Designed Max Outstanding Split Transactions
    nnpcieX_STATUS_MAX_CUM    = 0x1c000000,              // Designed Max Cumulative Read Size
    nnpcieX_STATUS_SPL_ERR    = 0x20000000,              // Rcvd Split Completion Error Msg
    nnpcieX_STATUS_266MHZ     = 0x40000000,              // 266 MHz capable
    nnpcieX_STATUS_533MHZ     = 0x80000000,              // 533 MHz capable
    nnpcieX_ECC_CSR           = 8,                       // ECC control and status
    nnpcieCAP_PCIX_SIZEOF_V0  = 8,                       // size of registers for Version 0
    nnpcieCAP_PCIX_SIZEOF_V1  = 24,                      // size for Version 1
    nnpcieCAP_PCIX_SIZEOF_V2  = nnpcieCAP_PCIX_SIZEOF_V1,  // Same for v2
};
#define nnpcieX_CMD_VERSION(x)  (((x)>> 12)& 3)       // Version


// Type 0 (bridge) devices
enum nnpcieXType1Registers
{
    nnpcieX_BRIDGE_SSTATUS    = 2,       // Secondary Status
    nnpcieX_SSTATUS_64BIT     = 0x0001,  // Secondary AD interface is 64 bits
    nnpcieX_SSTATUS_133MHZ    = 0x0002,  // 133 MHz capable
    nnpcieX_SSTATUS_FREQ      = 0x03c0,  // Secondary Bus Mode and Frequency
    nnpcieX_SSTATUS_VERS      = 0x3000,  // PCI-X Capability Version
    nnpcieX_SSTATUS_V1        = 0x1000,  // Mode 2, not Mode 1
    nnpcieX_SSTATUS_V2        = 0x2000,  // Mode 1 or Modes 1 and 2
    nnpcieX_SSTATUS_266MHZ    = 0x4000,  // 266 MHz capable
    nnpcieX_SSTATUS_533MHZ    = 0x8000,  // 533 MHz capable
    nnpcieX_BRIDGE_STATUS     = 4,       // Bridge Status
};

enum nnpcieBridgeSubsystemIDRegisters
{
    nnpcieSSVID_VENDOR_ID     = 4,   // PCI Bridge subsystem vendor ID
    nnpcieSSVID_DEVICE_ID     = 6,   // PCI Bridge subsystem device ID
};

enum nnpcieExpressCapabilityRegisters
{
    nnpcieEXP_FLAGS                   = 2,            // Capabilities register
    nnpcieEXP_FLAGS_VERS              = 0x000f,       // Capability version
    nnpcieEXP_FLAGS_TYPE              = 0x00f0,       // Device/Port type
    nnpcieEXP_TYPE_ENDPOINT           = 0x0,          // Express Endpoint
    nnpcieEXP_TYPE_LEG_END            = 0x1,          // Legacy Endpoint
    nnpcieEXP_TYPE_ROOT_PORT          = 0x4,          // Root Port
    nnpcieEXP_TYPE_UPSTREAM           = 0x5,          // Upstream Port
    nnpcieEXP_TYPE_DOWNSTREAM         = 0x6,          // Downstream Port
    nnpcieEXP_TYPE_nnpcieBRIDGE         = 0x7,          // PCIe to PCI/PCI-X Bridge
    nnpcieEXP_TYPE_PCIE_BRIDGE        = 0x8,          // PCI/PCI-X to PCIe Bridge
    nnpcieEXP_TYPE_RC_END             = 0x9,          // Root Complex Integrated Endpoint
    nnpcieEXP_TYPE_RC_EC              = 0xa,          // Root Complex Event Collector
    nnpcieEXP_FLAGS_SLOT              = 0x0100,       // Slot implemented
    nnpcieEXP_FLAGS_IRQ               = 0x3e00,       // Interrupt message number
    nnpcieEXP_DEVCAP                  = 4,            // Device capabilities
    nnpcieEXP_DEVCAP_PAYLOAD          = 0x00000007,   // Max_Payload_Size
    nnpcieEXP_DEVCAP_PHANTOM          = 0x00000018,   // Phantom functions
    nnpcieEXP_DEVCAP_EXT_TAG          = 0x00000020,   // Extended tags
    nnpcieEXP_DEVCAP_L0S              = 0x000001c0,   // L0s Acceptable Latency
    nnpcieEXP_DEVCAP_L1               = 0x00000e00,   // L1 Acceptable Latency
    nnpcieEXP_DEVCAP_ATN_BUT          = 0x00001000,   // Attention Button Present
    nnpcieEXP_DEVCAP_ATN_IND          = 0x00002000,   // Attention Indicator Present
    nnpcieEXP_DEVCAP_PWR_IND          = 0x00004000,   // Power Indicator Present
    nnpcieEXP_DEVCAP_RBER             = 0x00008000,   // Role-Based Error Reporting
    nnpcieEXP_DEVCAP_PWR_VAL          = 0x03fc0000,   // Slot Power Limit Value
    nnpcieEXP_DEVCAP_PWR_SCL          = 0x0c000000,   // Slot Power Limit Scale
    nnpcieEXP_DEVCAP_FLR              = 0x10000000,   // Function Level Reset
    nnpcieEXP_DEVCTL                  = 8,            // Device Control
    nnpcieEXP_DEVCTL_CERE             = 0x0001,       // Correctable Error Reporting En.
    nnpcieEXP_DEVCTL_NFERE            = 0x0002,       // Non-Fatal Error Reporting Enable
    nnpcieEXP_DEVCTL_FERE             = 0x0004,       // Fatal Error Reporting Enable
    nnpcieEXP_DEVCTL_URRE             = 0x0008,       // Unsupported Request Reporting En.
    nnpcieEXP_DEVCTL_RELAX_EN         = 0x0010,       // Enable relaxed ordering
    nnpcieEXP_DEVCTL_PAYLOAD          = 0x00e0,       // Max_Payload_Size
    nnpcieEXP_DEVCTL_EXT_TAG          = 0x0100,       // Extended Tag Field Enable
    nnpcieEXP_DEVCTL_PHANTOM          = 0x0200,       // Phantom Functions Enable
    nnpcieEXP_DEVCTL_AUX_PME          = 0x0400,       // Auxiliary Power PM Enable
    nnpcieEXP_DEVCTL_NOSNOOP_EN       = 0x0800,       // Enable No Snoop
    nnpcieEXP_DEVCTL_READRQ           = 0x7000,       // Max_Read_Request_Size
    nnpcieEXP_DEVCTL_BCR_FLR          = 0x8000,       // Bridge Configuration Retry / FLR
    nnpcieEXP_DEVSTA                  = 10,           // Device Status
    nnpcieEXP_DEVSTA_CED              = 0x0001,       // Correctable Error Detected
    nnpcieEXP_DEVSTA_NFED             = 0x0002,       // Non-Fatal Error Detected
    nnpcieEXP_DEVSTA_FED              = 0x0004,       // Fatal Error Detected
    nnpcieEXP_DEVSTA_URD              = 0x0008,       // Unsupported Request Detected
    nnpcieEXP_DEVSTA_AUXPD            = 0x0010,       // AUX Power Detected
    nnpcieEXP_DEVSTA_TRPND            = 0x0020,       // Transactions Pending
    nnpcieEXP_LNKCAP                  = 12,           // Link Capabilities
    nnpcieEXP_LNKCAP_SLS              = 0x0000000f,   // Supported Link Speeds
    nnpcieEXP_LNKCAP_SLS_2_5GB        = 0x00000001,   // LNKCAP2 SLS Vector bit 0
    nnpcieEXP_LNKCAP_SLS_5_0GB        = 0x00000002,   // LNKCAP2 SLS Vector bit 1
    nnpcieEXP_LNKCAP_MLW              = 0x000003f0,   // Maximum Link Width
    nnpcieEXP_LNKCAP_ASPMS            = 0x00000c00,   // ASPM Support
    nnpcieEXP_LNKCAP_L0SEL            = 0x00007000,   // L0s Exit Latency
    nnpcieEXP_LNKCAP_L1EL             = 0x00038000,   // L1 Exit Latency
    nnpcieEXP_LNKCAP_CLKPM            = 0x00040000,   // Clock Power Management
    nnpcieEXP_LNKCAP_SDERC            = 0x00080000,   // Surprise Down Error Reporting Capable
    nnpcieEXP_LNKCAP_DLLLARC          = 0x00100000,   // Data Link Layer Link Active Reporting Capable
    nnpcieEXP_LNKCAP_LBNC             = 0x00200000,   // Link Bandwidth Notification Capability
    nnpcieEXP_LNKCAP_PN               = 0xff000000,   // Port Number
    nnpcieEXP_LNKCTL                  = 16,           // Link Control
    nnpcieEXP_LNKCTL_ASPMC            = 0x0003,       // ASPM Control
    nnpcieEXP_LNKCTL_ASPM_L0S         = 0x0001,       // L0s Enable
    nnpcieEXP_LNKCTL_ASPM_L1          = 0x0002,       // L1 Enable
    nnpcieEXP_LNKCTL_RCB              = 0x0008,       // Read Completion Boundary
    nnpcieEXP_LNKCTL_LD               = 0x0010,       // Link Disable
    nnpcieEXP_LNKCTL_RL               = 0x0020,       // Retrain Link
    nnpcieEXP_LNKCTL_CCC              = 0x0040,       // Common Clock Configuration
    nnpcieEXP_LNKCTL_ES               = 0x0080,       // Extended Synch
    nnpcieEXP_LNKCTL_CLKREQ_EN        = 0x0100,       // Enable clkreq
    nnpcieEXP_LNKCTL_HAWD             = 0x0200,       // Hardware Autonomous Width Disable
    nnpcieEXP_LNKCTL_LBMIE            = 0x0400,       // Link Bandwidth Management Interrupt Enable
    nnpcieEXP_LNKCTL_LABIE            = 0x0800,       // Link Autonomous Bandwidth Interrupt Enable
    nnpcieEXP_LNKSTA                  = 18,           // Link Status
    nnpcieEXP_LNKSTA_CLS              = 0x000f,       // Current Link Speed
    nnpcieEXP_LNKSTA_CLS_2_5GB        = 0x0001,       // Current Link Speed 2.5GT/s
    nnpcieEXP_LNKSTA_CLS_5_0GB        = 0x0002,       // Current Link Speed 5.0GT/s
    nnpcieEXP_LNKSTA_CLS_8_0GB        = 0x0003,       // Current Link Speed 8.0GT/s
    nnpcieEXP_LNKSTA_NLW              = 0x03f0,       // Negotiated Link Width
    nnpcieEXP_LNKSTA_NLW_X1           = 0x0010,       // Current Link Width x1
    nnpcieEXP_LNKSTA_NLW_X2           = 0x0020,       // Current Link Width x2
    nnpcieEXP_LNKSTA_NLW_X4           = 0x0040,       // Current Link Width x4
    nnpcieEXP_LNKSTA_NLW_X8           = 0x0080,       // Current Link Width x8
    nnpcieEXP_LNKSTA_NLW_SHIFT        = 4,            // start of NLW mask in link status
    nnpcieEXP_LNKSTA_LT               = 0x0800,       // Link Training
    nnpcieEXP_LNKSTA_SLC              = 0x1000,       // Slot Clock Configuration
    nnpcieEXP_LNKSTA_DLLLA            = 0x2000,       // Data Link Layer Link Active
    nnpcieEXP_LNKSTA_LBMS             = 0x4000,       // Link Bandwidth Management Status
    nnpcieEXP_LNKSTA_LABS             = 0x8000,       // Link Autonomous Bandwidth Status
    nnpcieCAP_EXP_ENDPOINT_SIZEOF_V1  = 20,           // v1 endpoints end here
    nnpcieEXP_SLTCAP                  = 20,           // Slot Capabilities
    nnpcieEXP_SLTCAP_ABP              = 0x00000001,   // Attention Button Present
    nnpcieEXP_SLTCAP_PCP              = 0x00000002,   // Power Controller Present
    nnpcieEXP_SLTCAP_MRLSP            = 0x00000004,   // MRL Sensor Present
    nnpcieEXP_SLTCAP_AIP              = 0x00000008,   // Attention Indicator Present
    nnpcieEXP_SLTCAP_PIP              = 0x00000010,   // Power Indicator Present
    nnpcieEXP_SLTCAP_HPS              = 0x00000020,   // Hot-Plug Surprise
    nnpcieEXP_SLTCAP_HPC              = 0x00000040,   // Hot-Plug Capable
    nnpcieEXP_SLTCAP_SPLV             = 0x00007f80,   // Slot Power Limit Value
    nnpcieEXP_SLTCAP_SPLS             = 0x00018000,   // Slot Power Limit Scale
    nnpcieEXP_SLTCAP_EIP              = 0x00020000,   // Electromechanical Interlock Present
    nnpcieEXP_SLTCAP_NCCS             = 0x00040000,   // No Command Completed Support
    nnpcieEXP_SLTCAP_PSN              = 0xfff80000,   // Physical Slot Number
    nnpcieEXP_SLTCTL                  = 24,           // Slot Control
    nnpcieEXP_SLTCTL_ABPE             = 0x0001,       // Attention Button Pressed Enable
    nnpcieEXP_SLTCTL_PFDE             = 0x0002,       // Power Fault Detected Enable
    nnpcieEXP_SLTCTL_MRLSCE           = 0x0004,       // MRL Sensor Changed Enable
    nnpcieEXP_SLTCTL_PDCE             = 0x0008,       // Presence Detect Changed Enable
    nnpcieEXP_SLTCTL_CCIE             = 0x0010,       // Command Completed Interrupt Enable
    nnpcieEXP_SLTCTL_HPIE             = 0x0020,       // Hot-Plug Interrupt Enable
    nnpcieEXP_SLTCTL_AIC              = 0x00c0,       // Attention Indicator Control
    nnpcieEXP_SLTCTL_ATTN_IND_ON      = 0x0040,       // Attention Indicator on
    nnpcieEXP_SLTCTL_ATTN_IND_BLINK   = 0x0080,       // Attention Indicator blinking
    nnpcieEXP_SLTCTL_ATTN_IND_OFF     = 0x00c0,       // Attention Indicator off
    nnpcieEXP_SLTCTL_PIC              = 0x0300,       // Power Indicator Control
    nnpcieEXP_SLTCTL_PWR_IND_ON       = 0x0100,       // Power Indicator on
    nnpcieEXP_SLTCTL_PWR_IND_BLINK    = 0x0200,       // Power Indicator blinking
    nnpcieEXP_SLTCTL_PWR_IND_OFF      = 0x0300,       // Power Indicator off
    nnpcieEXP_SLTCTL_PCC              = 0x0400,       // Power Controller Control
    nnpcieEXP_SLTCTL_PWR_ON           = 0x0000,       // Power On
    nnpcieEXP_SLTCTL_PWR_OFF          = 0x0400,       // Power Off
    nnpcieEXP_SLTCTL_EIC              = 0x0800,       // Electromechanical Interlock Control
    nnpcieEXP_SLTCTL_DLLSCE           = 0x1000,       // Data Link Layer State Changed Enable
    nnpcieEXP_SLTSTA                  = 26,           // Slot Status
    nnpcieEXP_SLTSTA_ABP              = 0x0001,       // Attention Button Pressed
    nnpcieEXP_SLTSTA_PFD              = 0x0002,       // Power Fault Detected
    nnpcieEXP_SLTSTA_MRLSC            = 0x0004,       // MRL Sensor Changed
    nnpcieEXP_SLTSTA_PDC              = 0x0008,       // Presence Detect Changed
    nnpcieEXP_SLTSTA_CC               = 0x0010,       // Command Completed
    nnpcieEXP_SLTSTA_MRLSS            = 0x0020,       // MRL Sensor State
    nnpcieEXP_SLTSTA_PDS              = 0x0040,        // Presence Detect State
    nnpcieEXP_SLTSTA_EIS              = 0x0080,       // Electromechanical Interlock Status
    nnpcieEXP_SLTSTA_DLLSC            = 0x0100,       // Data Link Layer State Changed
    nnpcieEXP_RTCTL                   = 28,           // Root Control
    nnpcieEXP_RTCTL_SECEE             = 0x0001,       // System Error on Correctable Error
    nnpcieEXP_RTCTL_SENFEE            = 0x0002,       // System Error on Non-Fatal Error
    nnpcieEXP_RTCTL_SEFEE             = 0x0004,       // System Error on Fatal Error
    nnpcieEXP_RTCTL_PMEIE             = 0x0008,       // PME Interrupt Enable
    nnpcieEXP_RTCTL_CRSSVE            = 0x0010,       // CRS Software Visibility Enable
    nnpcieEXP_RTCAP                   = 30,           // Root Capabilities
    nnpcieEXP_RTCAP_CRSVIS            = 0x0001,       // CRS Software Visibility capability
    nnpcieEXP_RTSTA                   = 32,           // Root Status
    nnpcieEXP_RTSTA_PME               = 0x00010000,   // PME status
    nnpcieEXP_RTSTA_PENDING           = 0x00020000,   // PME pending

    // The following are present only on "PCIe Capability version 2" devices
    nnpcieEXP_DEVCAP2                 = 36,           // Device Capabilities 2
    nnpcieEXP_DEVCAP2_ARI             = 0x00000020,   // Alternative Routing-ID
    nnpcieEXP_DEVCAP2_LTR             = 0x00000800,   // Latency tolerance reporting
    nnpcieEXP_DEVCAP2_OBFF_MASK       = 0x000c0000,   // OBFF support mechanism
    nnpcieEXP_DEVCAP2_OBFF_MSG        = 0x00040000,   // New message signaling
    nnpcieEXP_DEVCAP2_OBFF_WAKE       = 0x00080000,   // Re-use WAKE# for OBFF
    nnpcieEXP_DEVCTL2                 = 40,           // Device Control 2
    nnpcieEXP_DEVCTL2_COMP_TIMEOUT    = 0x000f,       // Completion Timeout Value
    nnpcieEXP_DEVCTL2_ARI             = 0x0020,       // Alternative Routing-ID
    nnpcieEXP_DEVCTL2_IDO_REQ_EN      = 0x0100,       // Allow IDO for requests
    nnpcieEXP_DEVCTL2_IDO_CMP_EN      = 0x0200,       // Allow IDO for completions
    nnpcieEXP_DEVCTL2_LTR_EN          = 0x0400,       // Enable LTR mechanism
    nnpcieEXP_DEVCTL2_OBFF_MSGA_EN    = 0x2000,       // Enable OBFF Message type A
    nnpcieEXP_DEVCTL2_OBFF_MSGB_EN    = 0x4000,       // Enable OBFF Message type B
    nnpcieEXP_DEVCTL2_OBFF_WAKE_EN    = 0x6000,       // OBFF using WAKE# signaling
    nnpcieEXP_DEVSTA2                 = 42,           // Device Status 2
    nnpcieCAP_EXP_ENDPOINT_SIZEOF_V2  = 44,           // v2 endpoints end here
    nnpcieEXP_LNKCAP2                 = 44,           // Link Capabilities 2
    nnpcieEXP_LNKCAP2_SLS_2_5GB       = 0x00000002,   // Supported Speed 2.5GT/s
    nnpcieEXP_LNKCAP2_SLS_5_0GB       = 0x00000004,   // Supported Speed 5.0GT/s
    nnpcieEXP_LNKCAP2_SLS_8_0GB       = 0x00000008,   // Supported Speed 8.0GT/s
    nnpcieEXP_LNKCAP2_CROSSLINK       = 0x00000100,   // Crosslink supported
    nnpcieEXP_LNKCTL2                 = 48,           // Link Control 2
    nnpcieEXP_LNKSTA2                 = 50,           // Link Status 2
    nnpcieEXP_SLTCAP2                 = 52,           // Slot Capabilities 2
    nnpcieEXP_SLTCTL2                 = 56,           // Slot Control 2
    nnpcieEXP_SLTSTA2                 = 58,           // Slot Status 2
};

// Extended capability lists (PCI-X 2.0 and beyond)
enum nnpcieExtendedCapabilityLists
{
    nnpcieEXT_CAP_ID_ERR              = 0x01,         // Advanced Error Reporting
    nnpcieEXT_CAP_ID_VC               = 0x02,         // Virtual Channel Capability
    nnpcieEXT_CAP_ID_DSN              = 0x03,         // Device Serial Number
    nnpcieEXT_CAP_ID_PWR              = 0x04,         // Power Budgeting
    nnpcieEXT_CAP_ID_RCLD             = 0x05,         // Root Complex Link Declaration
    nnpcieEXT_CAP_ID_RCILC            = 0x06,         // Root Complex Internal Link Control
    nnpcieEXT_CAP_ID_RCEC             = 0x07,         // Root Complex Event Collector
    nnpcieEXT_CAP_ID_MFVC             = 0x08,         // Multi-Function VC Capability
    nnpcieEXT_CAP_ID_VC9              = 0x09,         // same as _VC
    nnpcieEXT_CAP_ID_RCRB             = 0x0A,         // Root Complex RB?
    nnpcieEXT_CAP_ID_VNDR             = 0x0B,         // Vendor-Specific
    nnpcieEXT_CAP_ID_CAC              = 0x0C,         // Config Access - obsolete
    nnpcieEXT_CAP_ID_ACS              = 0x0D,         // Access Control Services
    nnpcieEXT_CAP_ID_ARI              = 0x0E,         // Alternate Routing ID
    nnpcieEXT_CAP_ID_ATS              = 0x0F,         // Address Translation Services
    nnpcieEXT_CAP_ID_SRIOV            = 0x10,         // Single Root I/O Virtualization
    nnpcieEXT_CAP_ID_MRIOV            = 0x11,         // Multi Root I/O Virtualization
    nnpcieEXT_CAP_ID_MCAST            = 0x12,         // Multicast
    nnpcieEXT_CAP_ID_PRI              = 0x13,         // Page Request Interface
    nnpcieEXT_CAP_ID_AMD_XXX          = 0x14,         // Reserved for AMD
    nnpcieEXT_CAP_ID_REBAR            = 0x15,         // Resizable BAR
    nnpcieEXT_CAP_ID_DPA              = 0x16,         // Dynamic Power Allocation
    nnpcieEXT_CAP_ID_TPH              = 0x17,         // TPH Requester
    nnpcieEXT_CAP_ID_LTR              = 0x18,         // Latency Tolerance Reporting
    nnpcieEXT_CAP_ID_SECPCI           = 0x19,         // Secondary PCIe Capability
    nnpcieEXT_CAP_ID_PMUX             = 0x1A,         // Protocol Multiplexing
    nnpcieEXT_CAP_ID_PASID            = 0x1B,         // Process Address Space ID
    nnpcieEXT_CAP_ID_MAX              = nnpcieEXT_CAP_ID_PASID,
    nnpcieEXT_CAP_DSN_SIZEOF          = 12,
    nnpcieEXT_CAP_MCAST_ENDPOINT_SIZEOF = 40,
};
#define nnpcieEXT_CAP_ID(header)      (header& 0x0000ffff)
#define nnpcieEXT_CAP_VER(header)     ((header >> 16)& 0xf)
#define nnpcieEXT_CAP_NEXT(header)    ((header >> 20)& 0xffc)


enum nnpcieAdvancedErrorReporting
{
    nnpcieERR_UNCOR_STATUS            = 4,            // Uncorrectable Error Status
    nnpcieERR_UNC_UND                 = 0x00000001,   // Undefined
    nnpcieERR_UNC_DLP                 = 0x00000010,   // Data Link Protocol
    nnpcieERR_UNC_SURPDN              = 0x00000020,   // Surprise Down
    nnpcieERR_UNC_POISON_TLP          = 0x00001000,   // Poisoned TLP
    nnpcieERR_UNC_FCP                 = 0x00002000,   // Flow Control Protocol
    nnpcieERR_UNC_COMP_TIME           = 0x00004000,   // Completion Timeout
    nnpcieERR_UNC_COMP_ABORT          = 0x00008000,   // Completer Abort
    nnpcieERR_UNC_UNX_COMP            = 0x00010000,   // Unexpected Completion
    nnpcieERR_UNC_RX_OVER             = 0x00020000,   // Receiver Overflow
    nnpcieERR_UNC_MALF_TLP            = 0x00040000,   // Malformed TLP
    nnpcieERR_UNC_ECRC                = 0x00080000,   // ECRC Error Status
    nnpcieERR_UNC_UNSUP               = 0x00100000,   // Unsupported Request
    nnpcieERR_UNC_ACSV                = 0x00200000,   // ACS Violation
    nnpcieERR_UNC_INTN                = 0x00400000,   // internal error
    nnpcieERR_UNC_MCBTLP              = 0x00800000,   // MC blocked TLP
    nnpcieERR_UNC_ATOMEG              = 0x01000000,   // Atomic egress blocked
    nnpcieERR_UNC_TLPPRE              = 0x02000000,   // TLP prefix blocked
    nnpcieERR_UNCOR_MASK              = 8,            // Uncorrectable Error Mask
    nnpcieERR_UNCOR_SEVER             = 12,           // Uncorrectable Error Severity
    nnpcieERR_COR_STATUS              = 16,           // Correctable Error Status
    nnpcieERR_COR_RCVR                = 0x00000001,   // Receiver Error Status
    nnpcieERR_COR_BAD_TLP             = 0x00000040,   // Bad TLP Status
    nnpcieERR_COR_BAD_DLLP            = 0x00000080,   // Bad DLLP Status
    nnpcieERR_COR_REP_ROLL            = 0x00000100,   // REPLAY_NUM Rollover
    nnpcieERR_COR_REP_TIMER           = 0x00001000,   // Replay Timer Timeout
    nnpcieERR_COR_ADV_NFAT            = 0x00002000,   // Advisory Non-Fatal
    nnpcieERR_COR_INTERNAL            = 0x00004000,   // Corrected Internal
    nnpcieERR_COR_LOG_OVER            = 0x00008000,   // Header Log Overflow
    nnpcieERR_COR_MASK                = 20,           // Correctable Error Mask
    nnpcieERR_CAP                     = 24,           // Advanced Error Capabilities
    nnpcieERR_CAP_ECRC_GENC           = 0x00000020,   // ECRC Generation Capable
    nnpcieERR_CAP_ECRC_GENE           = 0x00000040,   // ECRC Generation Enable
    nnpcieERR_CAP_ECRC_CHKC           = 0x00000080,   // ECRC Check Capable
    nnpcieERR_CAP_ECRC_CHKE           = 0x00000100,   // ECRC Check Enable
    nnpcieERR_HEADER_LOG              = 28,           // Header Log Register (16 bytes)
    nnpcieERR_ROOT_COMMAND            = 44,           // Root Error Command
    nnpcieERR_ROOT_CMD_COR_EN         = 0x00000001,
    nnpcieERR_ROOT_CMD_NONFATAL_EN    = 0x00000002,
    nnpcieERR_ROOT_CMD_FATAL_EN       = 0x00000004,
    nnpcieERR_ROOT_STATUS             = 48,
    nnpcieERR_ROOT_COR_RCV            = 0x00000001,   // ERR_COR Received
    nnpcieERR_ROOT_MULTI_COR_RCV      = 0x00000002,
    nnpcieERR_ROOT_UNCOR_RCV          = 0x00000004,
    nnpcieERR_ROOT_MULTI_UNCOR_RCV    = 0x00000008,
    nnpcieERR_ROOT_FIRST_FATAL        = 0x00000010,   // First Fatal
    nnpcieERR_ROOT_NONFATAL_RCV       = 0x00000020,   // Non-Fatal Received
    nnpcieERR_ROOT_FATAL_RCV          = 0x00000040,   // Fatal Received
    nnpcieERR_ROOT_ERR_SRC            = 52,           // Error Source Identification
};
#define    nnpcieERR_CAP_FEP(x)((x)& 31)  // First Error Pointer



