﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバのパッドアクセス機能の API 宣言(C リンケージ向けインタフェース)
 */

#pragma once

#include <nnc/nn_Common.h>
#include <nnc/os/os_SystemEvent.h>
#include <nnc/nn_Result.h>

#include <nn/gpio/detail/gpio_Gen.h>
#include <nnc/gpio/gpio_Type.h>

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
#include <nnc/nn_DeviceCode.h>
#endif

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
#include <nnc/gpio/gpio_PadMap.h>
#endif

//--------------------------------------------------------------------------
//  C/C++ 共通の宣言・定義
//--------------------------------------------------------------------------

#ifdef __cplusplus
extern "C"
{
#endif // ifdef __cplusplus

/**
* @brief GPIO パッドのセッション用パラメータです。
*
* @see
*  nn::gpio::GpioPadSession
*/
typedef struct nngpioGpioPadSession
{
    void* _sessionHandle;
    nnosSystemEventType* _pEvent;
} nngpioGpioPadSession;


//! @name 初期化と終了処理
//! @{

/**
 * @brief 指定した GPIO パッドとのセッションをオープンします。すなわちスレーブデバイスとの通信を可能な状態にします。
 *
 * @param[in]   pOutSession           GPIO パッドのセッション用パラメータ
 * @param[in]   pad                   アクセス対象の GPIO パッドの指定子
 * @details
 *  この関数は、@ref nn::gpio::OpenSession() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::gpio::OpenSession()
 */

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
// [Gen2] XXX: C はオーバーロードできないので API 名は仮
nnResult nngpioOpenSession2(nngpioGpioPadSession* pOutSession, nnDeviceCode pad);
#endif
#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
void nngpioOpenSession(nngpioGpioPadSession* pOutSession, nngpioGpioPadName pad);
#endif

/**
 * @brief 指定した GPIO パッドとのセッションをクローズします。
 *
 * @param[in]   pSession       GPIO パッドのセッション用パラメータ
 * @details
 *  この関数は、@ref nn::gpio::CloseSession() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::gpio::CloseSession()
 */
void nngpioCloseSession(nngpioGpioPadSession* pSession);
//! @}
// ~初期化と終了処理

//! @name GPIO パッドに対する設定
//! @{
/**
 * @brief       GPIO の入出力方向を設定します。
 *
 * @param[in]   pSession         GPIO パッドのセッション用パラメータ
 * @param[in]   direction        pSession で指定された GPIO パッドの入出力方向を設定します。
 * @details
 *  この関数は、@ref nn::gpio::SetDirection() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::SetDirection()
 */
void nngpioSetDirection(nngpioGpioPadSession* pSession, nngpioDirection direction);

/**
 * @brief       GPIO パッドの入出力方向を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの入出力方向を返します。
 * @details
 *  この関数は、@ref nn::gpio::GetDirection() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::GetDirection()
 */
nngpioDirection nngpioGetDirection(nngpioGpioPadSession* pSession);

/**
 * @brief       GPIO パッドの割り込みモードを設定します。
 *
 * @param[in]   pSession         GPIO パッドのセッション用パラメータ
 * @param[in]   mode             pSession で指定された GPIO パッドの割り込みモードを設定します。
 * @details
 *  この関数は、@ref nn::gpio::SetInterruptMode() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::SetInterruptMode()
 */
void nngpioSetInterruptMode(nngpioGpioPadSession* pSession, nngpioInterruptMode mode);

/**
 * @brief       GPIO パッドの割り込みモードを取得します。
 *　
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの割り込みモードを返します。
 * @details
 *  この関数は、@ref nn::gpio::GetInterruptMode() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::GetInterruptMode()
 */
nngpioInterruptMode nngpioGetInterruptMode(nngpioGpioPadSession* pSession);

/**
 * @brief       GPIO パッドの割り込み許可状態を設定します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   enable      gpioPad で指定された GPIO パッドの割り込み許可状態を設定します。
 * @details
 *  この関数は、@ref nn::gpio::SetInterruptEnable() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::SetInterruptEnable()
 */
void nngpioSetInterruptEnable(nngpioGpioPadSession* pSession, bool enable);

/**
 * @brief       GPIO パッドの割り込み許可状態を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの割り込み許可状態を返します。
 * @retval      false    禁止
 * @retval      true     許可
 * @details
 *  この関数は、@ref nn::gpio::GetInterruptEnable() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::GetInterruptEnable()
 */
bool nngpioGetInterruptEnable(nngpioGpioPadSession* pSession);


/**
 * @brief       GPIO パッドの割り込み検出ステータスを取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの割り込み検出ステータスを返します。
 * @details
 *  この関数は、@ref nn::gpio::GetInterruptStatus() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::GetInterruptStatus()
 */
nngpioInterruptStatus nngpioGetInterruptStatus(nngpioGpioPadSession* pSession);


/**
 * @brief       GPIO パッドの割り込み検出ステータスをクリアします。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details
 *  この関数は、@ref nn::gpio::ClearInterruptStatus() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::ClearInterruptStatus()
 */
void nngpioClearInterruptStatus(nngpioGpioPadSession* pSession);

/**
 * @brief チャタリング防止機能の設定状態を切り替えます。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   isEnable          チャタリング防止機能の使用の有無
 *
 * @details
 *  この関数は、@ref nn::gpio::SetDebounceEnabled() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::SetDebounceEnabled()
 */
void nngpioSetDebounceEnabled(nngpioGpioPadSession* pSession, bool isEnable);

/**
 * @brief チャタリング防止機能の設定状態を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      チャタリング防止状態を返します。
 * @retval      true        チャタリング防止機能が設定されています。
 * @retval      false       チャタリング防止機能が設定されていません。
 *
 * @details
 *  この関数は、@ref nn::gpio::GetDebounceEnabled() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::GetDebounceEnabled()
 */
bool nngpioGetDebounceEnabled(nngpioGpioPadSession* pSession);

/**
 * @brief チャタリングとして判定する時間を設定します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   msecTime          チャタリングとして判定する時間 (mSec)
 *
 * @details
 *  この関数は、@ref nn::gpio::SetDebounceTime() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::SetDebounceTime()
 */
void nngpioSetDebounceTime(nngpioGpioPadSession* pSession, int msecTime);

/**
 * @brief チャタリングとして判定する時間を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      チャタリングとして判定する時間を返します。単位は mSec です。
 *
 * @details
 *  この関数は、@ref nn::gpio::GetDebounceTime() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::GetDebounceTime()
 */
int nngpioGetDebounceTime(nngpioGpioPadSession* pSession);
//! @}
// ~GPIO パッドに対する設定


//! @name GPIO パッドのコントロール
//! @{
/**
* @brief GPIO パッドの状態(H/L)を確認します。
*
* @param[in]   pSession          GPIO パッドのセッション用パラメータ
* @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
*              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
* @return      pSession で指定した GPIO パッドの状態を返します。
* @details
*  この関数は、@ref nn::gpio::GetValue() の C リンケージ向けインタフェースです。
*  詳細は C++ 向け版を参照してください。
* @see
*  nn::gpio::GetValue()
*/
nngpioGpioValue nngpioGetValue(nngpioGpioPadSession* pSession);

/**
* @brief GPIO パッドの状態(H/L)を設定します。
*
* @param[in]   pSession         GPIO パッドのセッション用パラメータ
* @param[in]   value            pSession で指定した GPIO パッドの状態を設定します。
* @details
*  この関数は、@ref nn::gpio::SetValue() の C リンケージ向けインタフェースです。
*  詳細は C++ 向け版を参照してください。
* @see
*  nn::gpio::SetValue()
*/
void nngpioSetValue(nngpioGpioPadSession* pSession, nngpioGpioValue value);
//! @}
// ~GPIO パッドのコントロール


//! @name GPIO パッドの割り込みとイベントの紐付けと紐付解除
//! @{
/**
 * @brief        GPIO パッドの割り込みとイベントを紐付けます。
 *
 * @param[in]    pEvent           割込みが発生した際にシグナル化するイベントを指定します。
 * @param[in]    pSession          GPIO パッドのセッション用パラメータ
 * @return       処理の結果を返します。
 * @retval       ResultSuccess        成功しました。
 * @retval       nngpioResultAlreadyBound   pSession で指定した GPIO パッドがすでに特定イベントと紐付けされています。
 * @details
 *  この関数は、@ref nn::gpio::BindInterrupt() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::BindInterrupt()
 */
nnResult nngpioBindInterrupt(nnosSystemEventType* pEvent, nngpioGpioPadSession* pSession);


/**
 * @brief        GPIO パッドの割り込みとイベントの紐付を解除します。
 *
 * @param[in]    pSession          GPIO パッドのセッション用パラメータ
 *
 * @details
 *  この関数は、@ref nn::gpio::UnbindInterrupt() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 * @see
 *  nn::gpio::UnbindInterrupt()
 */
void nngpioUnbindInterrupt(nngpioGpioPadSession* pSession);
//! @}
// ~割り込みのバインドとアンバインド

#ifdef __cplusplus
}
#endif // ifdef __cplusplus
