﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   振動に用いる帯域別 AMFM 方式のデコーダとエンコーダ
 */

#pragma once

#include <cstdint>
#include <nn/nn_Macro.h>

namespace nn { namespace xcd {

/**
 * @brief   帯域別 AMFM 方式のフォーマットを表す列挙型です。
 */
enum VibrationAmFmFormat
{
    VibrationAmFmFormat_5BitVer1,   //!< 5Bit AMFM 符号テーブル ver.1.0 (エンコード時には利用しません)
    VibrationAmFmFormat_5BitVer2,   //!< 5Bit AMFM 符号テーブル ver.2.0
};

/**
 * @brief   AMFM デコーダのコンテクストを表す構造体です。
 */
struct VibrationAmFmDecoderContext
{
    const void* _pConfig;
    uint32_t _logAmp;
    uint32_t _logFreq;
    uint32_t _baseFreq;
};

/**
 * @brief   AMFM エンコーダのコンテクストを表す構造体です。
 */
struct VibrationAmFmEncoderContext
{
    VibrationAmFmDecoderContext _decoderContext;
    uint16_t _relAmpCountMax;
    uint16_t _relFreqCountMax;
    uint16_t _relAmpCount;
    uint16_t _relFreqCount;
};

/**
 * @brief       AMFM デコーダのコンテクストをを初期化します。
 * @param[out]  pOutContext     VibrationAmFmDecoderContext 構造体へのポインタ
 * @param[in]   format          AMFM 符号化方式のフォーマット
 * @param[in]   baseFrequency   ベース周波数 (単位は Hz)
 */
void ResetVibrationAmFmDecoderContext(
    VibrationAmFmDecoderContext* pOutContext, VibrationAmFmFormat format, float baseFrequency) NN_NOEXCEPT;

/**
 * @brief       AMFM 符号をデコードして AMFM デコーダのコンテクストを更新します。
 * @param[in]       amFmCode    AMFM 符号
 * @param[in, out]  pContext    VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
void DecodeVibrationAmFmCode(int amFmCode, VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       7bit AM 符号をデコードして AMFM デコーダのコンテクストを更新します。
 * @param[in]       amCode      7bit AM 符号
 * @param[in, out]  pContext    VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
void DecodeVibrationAm7bitCode(int amCode, VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       7bit FM 符号をデコードして AMFM デコーダのコンテクストを更新します。
 * @param[in]       fmCode      7bit FM 符号
 * @param[in, out]  pContext    VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
void DecodeVibrationFm7bitCode(int fmCode, VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       AMFM デコーダのコンテクストから振幅値を取得します。
 * @return      振幅値を返します。(単位は Hz)
 * @param[in]   pContext        VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
float GetVibrationAmplitudeFromAmFmDecoder(const VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       AMFM デコーダのコンテクストから周波数値を取得します。
 * @return      周波数値を返します。
 * @param[in]   pContext        VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
float GetVibrationFrequencyFromAmFmDecoder(const VibrationAmFmDecoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       AMFM エンコーダのコンテクストをを初期化します。
 * @param[out]  pOutContext     VibrationAmFmEncoderContext 構造体へのポインタ
 * @param[in]   baseFrequency   ベース周波数 (単位は Hz)
 */
void ResetVibrationAmFmEncoderContext(VibrationAmFmEncoderContext* pOutContext, float baseFrequency) NN_NOEXCEPT;

/**
 * @brief       振幅値と周波数値をエンコードしてコンテクストを更新し、AMFM 符号を取得します。
 * @return      AMFM 符号を返します。
 * @param[in]       amplitude   振幅値
 * @param[in]       frequency   周波数値 (単位は Hz)
 * @param[in, out]  pContext    VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
int EncodeVibrationToAmFmCode(float amplitude, float frequency, VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       振幅値をエンコードしてコンテクストを更新し、7bit AM 符号を取得します。
 * @return      7bit AM 符号を返します。
 * @param[in]       amplitude   振幅値
 * @param[in, out]  pContext    VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
int EncodeVibrationToAm7bitCode(float amplitude, VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT;

/**
 * @brief       周波数値をエンコードしてコンテクストを更新し、7bit FM 符号を取得します。
 * @return      7bit FM 符号を返します。
 * @param[in]       frequency   周波数値 (単位は Hz)
 * @param[in, out]  pContext    VibrationAmFmDecoderContext 構造体へのポインタ
 * @pre         - pContext が初期化済み
 */
int EncodeVibrationToFm7bitCode(float frequency, VibrationAmFmEncoderContext* pContext) NN_NOEXCEPT;

}} // namespace nn::xcd
