﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイスの列挙などに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/hid/detail/hid_ColorTypes.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace xcd {

/**
 * @brief xcdの中で扱われるデバイス毎のハンドルです
 */

struct DeviceHandle final
{
    uint32_t _upper;      //!< 接続単位でユニークな値の領域
    uint32_t _lower;      //!< DeviceHandler を特定するために用いる領域

    friend bool operator== (const DeviceHandle& lhs,
                            const DeviceHandle& rhs) NN_NOEXCEPT
    {
        return (lhs._upper == rhs._upper);
    }

    friend bool operator!= (const DeviceHandle& lhs,
                            const DeviceHandle& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**------------------------------------------------------------------
     * @brief  常に無効な DeviceHandle を返します。
     */
    static DeviceHandle GetInvalidHandle() NN_NOEXCEPT
    {
        DeviceHandle handle = { 0, 0 };
        return handle;
    }

    /**------------------------------------------------------------------
     * @brief  DeviceHandle が有効かどうかを検証します
     */
    bool IsValid() const NN_NOEXCEPT
    {
        return (*this != GetInvalidHandle());
    }
};

const int DeviceCountMax = 16;   //!<   xcd内で扱えるデバイスの最大数です。

/**
 * @brief       デバイスの接続状態を表す構造体です。
 */
struct DeviceList
{
    int32_t deviceCount;                     //!< 接続されているデバイスの数です。
    DeviceHandle handleList[DeviceCountMax]; //!< 接続されているデバイスのハンドルのリストです
};

/**
 * @brief       デバイスの種類です。
 */
enum DeviceType : uint8_t
{
    DeviceType_Unknown = 0,      //!< 動作対象外のデバイスです
    DeviceType_Right,            //!< Joy-Con (R)
    DeviceType_Left,             //!< Joy-Con (L)
    DeviceType_FullKey,          //!< Switch Proコントローラー
    DeviceType_MiyabiLeft,       //!< MiyabiLeft
    DeviceType_MiyabiRight,      //!< MiyabiRight
    DeviceType_Tarragon,         //!< Tarragon
};

/**
 * @brief       デバイスの種類を補足するサブタイプです
 */
typedef uint8_t DeviceSubType;

/**
 * @brief       デバイスの情報を定義する構造体です。
 */
struct DeviceInfo
{
    DeviceType deviceType;                  //!< デバイスの種類です。
    ::nn::bluetooth::Address address;       //!< デバイスの識別子です。
};

const size_t IdentificationCodeLength = 16;

/**
 * @brief       デバイスの識別情報
 */
struct IdentificationCode
{
    char _code[IdentificationCodeLength];
};

/**
  * brief      色情報を定義した構造体
  */
typedef nn::hid::detail::DeviceColor DeviceColor;

/**
 * @brief       リンクが更新された際にシグナルされるイベントをバインドします
 *
 * @param[in]   pSystemEvent                  バインドするシステムイベント
 *
 * @pre         - pSystemEvent != nullptr
 */
void BindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       リンクが更新された際にシグナルされるイベントをアンバインドします
 *
 * @param[in]   pSystemEvent                  アンバインドするシステムイベント
 *
 * @pre         - pSystemEvent != nullptr
 */
void UnbindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;


/**
 * @brief       接続されているデバイスを列挙します
 *
 * @details     接続されているデバイスを列挙します。
 *              デバイスは本体に直接物理的に接続されている単位で列挙されます。
 *              あるデバイスの先に拡張機器が接続されている場合は、そのデバイスのいち入力情報として扱われます。
 *
 * @param[out]  pOutValue                    接続されているデバイスのリスト
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess                処理に成功しました。
 * @retval      ResultNotFound      　       デバイスは1台も接続されていません
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pOutValue != nullptr
 */
Result ListDevices(DeviceList* pOutValue) NN_NOEXCEPT;

/**
 * @brief      無線接続されているデバイスを切断します。
 *
 * @details    無線接続されているデバイスを切断します。
 *             切断処理は非同期で行われ、切断処理が完了するとInitialize()で設定したnn::os::SystemEventTypeで通知を受け取ることができます。
 *             有線接続されているデバイスは切断することができません。
 *
 * @param[in]  handle                        切断するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                  処理に成功しました。
 * @retval     ResultNotConnected             デバイスが接続されていません。
 * @retval     ResultNotWireless              指定のデバイスは無線接続されていないため切断できません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result Detach(DeviceHandle handle) NN_NOEXCEPT;

/**
* @brief      デバイスを Reset します。
*
* @details    デバイスを Reset したうえで再接続します。
*             Reboot 後再接続するか、しないかを選択できます
*
* @param[in]  handle                        切断するデバイスへのハンドル
* @param[in]  reconnect                     再接続するかどうか
*
* @return     実行結果を返します。
* @retval     ResultSuccess                  処理に成功しました。
* @retval     ResultNotConnected             デバイスが接続されていません。
*
* @pre        - xcd ライブラリは初期化済である必要があります。
*/
Result Reboot(DeviceHandle handle, bool reconnect) NN_NOEXCEPT;

/**
 * @brief      デバイスを Reset します。
 *
 * @details    デバイスを Reset したうえで再接続します。
 *             Reboot 後再接続するか、しないかを選択できます
 *
 * @param[in]  handle                        切断するデバイスへのハンドル
 * @param[in]  reconnect                     再接続するかどうか
 * @param[in]  isFlash                       即座に送信するかどうか
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                  処理に成功しました。
 * @retval     ResultNotConnected             デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result Reboot(DeviceHandle handle, bool reconnect, bool isFlash) NN_NOEXCEPT;

/**
 * @brief      TeraMcu を起動、停止します。
 *
 * @details    TeraMcu を起動、停止します。
 *             コマンドの完了は待ちません。
 *
 * @param[in]  handle                         起動、停止する TeraMcu が載ったデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                  処理に成功しました。
 * @retval     ResultNotConnected             デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result McuResume(DeviceHandle handle, bool isResume) NN_NOEXCEPT;

/**
 * @brief      接続されているデバイスの情報を取得します。
 *
 * @details    デバイスの種類など接続されているデバイスの情報を取得します。
 *             取得されるデバイスの情報は静的なもので変化することはありません、
 *             電池残量のように動的に変化するコントローラの状態についてはGetDeviceStatus()を使用してください。
 *
 * @param[in]  handle                        情報を取得するデバイスへのハンドル
 * @param[out] pOutValue                     取得したデバイスの情報
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 * @pre         - xcd ライブラリは初期化済である必要があります。
 *              - pOutValue!= nullptr
 */
Result GetDeviceInfo(DeviceInfo* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      任意のデバイスで新たなコントローラデータを受信したときに通知を受けるイベントを設定します。
 *
 * @details    任意のデバイスが新たなコントローラデータを受信したときに通知を受けるためのイベントを設定します。
 *
 * @param[in]  handle                      デバイスへのハンドル
 * @param[in]  pSamplingEvent              新たなサンプルを受信したときに通知をうけるイベント
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess               処理に成功しました。
 * @retval     ResultNotConnected          デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pSamplingEvent!= nullptr
 *             - *pSamplingEventは未初期化状態
 */
Result SetSamplingEvent(nn::os::SystemEventType* pSamplingEvent, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      任意のデバイスの IdetificationCode を読み出します
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 */
Result GetIdentificationCode(IdentificationCode* pOutCode, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      任意のデバイスの色情報を読み出します
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *
 * @retval     ResultSuccess                 処理に成功しました。
 * @retval     ResultNotConnected            デバイスが接続されていません。
 *
 */
Result GetDeviceColor(DeviceColor* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      任意のデバイスの色情報を書き換えます
 *
 */
Result UpdateDeviceColor(const ::nn::util::Color4u8Type& mainColor, const ::nn::util::Color4u8Type& subColor, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      任意のデバイスのデザイン情報を書き換えます
 *
 * @details    バージョン1以上のコントローラーで有効です。4色の色とデザイン情報をセットできます。
 *
 */
Result UpdateDesignInfo(const DeviceColor& color, uint8_t variation, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       有線デバイス登録を有効にします
 *
 */
void EnableWiredDeviceRegistration() NN_NOEXCEPT;

/**
 * @brief       有線デバイス登録を無効にします
 *
 */
void DisableWiredDeviceRegistration() NN_NOEXCEPT;

/**
 * @brief      パケットの受信状況を取得します
 */
nn::hid::detail::RxPacketHistory GetRxPacketHistory(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      充電状態をリセットします。
 *
 */
Result ResetBatteryChargeTimer(DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
