﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリの Attachment Device に関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Firmware.h>
#include <nn/xcd/xcd_TeraFirmware.h>

namespace nn { namespace xcd {

const size_t AttachmentSendCommandSizeMax    = 37;   //!<   SendCommandToAttachmentDevice() で送信できるコマンドサイズの最大値です。
const size_t AttachmentReceiveCommandSizeMax = 32;   //!<   GetReceivedCommandFromAttachmentDevice() で受信できるコマンドサイズの最大値です。
// TORIAEZU
const size_t AttachmentPollingReceiveCommandSizeMax = 38;   //!<   GetPollingDataForAttachmentDevice() で受信できるコマンドサイズの最大値です。

/**
 * @brief       拡張デバイス機能を使用するために必要な Joy-Con のバージョンの最低値
 */
struct AttachmentFunctionRequiredVersion
{
    BtFirmwareVersion btVersion;
    McuVersionData    mcuVersion;
};

/**
* @brief       拡張デバイスの情報を表す構造体です。
*/
struct ExtDevInfo
{
    uint8_t extDevType;
};

/**
* @brief       拡張デバイスのポーリングモードの種類です。
*/
enum AttachmentPollingMode
{
    AttachmentPollingMode_SixAxisSendorDisable, //!< Joy-Con の六軸センサを使用しないモードです。
    AttachmentPollingMode_SixAxisSendorEnable,  //!< Joy-Con の六軸センサと同時に使うモードです。
};

/**
* @brief       拡張デバイスが接続されているかどうかを取得します。
*
* @details     拡張デバイスが接続されているかどうかを取得します。
*              本機能では何らかの拡張デバイスが接続されていることが取得できます。
*              接続されている拡張デバイスの種類を取得したい場合は AttachmentEnable() で拡張デバイスを有効にした後に、
*              GetExtDevInfo() を呼んでデバイスの種類を取得してください。
*
* @param[in]   pAttachmentDataReceiveEvent 拡張デバイスからのデータ受信を通知するイベントへのポインタ
* @param[in]   handle                      コマンドを送信する拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,            処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*/
bool IsAttachmentDeviceAttached(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       拡張デバイスの有効化・無効化を切り替えます。
 *
 * @details     拡張デバイスの有効化・無効化を切り替えます。
 *              本機能は拡張デバイス側との通信準備が完了するまでブロッキングします。
 *
 * @param[in]   isEnabled                   有効にするか否か
 * @param[in]   version                     拡張デバイス機能を使用するために必要なバージョン
 * @param[in]   handle                      有効・無効を切り替えるデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                                     処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,                           デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,            拡張デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout, 拡張デバイスとの UART 通信準備のタイムアウトです。}
 *      @handleresult{nn::xcd::ResultAttachmentDeviceInquiryTimeout,         拡張デバイスへの Inquiry コマンドのタイムアウトです。}
 *      @handleresult{nn::xcd::ResultAttachmentDeviceNotSupported,           拡張デバイス接続に対応していないデバイスです。}
 * @endretresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 */
Result AttachmentEnable(bool isEnabled, AttachmentFunctionRequiredVersion version, DeviceHandle handle) NN_NOEXCEPT;

/**
* @brief       拡張デバイスの情報を取得します。
*
* @details     拡張デバイスの情報を取得します。
*
* @param[in]   pOutExtDevInfo              拡張デバイス情報
* @param[in]   handle                      拡張デバイスの情報を取得するデバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                 デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,  拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,     拡張デバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*/
Result GetExtDevInfo(ExtDevInfo* pOutExtDevInfo, DeviceHandle handle) NN_NOEXCEPT;

/**
* @brief       拡張デバイスへコマンドを送信します。
*
* @details     拡張デバイスへコマンドを送信します。
*
* @param[in]   pInCommand              　　送信するコマンドが入ったバッファへのポインタ
* @param[in]   inCommandSize               送信するコマンドのサイズ
* @param[in]   handle                      コマンドを送信する拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                 デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,  拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,     拡張デバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*  - inCommandSize は AttachmentSendCommandSizeMax 以下である必要があります。
*/
Result SendCommandToAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, DeviceHandle handle) NN_NOEXCEPT;

/**
* @brief       拡張デバイスからの受信済みコマンドを取得します。
*
* @details     拡張デバイスからの受信済みコマンドを取得します。
*
* @param[out]  pOutSize                 　 受信済みコマンドをコピーしたサイズ
* @param[out]  pOutCommand              　 受信済みコマンドを書き込むバッファへのポインタ
* @param[in]   outCommandSize              受信済みコマンドを書き込むバッファのサイズ
* @param[in]   handle                      コマンドを送信する拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                 デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,  拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,     拡張デバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*  - outCommandSize は AttachmentReceiveCommandSizeMax 以下である必要があります。
*/
Result GetReceivedCommandFromAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, DeviceHandle handle) NN_NOEXCEPT;


/**
* @brief       拡張デバイスからのデータ受信を通知するイベントを登録します。
*
* @details     拡張デバイスからのデータ受信を通知するイベントを登録します。
*
* @param[in]   pAttachmentDataReceiveEvent 拡張デバイスからのデータ受信を通知するイベントへのポインタ
* @param[in]   handle                      コマンドを送信する拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                 デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,  拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,     拡張デバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*/
nn::Result SetAttachmentEvent(nn::os::SystemEventType* pAttachmentDataReceiveEvent, DeviceHandle handle) NN_NOEXCEPT;


/**
* @brief       拡張デバイスの Polling Receive モードを有効化します。
*
* @details     拡張デバイスの Polling Receive モードを有効化します。
*              本機能の呼び出しにより、 Joy-Con から拡張デバイスへ定期的にコマンドを送信し、その結果を受け取ります。
*              通信間隔は Joy-Con の通信間隔に依存します。
*              本機能を有効化すると六軸センサーの最も古いデータが使用できなくなります。
*
* @param[in]   pInCommand              　　定期送信するコマンドが入ったバッファへのポインタ
* @param[in]   inCommandSize               定期送信するコマンドのサイズ
* @param[in]   handle                      定期送受信を行う拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                                     処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                           デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,            拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,               拡張デバイスとの通信ができない状態です。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout, 拡張デバイスとの通信準備のタイムアウトです。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*/
Result EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, DeviceHandle handle, AttachmentPollingMode mode) NN_NOEXCEPT;

/**
* @brief       拡張デバイスの Polling Receive モードを無効化します。
*
* @details     拡張デバイスの Polling Receive モードを無効化します。
*
* @param[in]   handle                      定期送受信を無効化する拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                                     処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                           デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,            拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,               拡張デバイスとの通信ができない状態です。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout, 拡張デバイスとの通信準備のタイムアウトです。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*/
Result DisablePollingReceiveModeForAttachmentDevice(DeviceHandle handle) NN_NOEXCEPT;

/**
* @brief       Polling Mode で拡張デバイスから受信したデータを取得します。
*
* @details     Polling Mode で拡張デバイスから受信したデータを取得します。
*
* @param[out]  pOutSize                 　 受信済みコマンドをコピーしたサイズ
* @param[out]  pOutCommand              　 受信済みコマンドを書き込むバッファへのポインタ
* @param[in]   outCommandSize              受信済みコマンドを書き込むバッファのサイズ
* @param[in]   handle                      データを取得数拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                 デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,  拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,     拡張デバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - XCD ライブラリは初期化済である必要があります。
*  - AttachmentEnable() で拡張デバイスが有効になっている必要があります。
*  - outCommandSize は AttachmentReceiveCommandSizeMax 以下である必要があります。
*/
Result GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
