﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Display mode settings for the VI library.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_ContentType.h>
#include <nn/vi/vi_RgbRange.h>
#include <nn/vi/vi_HotplugState.h>

namespace nn { namespace vi {

    /**
     * @brief Maximum number of modes that may occur on a given platform.
     */
    const int DisplayModeCountMax = 16;

    struct DisplayModeInfo;
    class Display;

    /**
     * @brief List all available modes on the display.
     *
     * @param[out] pOutModes    Buffer to write mode information.
     * @param[in]  modeCountMax The maximum number of DisplayModeInfo objects that exist
     *                          in the buffer.
     * @param[in]  pDisplay     Display handle.
     *
     * @return The number of valid DisplayModeInfo objects written to @a pOutModes.
     *
     * @pre  VI library is initialized. @n
     *       @a modeCountMax >= 0 @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post Information about the available display modes is written to @a pOutModes and
     *       the number of modes is returned.
     *
     * @details There may be multiple modes available for a given resolution.  Each permutation of
     *          the refresh rate and stereoscopic availability will be represented as a unique entry.
     *
     *          The platform may further restrict the modes available for use.  Modes not supported
     *          by the platform will not be returned by this function.
     *
     *          There will always be at least one mode available on the display.
     *          For HDMI-capable displays, 480p must be a valid option even in
     *          case of errors.
     *
     *          The maximum number of modes that may be returned on the current platform is defined
     *          as @ref DisplayModeCountMax.  Please use this constant for determining the buffer size
     *          for @a pOutModes to avoid dropped results.
     */
    int ListDisplayModes(DisplayModeInfo* pOutModes, int modeCountMax, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief List all available RGB ranges on the display.
     *
     * @param[out] pOutRanges  Buffer to write range information.
     * @param[in]  rgbCountMax The maximum number of RGB range enumerations that exist
     *                         in the buffer.
     * @param[in]  pDisplay    Display handle.
     *
     * @return The number of valid RGB ranges written to @a pOutRanges.
     *
     * @pre  VI library is initialized. @n
     *       @a rgbCountMax >= 0 @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post Information about the available RGB ranges is written to @a pOutRanges and
     *       the number of ranges is returned.
     *
     * @details This function will only query the capabilities of the display, not the
     *          current setting.
     *
     *          The maximum number of RGB ranges that may be returned on the current platform
     *          is defined as @ref RgbRange_Max.  Please use this value for determining the buffer
     *          size for @a pOutRanges to avoid dropped results.
     */
    int ListDisplayRgbRanges(RgbRange* pOutRanges, int rgbCountMax, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief List all available content types on the display.
     *
     * @param[out] pOutTypes           Buffer to write content type information.
     * @param[in]  contentTypeCountMax The maximum number of content type enumerations that exist
     *                                 in the buffer.
     * @param[in]  pDisplay            Display handle.
     *
     * @return The number of valid content types written to @a pOutTypes.
     *
     * @pre  VI library is initialized. @n
     *       @a contentTypeCountMax >= 0 @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post Information about the available content types is written to @a pOutTypes and
     *       the number of types is returned.
     *
     * @details This function will only query the capabilities of the display, not the
     *          current setting.
     *
     *          The maximum number of content types that may be returned on the current platform is
     *          defined as @ref ContentType_Max.  Please use this constant for determining the buffer
     *          size for @a pOutTypes to avoid dropped results.
     */
    int ListDisplayContentTypes(ContentType* pOutTypes, int contentTypeCountMax, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Retrieves information about the mode currently set for the display.
     *
     * @param[out] pOutMode Display mode information.
     * @param[in]  pDisplay Display handle.
     *
     * @return                           Whether the mode was retrieved successfully.
     * @retval ResultSuccess             The mode for @a pDisplay was retrieved successfully.
     * @retval ResultDisplayDisconnected The value cannot be retrieved because the display is disconnected.
     * @retval ResultOperationFailed     An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutMode != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The mode of @a pDisplay is written to @a pOutMode.
     */
    Result GetDisplayMode(DisplayModeInfo* pOutMode, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the mode of the display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] pMode    The new display mode to use.
     *
     * @return                           Whether the mode was changed successfully.
     * @retval ResultSuccess             The display's mode matches @a pMode.
     * @retval ResultNotSupported        The display cannot support the requested mode.
     * @retval ResultDisplayDisconnected The value cannot be set because the display is disconnected.
     * @retval ResultOperationFailed     An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pMode != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The mode of @a pDisplay now matches that of @a pMode.
     */
    Result SetDisplayMode(Display* pDisplay, const DisplayModeInfo* pMode) NN_NOEXCEPT;

    /**
     * @brief Retrieves how much underscan is present on the display.
     *
     * @param[out] pOutUnderscan The amount of underscan used on the display.
     * @param[in]  pDisplay      Display handle.
     *
     * @return                       Whether the underscan of @a pDisplay was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's underscan.
     * @retval ResultNotSupported    The display does not support underscan.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutUnderscan != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The amount of underscan used on @a pDisplay is written to @a pOutUnderscan.
     *
     * @details The amount of underscan is represented as the reduction (in percent) of the active area.
     *
     *          Active area calculation:
     *            - width  = resolutionWidth * (1 - underscan / 100.f)
     *            - height = resolutionHeight * (1 - underscan / 100.f)
     */
    Result GetDisplayUnderscan(int* pOutUnderscan, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the amount of underscan used by this display.
     *
     * @param[in] pDisplay  Display handle.
     * @param[in] underscan The requested percentage of underscan.
     *
     * @return                       Whether the amount of underscan used by @a pDisplay matches that of
     *                               @a underscan.
     * @retval ResultSuccess         The display's underscan matches @a underscan.
     * @retval ResultInvalidRange    The valid range is 0 <= @a underscan <= 20.
     * @retval ResultNotSupported    The display does not support underscan.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The underscan used by @a pDisplay matches that of @a underscan.
     *
     * @details The amount of underscan is represented as the reduction (in percent) of the active area.
     *
     *          Active area calculation:
     *            - width  = resolutionWidth * (1 - underscan / 100.f)
     *            - height = resolutionHeight * (1 - underscan / 100.f)
     */
    Result SetDisplayUnderscan(Display* pDisplay, int underscan) NN_NOEXCEPT;

    /**
     * @brief Retrieves the content type used by the display.
     *
     * @param[out] pOutType The content type used by the display.
     * @param[in]  pDisplay Display handle.
     *
     * @return                       Whether the content type of @a pDisplay was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's content type.
     * @retval ResultNotSupported    The display's current configuration does not support
     *                               content types.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutType != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The content type used on @a pDisplay is written to @a pOutType.
     */
    Result GetDisplayContentType(ContentType* pOutType, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the content type used by the display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] type     The content type to use on the display.
     *
     * @return                       Whether the content type of @a type is
     *                               currently used on @a pDisplay.
     * @retval ResultSuccess         The display's content type matches @a type.
     * @retval ResultNotSupported    The current display does not support this content type.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The content type used by @a pDisplay matches that of @a type.
     */
    Result SetDisplayContentType(Display* pDisplay, ContentType type) NN_NOEXCEPT;

    /**
     * @brief Retrieves the RGB range sent to the display.
     *
     * @param[out] pOutRange The RGB range.
     * @param[in] pDisplay   Display handle.
     *
     * @return                       Whether the RGB range used by the display was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's RGB range.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutRange != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The RGB range used on @a pDisplay is written to @a pOutRange.
     *
     * @details This function only queries what range is being sent to the display.
     *          It does not enumerate the capabilities.
     */
    Result GetDisplayRgbRange(RgbRange* pOutRange, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the RGB range used by the display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] range    The RGB range to use on the display.
     *
     * @return                       Whether the RGB range on @a pDisplay matches that of @a range.
     * @retval ResultSuccess         The display's RGB range matches @a range.
     * @retval ResultNotSupported    This RGB range is not supported on this display.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The RGB range used by @a pDisplay matches that of @a range.
     *
     * @details This function only sets what range is being sent to the display.
     */
    Result SetDisplayRgbRange(Display* pDisplay, RgbRange range) NN_NOEXCEPT;

}}
