﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   System events relating to displays.
 * @details The functions described in this file are for internal developers only
 *          and not to be released to third parties.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/vi/vi_Types.h>

namespace nn { namespace vi {

    const int ErrorLengthMax = 4096;

    struct CompositorError
    {
        char buffer[ErrorLengthMax];
    };

    /**
     * @brief Initializes an event that will signal on Error.
     *
     * @param[out] pOutEvent Event to initialize.
     * @param[in]  pDisplay  Display that will trigger Error events.
     *
     * @return                       Whether the event was initialized successfully.
     * @retval ResultSuccess         The event was initialized successfully.
     * @retval ResultNotSupported    @a pDisplay cannot support these notifications.
     * @retval ResultDenied          Not allowed to open any more events of this type.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pOutEvent != nullptr
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post @a pOutEvent will signal on @a pDisplay Error events.
     *
     * @details @a pOutEvent will autoclear.  Only one thread will wake up when the event is signalled.
     *
     *          This function should only be used for internal debugging.
     */
    nn::Result GetDisplayErrorEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT;

    /**
    * @brief Retrieves the compositor error information.
    *
    * @param[out] pOutErrorInfo Compositor error information strings.
    * @param[in]  errorID   compositor error id .
    * @param[in]  pDisplay  Display handle.
    *
    * @return         compositor's error information was written
    *                 to @a pOutErrorInfo.
    * @retval         The length of the strings which was written to @a pOutErrorInfo buffer.
    *                 Returns -1 if it fails to retrieve the information
    *
    * @pre  VI library is initialized. @n
    *       @a pOutState != nullptr @n
    *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
    *
    * @details to retrieve valid error information, @a errorID has to match with nn::erpt::CompositorXXXState
    *          output path.
    */
    int GetCompositorErrorInfo(CompositorError* pOutErrorInfo, int errorID, Display* pDisplay) NN_NOEXCEPT;

}}
