﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Functions to manipulate display configuration.
 * @details The functions described in this file are for internal developers only
 *          and not to be released to third parties.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/vi_PowerState.h>

namespace nn { namespace vi {

    /**
     * @brief Sets the alpha value for all layers on a particular display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] alpha    Requested alpha value.
     *
     * @return                       Whether the alpha value matches all layers on the requested display.
     * @retval ResultSuccess         All layers' alpha on @a pDisplay matches that of @a alpha.
     * @retval ResultInvalidRange    The valid range is 0.f <= @a alpha <= 1.f.
     * @retval ResultNotSupported    The display doesn't support this operation.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post All layers on @a pDisplay have an alpha value of @a alpha.
     */
    nn::Result SetDisplayAlpha(Display* pDisplay, float alpha) NN_NOEXCEPT;

    /**
     * @brief Sets the physical display (i.e. LCD or DP/HDMI) to use as the default display and
     *        performs a transition sequence if necessary.
     *
     * @param[in] pDisplay Display handle.
     *
     * @return                                Whether the requested display is set as the new default display.
     * @retval ResultSuccess                  @a pDisplay is now the default display.
     * @retval ResultNotSupported             @a pDisplay is not capable of being the default display.
     * @retval ResultDisplayControllerFailure Failed to enable display controller for @a pDisplay.
     * @retval ResultDisplayInterfaceFailure  Failed to enable the interface (i.e. HDMI/DP/DSI) for @a pDisplay.
     * @retval ResultOperationFailed          An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post @a pDisplay is now the default display.
     *
     * @details If there is another default display currently set, this function will
     *          transition to the new display.  To avoid blocking applets actively flipping
     *          to the display, do not disable the current display prior to calling this function.
     *
     *          It is valid to call this function even if the requested display is not connected.
     */
    nn::Result SetDefaultDisplay(Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the layer stack for a particular display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] id       The new layer stack to use for the display.
     *
     * @return                       Whether the display is using the requested layer stack.
     * @retval ResultSuccess         The display is now using the requested layer stack.
     * @retval ResultNotSupported    The display doesn't support this operation.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post @a pDisplay is now using layer stack @a id.
     */
    nn::Result SetDisplayLayerStack(Display* pDisplay, LayerStack id) NN_NOEXCEPT;

    /**
     * @brief Set the display's power state.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] state    The power state the display should use.
     *
     * @return                                Whether the power state was set successfully.
     * @retval ResultSuccess                  The display's power state matches @a state.
     * @retval ResultNotSupported             The display cannot support the requested power state.
     * @retval ResultDisplayControllerFailure The display controller for @a pDisplay could not be adjusted.
     * @retval ResultDisplayInterfaceFailure  Failed to adjust the interface (i.e. HDMI/DP/DSI) for @a pDisplay.
     * @retval ResultOperationFailed          An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized.
     *      - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The power state of @a pDisplay now matches that of @a state.
     *
     * @details When transitioning to nn::vi::PowerState_Blank, the exact operation that occurs depends on the
     *          current state of the display pipeline.  If the interface (i.e. HDMI/DP/DSI) is fully enabled,
     *          then the DC is left running after disabling the interface.  When transitioning from the
     *          nn::vi::PowerState_Off state, only DC is enabled.
     */
    nn::Result SetDisplayPowerState(Display* pDisplay, PowerState state) NN_NOEXCEPT;

}}
