﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdarg>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <system/window.h>
#include <android/native_window.h>
#include <utils/Errors.h>
#include <nn/vi/vi_Types.h>

namespace nn{ namespace vi{ namespace native{

    template<typename T>
    class TNativeWindow
        : public ANativeWindow
    {
    public:
        typedef T SelfType;
    public:
        TNativeWindow() NN_NOEXCEPT
            : ANativeWindow()
        {
            this->common.incRef = IncRefImpl;
            this->common.decRef = DecRefImpl;

            const_cast<int&>(this->ANativeWindow::minSwapInterval) = 0;
            const_cast<int&>(this->ANativeWindow::maxSwapInterval) = INT32_MAX;

            this->ANativeWindow::setSwapInterval = NativeSetSwapInterval;
            this->ANativeWindow::dequeueBuffer_DEPRECATED = NativeDequeueBufferDeprecated;
            this->ANativeWindow::lockBuffer_DEPRECATED = NativeLockBufferDeprecated;
            this->ANativeWindow::queueBuffer_DEPRECATED = NativeQueueBufferDeprecated;
            this->ANativeWindow::query = NativeQuery;
            this->ANativeWindow::perform = NativePerform;
            this->ANativeWindow::cancelBuffer_DEPRECATED = NativeCancelBufferDeprecated;
            this->ANativeWindow::dequeueBuffer = NativeDequeueBuffer;
            this->ANativeWindow::queueBuffer = NativeQueueBuffer;
            this->ANativeWindow::cancelBuffer = NativeCancelBuffer;
        }

        NativeWindowHandle GetNativeWindowHandle() NN_NOEXCEPT
        {
            return static_cast<ANativeWindow*>(this);
        }

    private:
        static void IncRefImpl(struct android_native_base_t*) NN_NOEXCEPT
        {
            // we don't use refcount
        }

        static void DecRefImpl(struct android_native_base_t*) NN_NOEXCEPT
        {
            // we don't use refcount
        }

        static SelfType* GetSelf(ANativeWindow* p) NN_NOEXCEPT
        {
            return static_cast<SelfType*>(p);
        }

        static const SelfType* GetSelf(const ANativeWindow* p) NN_NOEXCEPT
        {
            return static_cast<const SelfType*>(p);
        }

        // ANativeWindow functions
        static int NativeSetSwapInterval(ANativeWindow* p,  int interval) NN_NOEXCEPT
        {
            return GetSelf(p)->SetSwapIntervalImpl(interval);
        }

        static int NativeDequeueBuffer(ANativeWindow* p, ANativeWindowBuffer** pOutBuffer, int* pOutFenceFd) NN_NOEXCEPT
        {
            return GetSelf(p)->DequeueBufferImpl(pOutBuffer, pOutFenceFd);
        }

        static int NativeCancelBuffer(ANativeWindow* p, ANativeWindowBuffer* pBuffer, int fenceFd) NN_NOEXCEPT
        {
            return GetSelf(p)->CancelBufferImpl(pBuffer, fenceFd);
        }

        static int NativeQueueBuffer(ANativeWindow* p, ANativeWindowBuffer* pBuffer, int fenceFd) NN_NOEXCEPT
        {
            return GetSelf(p)->QueueBufferImpl(pBuffer, fenceFd);
        }

        static int NativeDequeueBufferDeprecated(ANativeWindow* p, ANativeWindowBuffer** pOutBuffer) NN_NOEXCEPT
        {
            return GetSelf(p)->DequeueBufferImpl(pOutBuffer, nullptr);
        }

        static int NativeCancelBufferDeprecated(ANativeWindow* p, ANativeWindowBuffer* pBuffer) NN_NOEXCEPT
        {
            return GetSelf(p)->CancelBufferImpl(pBuffer, -1);
        }

        static int NativeLockBufferDeprecated(ANativeWindow* p, ANativeWindowBuffer* pBuffer) NN_NOEXCEPT
        {
            NN_UNUSED(p);
            NN_UNUSED(pBuffer);
            return android::OK;
        }

        static int NativeQueueBufferDeprecated(ANativeWindow* p, ANativeWindowBuffer* pBuffer) NN_NOEXCEPT
        {
            return GetSelf(p)->QueueBufferImpl(pBuffer, -1);
        }

        static int NativeQuery(const ANativeWindow* p, int what, int* pOutValue) NN_NOEXCEPT
        {
            switch(what)
            {
            case NATIVE_WINDOW_FORMAT:
                return GetSelf(p)->QueryFormatImpl(pOutValue);
            case NATIVE_WINDOW_QUEUES_TO_WINDOW_COMPOSER:
                return GetSelf(p)->QueryQueuesToWindowComposerImpl(pOutValue);
            case NATIVE_WINDOW_CONCRETE_TYPE:
                return GetSelf(p)->QueryConcreteTypeImpl(pOutValue);
            case NATIVE_WINDOW_DEFAULT_WIDTH:
                return GetSelf(p)->QueryDefaultWidthImpl(pOutValue);
            case NATIVE_WINDOW_DEFAULT_HEIGHT:
                return GetSelf(p)->QueryDefaultHeightImpl(pOutValue);
            case NATIVE_WINDOW_TRANSFORM_HINT:
                return GetSelf(p)->QueryTransformHintImpl(pOutValue);
            case NATIVE_WINDOW_WIDTH:
                return GetSelf(p)->QueryWidthImpl(pOutValue);
            case NATIVE_WINDOW_HEIGHT:
                return GetSelf(p)->QueryHeightImpl(pOutValue);
            case NATIVE_WINDOW_MIN_UNDEQUEUED_BUFFERS:
                return GetSelf(p)->QueryMinUndequeuedBuffersImpl(pOutValue);
            case NATIVE_WINDOW_CONSUMER_RUNNING_BEHIND:
                return GetSelf(p)->QueryConsumerRunningBehindImpl(pOutValue);
            case NATIVE_WINDOW_CONSUMER_USAGE_BITS:
                return GetSelf(p)->QueryConsumerUsageBits(pOutValue);
            default:
                return GetSelf(p)->QueryDefault(pOutValue, what);
            }
        }

        static int NativePerform(ANativeWindow* p, int operation, ...) NN_NOEXCEPT
        {
            std::va_list args;
            va_start(args, operation);

            int result = android::NO_ERROR;
            switch(operation)
            {
            case NATIVE_WINDOW_CONNECT:
                {
                    result = android::NO_ERROR;
                    break;
                }
            case NATIVE_WINDOW_DISCONNECT:
                {
                    result = android::NO_ERROR;
                    break;
                }
            case NATIVE_WINDOW_SET_USAGE:
                {
                    uint32_t usage = va_arg(args, uint32_t);
                    result = GetSelf(p)->PerformSetUsageImpl(usage);
                    break;
                }
            case NATIVE_WINDOW_SET_CROP:
                {
                    const android_native_rect_t* pRect = va_arg(args, android_native_rect_t*);
                    result = GetSelf(p)->PerformSetCropImpl(pRect);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFER_COUNT:
                {
                    size_t bufferCount = va_arg(args, size_t);
                    result = GetSelf(p)->PerformSetBufferCountImpl(bufferCount);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_GEOMETRY:
                {
                    int w = va_arg(args, int);
                    int h = va_arg(args, int);
                    int f = va_arg(args, int);
                    result = GetSelf(p)->PerformSetBuffersGeometryImpl(w, h, f);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_TRANSFORM:
                {
                    int transform = va_arg(args, int);
                    result = GetSelf(p)->PerformSetBuffersTransformImpl(transform);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_FLAGS:
                {
                    uint32_t flags = va_arg(args, uint32_t);
                    result = GetSelf(p)->PerformSetBuffersFlagsImpl(flags);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_TIMESTAMP:
                {
                    int64_t timestamp = va_arg(args, int64_t);
                    result = GetSelf(p)->PerformSetBuffersTimestampImpl(timestamp);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_DIMENSIONS:
                {
                    int w = va_arg(args, int);
                    int h = va_arg(args, int);
                    result = GetSelf(p)->PerformSetBuffersDimensionsImpl(w, h);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_USER_DIMENSIONS:
                {
                    int w = va_arg(args, int);
                    int h = va_arg(args, int);
                    result = GetSelf(p)->PerformSetBuffersUserDimensionsImpl(w, h);
                    break;
                }
            case NATIVE_WINDOW_SET_BUFFERS_FORMAT:
                {
                    int f = va_arg(args, int);
                    result = GetSelf(p)->PerformSetBuffersFormatImpl(f);
                    break;
                }
            case NATIVE_WINDOW_LOCK:
                {
                    ANativeWindow_Buffer* pOutBuffer = va_arg(args, ANativeWindow_Buffer*);
                    ARect* pInOutDirtyBounds = va_arg(args, ARect*);
                    result = GetSelf(p)->PerformLockImpl(pOutBuffer, pInOutDirtyBounds);
                    break;
                }
            case NATIVE_WINDOW_UNLOCK_AND_POST:
                {
                    result = GetSelf(p)->PerformUnlockAndPostImpl();
                    break;
                }
            case NATIVE_WINDOW_SET_SCALING_MODE:
                {
                    int m = va_arg(args, int);
                    result = GetSelf(p)->PerformSetScalingModeImpl(m);
                    break;
                }
            case NATIVE_WINDOW_API_CONNECT:
                {
                    int api = va_arg(args, int);
                    result = GetSelf(p)->PerformConnectImpl(api);
                    break;
                }
            case NATIVE_WINDOW_API_DISCONNECT:
                {
                    int api = va_arg(args, int);
                    result = GetSelf(p)->PerformDisconnectImpl(api);
                    break;
                }
            case NATIVE_WINDOW_SET_SIDEBAND_STREAM:
                {
                    native_handle_t* pHandle = va_arg(args, native_handle_t*);
                    result = GetSelf(p)->PerformSetSidebandStreamImpl(pHandle);
                    break;
                }
            case NATIVE_WINDOW_SET_PREALLOCATED_BUFFER:
                {
                    size_t slot = va_arg(args, size_t);
                    const native_handle_t* handle = va_arg(args, const native_handle_t*);
                    int width = va_arg(args, int);
                    int height = va_arg(args, int);
                    int format = va_arg(args, int);
                    int usage = va_arg(args, int);
                    int stride = va_arg(args, int);
                    result = GetSelf(p)->PerformSetPreallocatedBufferImpl(slot, handle, width, height, format, usage, stride);
                    break;
                }
            default:
                {
                    result = GetSelf(p)->PerformDefault(operation, args);
                    break;
                }
            }

            va_end(args);
            return result;
        }// NOLINT(impl/function_size)
    };

}}}
