﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace usb {
namespace detail {

template<class T>
class RingBase
{
public:
    typedef T ItemType;

public:
    RingBase() NN_NOEXCEPT
        : m_pStorage(nullptr)
        , m_ItemCount(0)
    {
    }

    /*
     * RingBase is not supposed to be used as is. You have to derive from it
     * and provide some means to initialize m_pStorage and m_ItemCount
     *
     * Use a pure virtual destructor to enforce that.
     */
    virtual ~RingBase() NN_NOEXCEPT = 0;

    bool IsFull() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);

        return Advance(m_pStorage->head) == m_pStorage->tail;
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);

        return m_pStorage->head == m_pStorage->tail;
    }

    T* Head() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);

        const uint32_t head = m_pStorage->head;

        return IsValid(head) ? Get(head) : nullptr;
    }

    T* Tail() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);

        const uint32_t tail = m_pStorage->tail;

        return IsValid(tail) ? Get(tail) : nullptr;
    }

    T* Next(T* pItem) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);

        if (!IsValid(pItem))
        {
            return nullptr;
        }

        if (pItem + 1 == Get(m_ItemCount))
        {
            return Get(0);
        }
        else
        {
            return IsValid(pItem + 1) ? pItem + 1 : nullptr;
        }
    }

    T* Expand(uint32_t count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);
        NN_SDK_REQUIRES(count > 0);

        const uint32_t head = m_pStorage->head;
        const uint32_t tail = m_pStorage->tail;

        if (!IsValid(head) || !IsValid(tail))
        {
            return nullptr;
        }

        const uint32_t available = tail + m_ItemCount * (tail <= head ? 1 : 0) - head - 1;

        if (count > available)
        {
            return nullptr;
        }
        else
        {
            m_pStorage->head = (head + count) % m_ItemCount;
            return Get(head);
        }
    }

    T* Expand() NN_NOEXCEPT
    {
        return Expand(1);
    }

    T* Shrink(uint32_t count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pStorage != nullptr);
        NN_SDK_REQUIRES(m_ItemCount > 0);
        NN_SDK_REQUIRES(count > 0);

        const uint32_t head = m_pStorage->head;
        const uint32_t tail = m_pStorage->tail;

        if (!IsValid(head) || !IsValid(tail))
        {
            return nullptr;
        }

        const uint32_t available = head + m_ItemCount * (head < tail ? 1 : 0) - tail;

        if (count > available)
        {
            return nullptr;
        }
        else
        {
            m_pStorage->tail = (tail + count) % m_ItemCount;
            return Get(tail);
        }
    }

    T* Shrink() NN_NOEXCEPT
    {
        return Shrink(1);
    }

protected:
    /*
     * We prefer that the data and metadata of the ring can be placed in one
     * memory block, so it can be transferred as a whole.
     */
    typedef struct Storage {
        uint32_t         head;
        uint32_t         tail;

        // At least one sentinel entry
        T                data[1];
    } Storage;

    Storage  *m_pStorage;
    uint32_t  m_ItemCount;  // too critical to put into shared memory

protected:
    uint32_t Advance(uint32_t i) const NN_NOEXCEPT
    {
        return (i + 1) % m_ItemCount;
    }

    T* Get(uint32_t i) const NN_NOEXCEPT
    {
        return &m_pStorage->data[i];
    }

    bool IsValid(uint32_t index) const NN_NOEXCEPT
    {
        return index < m_ItemCount;
    }

    bool IsValid(T *pItem) const NN_NOEXCEPT
    {
        return pItem >= Get(0) && pItem + 1 <= Get(m_ItemCount);
    }
};

template<class T>
RingBase<T>::~RingBase() NN_NOEXCEPT
{
}

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn


