﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバライブラリのポートアクセス機能に関する型や定数の定義。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_TypedStorage.h>

namespace nn {
namespace uart {

/**
 * @brief   ボーレートを表す型です。
 */
enum BaudRate
{
    /**
     * @brief   57600 baud
     */
    BaudRate_57600 = 57600,

    /**
     * @brief   115200 baud
     */
    BaudRate_115200 = 115200,

    /**
     * @brief   1M baud
     */
    BaudRate_1M = 1000000,

    /**
     * @brief   3M baud
     */
    BaudRate_3M = 3000000
};

/**
 * @brief   フロー制御モードを表す型です。
 */
enum FlowControlMode
{
    /**
     * @brief   フロー制御を行いません。
     */
    FlowControlMode_None,

    /**
     * @brief   CTS/RTS 信号ベースのハードウェアフロー制御を行います。
     */
    FlowControlMode_Hardware
};

/**
 * @brief   ポート設定オブジェクトです。
 *
 * @details
 *  ポート設定オブジェクトは、ポートのオープン時に指定する設定パラメータを保持する構造体です。@n
 *  メンバーに直接アクセスしないでください。@n
 *
 *  本オブジェクトのパラメータ設定には @ref nn::uart::InitializePortConfig() または @ref nn::uart::PortConfig() を使用してください。
 *
 * @see
 *  nn::uart::InitializePortConfig(), nn::uart::PortConfig()
 */
struct PortConfigType
{
    BaudRate        _baudRate;
    FlowControlMode _flowControlMode;
    char*           _sendBuffer;
    size_t          _sendBufferLength;
    char*           _receiveBuffer;
    size_t          _receiveBufferLength;
    bool            _isInvertTx;
    bool            _isInvertRx;
    bool            _isInvertRts;
    bool            _isInvertCts;
};

/**
 * @brief   ポートの状態変化の種類を表す型です。
 *
 * @platformbegin{NX}
 * - @ref PortEventType_ReceiveEnd は、連続したデータの受信後、データを 4 キャラクタ分の期間受信しなかった場合に発生します。
 *
 * @platformend
 */
enum PortEventType
{
    /**
     * @brief   送信予約済のすべてのデータの送信が完了しました。
     */
    PortEventType_SendBufferEmpty,

    /**
     * @brief   送信バッファに閾値以上の空きがあります。
     */
    PortEventType_SendBufferReady,

    /**
     * @brief   受信バッファに閾値以上の受信済データがあります。
     */
    PortEventType_ReceiveBufferReady,

    /**
     * @brief   連続したデータの受信が終了しました。@n
     *          この状態変化は、1 バイト以上の連続したデータを受信した後、一定時間データを受信しなかった場合に受信の区切りとして発生します。
     */
    PortEventType_ReceiveEnd
};

} // uart
} // nn

