﻿/*
 * Copyright (c) 2015-2016, NVIDIA CORPORATION.  All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property
 * and proprietary rights in and to this software, related documentation
 * and any modifications thereto.  Any use, reproduction, disclosure or
 * distribution of this software and related documentation without an express
 * license agreement from NVIDIA Corporation is strictly prohibited.
 */

/**
 * @file
 * @brief Tegra System Profiler 機能の API 宣言。
 */

#pragma once

/**
 * @namespace   nn::tsp
 * @brief       Tegra System Profiler (TSP) ライブラリの名前空間です。
 */

namespace nn {
namespace tsp {

/**
 * @brief Tegra System Profiler がホストPCとの通信に利用するコア
 */
enum ProfilerWorkerThreadCoreNo
{
    ProfilerWorkerThreadCoreNo_Default,  //!< ターゲット環境における規定のコアを利用します
    ProfilerWorkerThreadCoreNo_Core0,    //!< コア 0 を利用します
    ProfilerWorkerThreadCoreNo_Core1,    //!< コア 1 を利用します
    ProfilerWorkerThreadCoreNo_Core2,    //!< コア 2 を利用します
    ProfilerWorkerThreadCoreNo_Core3,
};

//--------------------------------------------------------------------------
/**
 * @brief Tegra System Profiler の初期化を行います
 *
 * @param [in] userBuffer               TSP が利用するバッファ
 * @param [in] userBufferSize           バッファのサイズ
 * @param [in] threadCoreNo             ホスト PC と通信するスレッドを立てるコア番号
 * @param [in] threadPriority           ホスト PC と通信するスレッドの優先度
 * @param [in] waitForProfilerToConnect ホスト PC との接続を待つかどうか
 *
 * @pre userBuffer は nn::os::MemoryPageSize (4KB) でアラインされている。
 *
 * @details
 *  Tegra System Profiler (TSP) の初期化を行います。
 *
 *  TSP は userBuffer で指定されたメモリを使用します。
 *  バッファとして最低 64MB のメモリが必要です。多くの場合 128MB あれば問題なく動作します。
 *
 *  サンプリング機能のための処理スレッドを利用できるすべてのコアに生成します。
 *  また、ホスト PC と通信するためのスレッドを、threadCoreNo で指定したコアに
 *  threadPriority で指定した優先度で生成します。
 *  ProfilerWorkerThreadCoreNo_Default を指定した場合、ターゲット環境における規定のコアに生成します。
 *
 *  waitForProfilerToConnect に true を指定した場合、
 *  ホスト PC 側の TSP と接続されるまで 本 API 内部でブロックします。
 *
 *  Tegra System Profiler 利用時には本 API が呼び出し済みである必要があります。
 *
 */
void Initialize(void* userBuffer, size_t userBufferSize,
                ProfilerWorkerThreadCoreNo threadCoreNo,
                int threadPriority, bool waitForProfilerToConnect);

//--------------------------------------------------------------------------
/**
 * @brief Tegra System Profiler の初期化を行います
 *
 * @param [in] userBuffer     TSP が利用するバッファ
 * @param [in] userBufferSize バッファのサイズ
 *
 * @pre userBuffer は nn::os::MemoryPageSize (4KB) でアラインされている。
 *
 * @details
 *  Tegra System Profiler (TSP) の初期化を行います。
 *
 *  TSP は userBuffer で指定されたメモリを使用します。
 *  バッファとして最低 64MB のメモリが必要です。多くの場合 128MB あれば問題なく動作します。
 *
 *  サンプリング機能のための処理スレッドを利用できるすべてのコアに生成します。
 *  また、ホスト PC と通信するためのスレッドを 優先度 17 でターゲット環境における規定のコアに生成します。
 *
 *  ホスト PC 側の TSP と接続されていない場合でも、本 API はブロックせずに抜けます。
 *
 *  Tegra System Profiler 利用時には本 API が呼び出し済みである必要があります。
 *
 */
void Initialize(void* userBuffer, size_t userBufferSize);

//--------------------------------------------------------------------------
/**
 * @brief 本 API は deprecated になりました。引数を指定する Initialize() を使用してください。
 *
 * @details
 *  Tegra System Profiler (TSP) の初期化を行います。
 *
 *  サンプリング機能のための処理スレッドを利用できるすべてのコアに生成します。
 *  また、ホスト PC と通信するためのスレッドを nn::os::DefaultThreadPriority よりひとつ下の優先度で
 *  ターゲット環境における規定のコアに生成します。
 *
 *  本 API を使用した場合、100MB の静的メモリを確保します。
 *  引数を指定する Initialize() を使用することで
 *  TSP が使用するメモリを指定することができます。
 *
 *  ホスト PC 側の TSP と接続されていない場合でも、本 API はブロックせずに抜けます。
 *
 *  Tegra System Profiler 利用時には本 API が呼び出し済みである必要があります。
 *
 * @deprecated 本 API は deprecated になりました。引数を指定する Initialize() を使用してください。
 *
 */
void Initialize();

//--------------------------------------------------------------------------
/**
 * @brief Tegra System Profiler の終了処理を行います
 *
 * @details
 *  Tegra System Profiler の終了処理を行います。
 *
 */
void Finalize();

//--------------------------------------------------------------------------
/**
 * @brief NVN API の呼び出しの追跡を有効にします
 *
 * @pre tsp::Initialize() が呼ばれている
 * @pre nv::SetGraphicsAllocator() が呼ばれている
 * @pre nv::InitializeGraphics() が呼ばれている
 *
 * @details
 *  Tegra System Profiler から NVN API の呼び出しを追跡します。
 *
 *  本 API を呼び出した上でプロファイリングを行うことで、Tegra System Profiler のタイムライン上に
 *  NVN API の呼び出し状況が表示されるようになります。
 *
 *  本 API は、tsp::Initialize()、nv::SetGraphicsAllocator()、nv::InitializeGraphics()
 *  よりも後に呼び出す必要があります。
 *
 */
void EnableNvnTrace();

//--------------------------------------------------------------------------
/**
 * @brief NVN API の呼び出しの追跡を無効にします
 *
 * @details
 *  Tegra System Profiler による NVN API の呼び出し追跡を無効にします。
 *
 */
void DisableNvnTrace();

/**
 * @brief GPU トレースを有効にします
 *
 * @param [in] userBuffer     GPU トレースのために TSP が利用するバッファ
 * @param [in] userBufferSize バッファのサイズ
 *
 * @pre userBuffer は nn::os::MemoryPageSize (4KB) でアラインされている。
 * @pre tsp::EnableNvnTrace() が呼び出し済みである。
 *
 * @details
 *  Tegra System Profiler (TSP) による GPU 内部のイベントのトレースを有効にします。
 *  userBuffer に必要なメモリサイズについては TSP のドキュメントを参照してください。
 *
 *  本 API は、アプリケーションで利用する NVN キューが 1 の場合に利用可能です。
 *  より大きいキューを利用する場合には EnableNvnGpuTrace(void* userBuffer, size_t userBufferSize, size_t queueCount) を利用してください。
 *
 *  本 API は tsp::EnableNvnTrace() よりも後に呼び出す必要があります。
 *
 */
void EnableNvnGpuTrace(void* userBuffer, size_t userBufferSize);

/**
 * @brief GPU トレースを有効にします
 *
 * @param [in] userBuffer     GPU トレースのために TSP が利用するバッファ
 * @param [in] userBufferSize バッファのサイズ
 * @param [in] queueCount     アプリケーションで利用する NVN キューの数
 *
 * @pre userBuffer は nn::os::MemoryPageSize (4KB) でアラインされている。
 * @pre tsp::EnableNvnTrace() が呼び出し済みである。
 *
 * @details
 *  Tegra System Profiler (TSP) による GPU 内部のイベントのトレースを有効にします。
 *  userBuffer に必要なメモリサイズについては TSP のドキュメントを参照してください。
 *
 *  アプリケーションで利用する NVN キューと同数を queueCount に与えてください。
 *  TSP は与えられたバッファをキューの数で分割して利用します。
 *  queueCount が少ない場合、TSP は動作を停止します。
 *
 *  本 API は tsp::EnableNvnTrace() よりも後に呼び出す必要があります。
 *
 */
void EnableNvnGpuTrace(void* userBuffer, size_t userBufferSize, size_t queueCount);

/**
 * @brief GPU トレースを無効にします
 *
 * @details
 *  GPU トレースを無効にします。
 *
 */
void DisableNvnGpuTrace();

/**
 * @brief スレッド状態トレースを有効にします
 *
 * @pre tsp::Initialize() が呼ばれている
 *
 * @details
 *  Tegra System Profiler (TSP) からスレッド状態の変化をもたらす可能性のある機能の呼び出しを追跡します。
 *  例えば、スレッドが待ち状態に入る可能性のある機能の呼び出しの検出に利用できます。
 *
 *  ただし、スレッドの状態変化を起こす可能性のある全ての機能を追跡できるわけではありません。
 *  本機能が対象としない条件の変化によってスレッドの状態が変化する場合もあります。
 *
 *  本機能を利用するには nnSdk.nso ではなく nnSdkProfiler(Jp|En).nso を利用してください。
 *  詳細は TSP のドキュメントを参照してください。
 *
 */
void EnableThreadStateTrace();

/**
 * @brief スレッド状態トレースを無効にします
 *
 * @details
 *  スレッド状態トレースを無効にします
 *
 */
void DisableThreadStateTrace();

}} // namespace nn::tsp
