﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/os.h>

#include <nn/time/time_LocationName.h>
#include <nn/time/time_SystemClockContext.h>
#include <nn/time/detail/time_CommonDetail.h>
#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>

namespace nn
{
namespace timesrv
{
namespace detail
{

namespace service
{
    class PosixTimeNotifierToFs;
}

namespace core
{

    class SharedMemoryManager;

    //!< nn::settings の Write や RTC 読み込みなど、待ち時間の長い処理を非同期に行うクラス
    class WorkerThread
    {
        NN_DISALLOW_COPY(WorkerThread);
        NN_DISALLOW_MOVE(WorkerThread);

    public:
        WorkerThread(
            core::StandardSteadyClockCore* pStandardSteadyClockCore,
            core::SharedMemoryManager* pSharedMemoryManager,
            nn::timesrv::detail::service::PosixTimeNotifierToFs* pPosixTimeNotifierToFs) NN_NOEXCEPT;
        ~WorkerThread() NN_NOEXCEPT;

        void WriteLocalSystemClockContextAsync(const nn::time::SystemClockContext& value) NN_NOEXCEPT;
        void WriteNetworkSystemClockContextAsync(const nn::time::SystemClockContext& value) NN_NOEXCEPT;

        static void ThreadFunction(void* p) NN_NOEXCEPT
        {
            reinterpret_cast<WorkerThread*>(p)->ThreadFunctionImpl();
        }

    private:
        nn::os::ThreadType m_Thread;
        nn::os::Event m_ThreadStopEvent;
        core::StandardSteadyClockCore* m_pStandardSteadyClockCore;
        core::SharedMemoryManager* m_pSharedMemoryManager;
        nn::timesrv::detail::service::PosixTimeNotifierToFs* m_pPosixTimeNotifierToFs;

        template<typename T>
        class EventData
        {
        public:
            EventData():
                m_Lock(false),
                m_Event(nn::os::EventClearMode_ManualClear)
            {}

            nn::os::EventType* GetEventBase() NN_NOEXCEPT
            {
                return m_Event.GetBase();
            }

            void SetAndEventSignal(const T& value) NN_NOEXCEPT
            {
                std::lock_guard<nn::os::Mutex> lock(m_Lock);
                m_Data = value;
                m_Event.Signal();
            }

            T GetAndEventClear() NN_NOEXCEPT
            {
                std::lock_guard<nn::os::Mutex> lock(m_Lock);
                m_Event.Clear();
                return m_Data;
            }

        private:
            nn::os::Mutex m_Lock;
            T m_Data;
            nn::os::Event m_Event;
        };

        EventData<nn::time::SystemClockContext> m_LocalSystemClockContextEventData;
        EventData<nn::time::SystemClockContext> m_NetworkSystemClockContextEventData;
        nn::os::TimerEvent m_SteadyClockBaseTimeUpdateEvent;
        nn::os::TimerEvent m_NotifyPosixTimeEvent;

        void ThreadFunctionImpl() NN_NOEXCEPT;
    };
}
}
}
}
