﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/time/detail/time_CommonDetail.h>

#include <nn/time/time_Types.h>


namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

class SteadyClockCore;

/**
 * @brief   時計のコンテキスト更新時に呼ばれるコールバッククラス
 */
class SystemClockContextUpdateCallback
{
    NN_DISALLOW_COPY(SystemClockContextUpdateCallback);
    NN_DISALLOW_MOVE(SystemClockContextUpdateCallback);

public:
    SystemClockContextUpdateCallback() NN_NOEXCEPT {}
    virtual Result Set(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT = 0;
};

class SystemClockCore
{
private:
    virtual Result GetSystemClockContextImpl(nn::time::SystemClockContext* pSystemClockContext) const NN_NOEXCEPT;
    virtual Result SetSystemClockContextImpl(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT;

    SystemClockContextUpdateCallback* m_pSystemClockContextUpdateCallback;

protected:
    SteadyClockCore* m_pSteadyClockCore;
    nn::time::SystemClockContext m_SystemClockContext;

public:
    explicit SystemClockCore(SteadyClockCore* pSteadyClockCore) NN_NOEXCEPT;
    virtual ~SystemClockCore() NN_NOEXCEPT;

    /**
     * @brief   時計の指し示す現在時刻を取得します。
     *
     * @param[out]  pOutPosixTime   時刻を代入するバッファへのポインタを指定します。
     *
     * @pre
     *  - pOutPosixTime != nullptr
     *
     * @return  操作の結果を返します。 TODO: Result の詳細を追記
     *
     * @details
     */
    Result GetCurrentTime(nn::time::PosixTime* pOutPosixTime) NN_NOEXCEPT;

    /**
     * @brief   現在時刻を指定して時計を補正します。
     *
     * @param[in]   posixTime   補正後の時刻への参照を指定します。
     *
     * @return  操作の結果を返します。 TODO: Result の詳細を追記
     *
     * @details
     *  補正はオフセットを変更する形で実施され、単調増加クロックには影響を与えません。
     */
    Result SetCurrentTime(const nn::time::PosixTime& posixTime) NN_NOEXCEPT;

    /**
     * @brief   時計が利用する単調増加クロックへの参照を返します。
     *
     * @return  時計が利用する単調増加クロックへの参照が返ります。
     *
     * @details
     *  この関数が返す単調増加クロックは、ほかの時計も利用している可能性があることに注意してください。
     */
    SteadyClockCore& GetSteadyClockCore() NN_NOEXCEPT;

    /**
     * @brief   時計が利用する単調増加クロックへの const 参照を返します。
     *
     * @return  時計が利用する単調増加クロックへの const 参照が返ります。
     *
     * @details
     *  この関数が返す単調増加クロックは、ほかの時計も利用している可能性があることに注意してください。
     */
    const SteadyClockCore& GetSteadyClockCore() const NN_NOEXCEPT;

    /**
     * @brief   時計が利用している SystemClockContext の値を取得します。
     *
     * @param[out]  pOutSystemClockContext  SystemClockContext を代入するバッファへのポインタを指定します。
     *
     * @pre
     *  - pOutSystemClockContext != nullptr
     *
     * @return  操作の結果を返します。 TODO: Result の詳細を追記
     *
     * @details
     */
    Result GetSystemClockContext(nn::time::SystemClockContext* pOutSystemClockContext) const NN_NOEXCEPT;

    /**
     * @brief   時計が利用する SystemClockContext の値を直接設定します。
     *
     * @param[in]   systemClockContext  設定する systemClockContext への参照を指定します。
     *
     * @return  操作の結果を返します。 TODO: Result の詳細を追記
     *
     * @details
     *  異なるソース ID のオフセットを指定すると、時計が無効化されることに注意してください。
     */
    Result SetSystemClockContext(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT;

    /**
     * @berief  時計が有効かどうかを判定します
     * @retval  true    有効（時刻を返せる状態です）
     * @retval  false   無効（時刻を返せない状態です）
     * @details
     */
    bool IsAvailable() const NN_NOEXCEPT;

    /**
     * @brief       SystemClockContext の更新時に呼ばれるコールバッククラスを登録します。
     * @param[in]   pCallback   SystemClockContextUpdateCallback インスタンスへのポインタ
     */
    void SetSystemClockContextUpdateCallback(SystemClockContextUpdateCallback* pCallback) NN_NOEXCEPT;

    /**
     * @brief   単調増加クロックの値と SystemClockContext から時刻を計算します。
     *
     * @param[out]  pOutPosixTime           時刻を代入するバッファへのポインタを指定します。
     * @param[in]   steadyClockTimePoint    設定する SteadyClockTimePoint への参照を指定します。
     * @param[in]   systemClockContext      利用する SystemClockContext への参照を指定します。
     *
     * @return  操作の結果を返します。 TODO: Result の詳細を追記
     *
     * @details
     *  ソース ID の異なる単調増加クロック値の値と SystemClockContext の組み合わせでは時刻を計算することはできません。
     */
    static Result CalculatePosixTime(nn::time::PosixTime* pOutPosixTime, const nn::time::SteadyClockTimePoint& steadyClockTimePoint, const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT;

    /**
     * @brief   現在時刻と単調増加クロックの値から SystemClockContext を計算します。
     *
     * @param[out]  pOutSystemClockContext  SystemClockContext を代入するバッファへのポインタを指定します。
     * @param[in]   posixTime               利用する PosixTime への参照を指定します。
     * @param[in]   steadyClockTimePoint    設定する SteadyClockTimePoint への参照を指定します。
     *
     * @return  操作の結果を返します。 TODO: Result の詳細を追記
     *
     * @details
     */
    static void CalculateSystemClockContext(nn::time::SystemClockContext* pOutSystemClockContext, const nn::time::PosixTime& posixTime, const nn::time::SteadyClockTimePoint& steadyClockTimePoint) NN_NOEXCEPT;
};

}
}
}
}
