﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 温度制御モジュールの API 宣言。
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/os.h>

#include "tc_Types.h"

namespace nn { namespace tc {

//! @name 仮想温度操作 API
//! @{

/**
 * @brief       指定した温度センサーの仮想温度の設定を有効にします。
 *
 * @param[in]   location 温度センサーを表す @ref nn::tc::Location 型の列挙値です。
 *
 * @return      処理の結果を返します。
 *
 * @details     指定した温度センサーの仮想温度の設定を有効にします。@n
 *              温度センサーの仮想温度は @ref nn::tc::SetVirtualTemperature 関数で設定します。
 */
nn::Result EnableVirtualTemperature(Location location) NN_NOEXCEPT;

/**
 * @brief       指定した温度センサーの仮想温度の設定を無効にします。
 *
 * @param[in]   location 温度センサーを表す @ref nn::tc::Location 型の列挙値です。
 *
 * @return      処理の結果を返します。
 *
 * @details     指定した温度センサーの仮想温度の設定を無効にします。
 */
nn::Result DisableVirtualTemperature(Location location) NN_NOEXCEPT;

/**
 * @brief       指定した温度センサーの仮想温度を設定します。
 *
 * @param[in]   location 温度センサーを表す @ref nn::tc::Location 型の列挙値です。
 * @param[in]   temperature 仮想温度を表す @ref nn::tc::TemperatureMilliC 型の値です。
 *
 * @return      処理の結果を返します。
 *
 * @details     指定した温度センサーの仮想温度を設定します。
 *              仮想温度を有効にするには設定後に @ref nn::tc::EnableVirtualTemperature 関数を実行してください。@n
 *              仮想温度はファンの回転数および @ref nn::tc::GetThermalEvent で取得したイベントの挙動を実温度に依らずに操作するための仕組みです。@n
 *              仮想温度が有効である時、実温度はこれらのハンドリングにおいて無視され、仮想温度が代わりに使用されます。@n
 *              なお、仮想温度に低い温度を設定するとファン回転数が低下して冷却能力が不足し、ハードウェアシャットダウンが発生する可能性があります。@n
 */
nn::Result SetVirtualTemperature(Location location, TemperatureMilliC temperature) NN_NOEXCEPT;

//! @}

}} // namepsace nn::tc
