﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/srepo/srepo_Types.h>

namespace nn { namespace srepo { namespace detail {

/*!
    @brief      プレイレポートデータの生成器です。
*/
class SystemReportGenerator
{
public:
    /*!
        @brief      初期化します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
    */
    static void Initialize(size_t* outPosition, Bit8* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      キーを追加します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  key         キー。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
        @param[in]  position    ストリーム位置。

        @return     処理結果。
    */
    static nn::Result AddKey(size_t* outPosition,
        const char* key, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（符号付き整数値）を追加します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  key         キー。
        @param[in]  value       バリュー。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
        @param[in]  position    ストリーム位置。

        @return     処理結果。
    */
    static nn::Result AddKeyValue(size_t* outPosition,
        const char* key, int64_t value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（任意の 64 ビット長の ID）を追加します。

        @detail
                    内部では 16 進数表現の文字列（上位桁 0 埋め 16 桁小文字）で記録します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  key         キー。
        @param[in]  value       バリュー。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
        @param[in]  position    ストリーム位置。

        @return     処理結果。
    */
    static nn::Result AddKeyValue(size_t* outPosition,
        const char* key, const Any64BitId& value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（倍精度浮動小数点数）を追加します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  key         キー。
        @param[in]  value       バリュー。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
        @param[in]  position    ストリーム位置。

        @return     処理結果。
    */
    static nn::Result AddKeyValue(size_t* outPosition,
        const char* key, double value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（文字列値）を追加します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  key         キー。
        @param[in]  value       バリュー。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
        @param[in]  position    ストリーム位置。

        @return     処理結果。
    */
    static nn::Result AddKeyValue(size_t* outPosition,
        const char* key, const char* value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（バイナリ）を追加します。

        @param[out] outPosition 新しいストリーム位置。
        @param[in]  key         キー。
        @param[in]  value       バリュー。
        @param[in]  size        バリューのサイズ。
        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
        @param[in]  position    ストリーム位置。

        @return     処理結果。
    */
    static nn::Result AddKeyValue(size_t* outPosition,
        const char* key, const void* value, size_t size, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT;

    /*!
        @brief      キーバリューの数をインクリメントします。

        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
    */
    static void IncrementKeyValueCount(Bit8* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      キーバリューの数を取得します。

        @param[in]  buffer      バッファ。
        @param[in]  bufferSize  バッファサイズ。
    */
    static int GetKeyValueCount(const Bit8* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      キーの重複を確認します。

        @param[in]  data        既存の入力データ。
        @param[in]  dataSize    既存の入力データのサイズ。
        @param[in]  key         キー。

        @return     キーが重複しているかどうか。
    */
    static bool CheckKeyDuplication(const Bit8* data, size_t dataSize, const char* key) NN_NOEXCEPT;
};

}}}
