﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/srepo/detail/msgpack/srepo_TypesMessagePack.h>

namespace nn { namespace srepo { namespace detail { namespace msgpack {

/*!
    @brief      FixNum 型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     FixNum 型かどうか。

    @details
                FixNum 型は、数値が -31 ～ 127 の場合に 1 バイトで型と値を表現できるフォーマットです。
*/
inline bool IsFixNum(Bit8 c) NN_NOEXCEPT
{
    return (c & 0x80) == 0x00 || (c & 0xE0) == 0xE0;
}

/*!
    @brief      符号付き整数型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     符号付き整数型かどうか。
*/
inline bool IsSignedInteger(Bit8 c) NN_NOEXCEPT
{
    return IsFixNum(c) || c == DataType_S8 || c == DataType_S16 || c == DataType_S32 || c == DataType_S64;
}

/*!
    @brief      符号なし整数型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     符号なし整数型かどうか。
*/
inline bool IsUnsignedInteger(Bit8 c) NN_NOEXCEPT
{
    return c == DataType_U8 || c == DataType_U16 || c == DataType_U32 || c == DataType_U64;
}

/*!
    @brief      FixMap 型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     FixMap 型かどうか。

    @details
                FixMap 型は、要素数が 15 以下の場合に 1 バイトで型と要素数を表現できるフォーマットです。
*/
inline bool IsFixMap(Bit8 c) NN_NOEXCEPT
{
    return (c & 0xF0) == 0x80;
}

/*!
    @brief      連想配列型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     連想配列型かどうか。
*/
inline bool IsMap(Bit8 c) NN_NOEXCEPT
{
    return IsFixMap(c) || c == DataType_Map16 || c == DataType_Map32;
}

/*!
    @brief      FixMap 型のデータを生成します。

    @param[in]  num 要素数。

    @return     メッセージデータ。
*/
inline Bit8 MakeFixMap(uint8_t num) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(num, 15u);
    return 0x80 | num;
}

/*!
    @brief      FixMap 型のデータから要素数を取得します。

    @param[in]  c   メッセージデータ。

    @return     要素数。
*/
inline int GetFixMapLength(Bit8 c) NN_NOEXCEPT
{
    return (c & 0x0F);
}

/*!
    @brief      FixArray 型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     FixArray 型かどうか。

    @details
                FixArray 型は、要素数が 15 以下の場合に 1 バイトで型と要素数を表現できるフォーマットです。
*/
inline bool IsFixArray(Bit8 c) NN_NOEXCEPT
{
    return (c & 0xF0) == 0x90;
}

/*!
    @brief      配列型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     配列型かどうか。
*/
inline bool IsArray(Bit8 c) NN_NOEXCEPT
{
    return IsFixArray(c) || c == DataType_Array16 || c == DataType_Array32;
}

/*!
    @brief      FixArray 型のデータを生成します。

    @param[in]  num 要素数。

    @return     メッセージデータ。
*/
inline Bit8 MakeFixArray(uint8_t num) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(num, 15u);
    return 0x90 | num;
}

/*!
    @brief      FixArray 型のデータから要素数を取得します。

    @param[in]  c   メッセージデータ。

    @return     要素数。
*/
inline int GetFixArrayLength(Bit8 c) NN_NOEXCEPT
{
    return (c & 0x0F);
}

/*!
    @brief      FixString 型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     FixString 型かどうか。

    @details
                FixString 型は、バイト数が 31 以下の場合に 1 バイトで型とバイト数を表現できるフォーマットです。
*/
inline bool IsFixString(Bit8 c) NN_NOEXCEPT
{
    return (c & 0xE0) == 0xA0;
}

/*!
    @brief      文字列型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     文字列型かどうか。
*/
inline bool IsString(Bit8 c) NN_NOEXCEPT
{
    return IsFixString(c) || c == DataType_String8 || c == DataType_String16 || c == DataType_String32;
}

/*!
    @brief      FixString 型のデータを生成します。

    @param[in]  length  文字列長。

    @return     メッセージデータ。
*/
inline Bit8 MakeFixString(uint8_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(length, 31u);
    return 0xA0 | length;
}

/*!
    @brief      FixString 型のデータから文字列長を取得します。

    @param[in]  c   メッセージデータ。

    @return     文字列長。
*/
inline int GetFixStringLength(Bit8 c) NN_NOEXCEPT
{
    return (c & 0x1F);
}

/*!
    @brief      バイナリデータ型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     バイナリデータ型かどうか。
*/
inline bool IsBinary(Bit8 c) NN_NOEXCEPT
{
    return c == DataType_Binary8 || c == DataType_Binary16 || c == DataType_Binary32;
}

/*!
    @brief      拡張データ型かどうかを判定します。

    @param[in]  c   メッセージデータ。

    @return     拡張データ型かどうか。
*/
inline bool IsExtension(Bit8 c) NN_NOEXCEPT
{
    return c == DataType_FixExtension1 || c == DataType_FixExtension2 ||
        c == DataType_FixExtension4 || c == DataType_FixExtension8 || c == DataType_FixExtension16 ||
        c == DataType_Extension8 || c == DataType_Extension16 || c == DataType_Extension32;
}

/*!
    @brief      現在位置に存在する任意のデータ型のオブジェクトを読み飛ばします。

    @param[in]  stream  入力ストリーム。

    @return     読み飛ばせたかどうか。

    @details
                データフォーマットが不正な場合、本関数は失敗します。
*/
bool ReadCurrent(AnyData* outData, InputStreamParam* stream) NN_NOEXCEPT;

/*!
    @brief      FixMap 型データを書き込みます。

    @param[in]  stream      出力ストリーム。
    @param[in]  numObjects  要素数。

    @return     書き込めたかどうか。

    @details
                要素数が 15 より大きい場合、書き込みは失敗します。
*/
bool WriteFixMap(OutputStreamParam* stream, uint8_t numObjects) NN_NOEXCEPT;

/*!
    @brief      連想配列の宣言を書き込みます。（16 ビット長）

    @param[in]  stream      出力ストリーム。
    @param[in]  numObjects  要素数。

    @return     書き込めたかどうか。

    @details
                要素数によらず、 DataType_Map16 型で書き込みます。
*/
bool WriteMap16(OutputStreamParam* stream, uint16_t numObjects) NN_NOEXCEPT;

/*!
    @brief      連想配列の宣言を書き込みます。（32 ビット長）

    @param[in]  stream      出力ストリーム。
    @param[in]  numObjects  要素数。

    @return     書き込めたかどうか。

    @details
                要素数によらず、 DataType_Map32 型で書き込みます。
*/
bool WriteMap32(OutputStreamParam* stream, uint32_t numObjects) NN_NOEXCEPT;

/*!
    @brief      連想配列の宣言を書き込みます。（自動サイズ調整）

    @param[in]  stream      出力ストリーム。
    @param[in]  numObjects  要素数。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、要素数によってデータ型が異なります。
*/
bool WriteMapAutoSize(OutputStreamParam* stream, uint32_t numObjects) NN_NOEXCEPT;

/*!
    @brief      FixArray 型データを書き込みます。

    @param[in]  stream      出力ストリーム。
    @param[in]  numElements 要素数。

    @return     書き込めたかどうか。

    @details
                要素数が 15 より大きい場合、書き込みは失敗します。
*/
bool WriteFixArray(OutputStreamParam* stream, uint8_t numElements) NN_NOEXCEPT;

/*!
    @brief      配列の宣言を書き込みます。（16 ビット長）

    @param[in]  stream      出力ストリーム。
    @param[in]  numElements 要素数。

    @return     書き込めたかどうか。

    @details
                要素数によらず、 DataType_Array16 型で書き込みます。
*/
bool WriteArray16(OutputStreamParam* stream, uint16_t numElements) NN_NOEXCEPT;

/*!
    @brief      配列の宣言を書き込みます。（32 ビット長）

    @param[in]  stream      出力ストリーム。
    @param[in]  numElements 要素数。

    @return     書き込めたかどうか。

    @details
                要素数によらず、 DataType_Array32 型で書き込みます。
*/
bool WriteArray32(OutputStreamParam* stream, uint32_t numElements) NN_NOEXCEPT;

/*!
    @brief      配列の宣言を書き込みます。（自動サイズ調整）

    @param[in]  stream      出力ストリーム。
    @param[in]  numElements 要素数。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、要素数によってデータ型が異なります。
*/
bool WriteArrayAutoSize(OutputStreamParam* stream, uint32_t numElements) NN_NOEXCEPT;

/*!
    @brief      文字列を書き込みます。

    @param[in]  stream  出力ストリーム。
    @param[in]  string  文字列。
    @param[in]  length  文字列長。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、文字列長によってデータ型が異なります。
*/
bool WriteString(OutputStreamParam* stream, const char* string, uint32_t length) NN_NOEXCEPT;

/*!
    @brief      バイナリ列を書き込みます。

    @param[in]  stream  出力ストリーム。
    @param[in]  binary  バイナリ列。
    @param[in]  size    バイナリ列のサイズ。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、サイズによってデータ型が異なります。
*/
bool WriteBinary(OutputStreamParam* stream, const void* binary, uint32_t size) NN_NOEXCEPT;

/*!
    @brief      拡張データを書き込みます。

    @param[in]  stream      出力ストリーム。
    @param[in]  type        アプリ指定の拡張データ型。
    @param[in]  extension   拡張データ。
    @param[in]  size        拡張データのサイズ。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、サイズによってデータ型が異なります。
*/
bool WriteExtension(OutputStreamParam* stream, Bit8 type, const void* extension, uint32_t size) NN_NOEXCEPT;

/*!
    @brief      Nil を書き込みます。

    @param[in]  stream  出力ストリーム。

    @return     書き込めたかどうか。
*/
bool WriteNil(OutputStreamParam* stream) NN_NOEXCEPT;

/*!
    @brief      Bool を書き込みます。

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。
*/
bool WriteBool(OutputStreamParam* stream, bool value) NN_NOEXCEPT;

/*!
    @brief      FixNum 型データを書き込みます。

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値が -31 ～ 127 の範囲外の場合、書き込みは失敗します。
*/
bool WriteFixNum(OutputStreamParam* stream, int8_t value) NN_NOEXCEPT;

/*!
    @brief      符号付き整数を書き込みます。（8 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_S8 型で書き込みます。
*/
bool WriteSignedInteger8(OutputStreamParam* stream, int8_t value) NN_NOEXCEPT;

/*!
    @brief      符号付き整数を書き込みます。（16 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_S16 型で書き込みます。
*/
bool WriteSignedInteger16(OutputStreamParam* stream, int16_t value) NN_NOEXCEPT;

/*!
    @brief      符号付き整数を書き込みます。（32 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_S32 型で書き込みます。
*/
bool WriteSignedInteger32(OutputStreamParam* stream, int32_t value) NN_NOEXCEPT;

/*!
    @brief      符号付き整数を書き込みます。（64 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_S64 型で書き込みます。
*/
bool WriteSignedInteger64(OutputStreamParam* stream, int64_t value) NN_NOEXCEPT;

/*!
    @brief      符号付き整数を書き込みます。（自動サイズ調整）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、値によってデータ型が異なります。
*/
bool WriteSignedIntegerAutoSize(OutputStreamParam* stream, int64_t value) NN_NOEXCEPT;

/*!
    @brief      符号なし整数を書き込みます。（8 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_U8 型で書き込みます。
*/
bool WriteUnsignedInteger8(OutputStreamParam* stream, uint8_t value) NN_NOEXCEPT;

/*!
    @brief      符号なし整数を書き込みます。（16 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_U16 型で書き込みます。
*/
bool WriteUnsignedInteger16(OutputStreamParam* stream, uint16_t value) NN_NOEXCEPT;

/*!
    @brief      符号なし整数を書き込みます。（32 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_U32 型で書き込みます。
*/
bool WriteUnsignedInteger32(OutputStreamParam* stream, uint32_t value) NN_NOEXCEPT;

/*!
    @brief      符号なし整数を書き込みます。（64 ビット長）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                値によらず、 DataType_U64 型で書き込みます。
*/
bool WriteUnsignedInteger64(OutputStreamParam* stream, uint64_t value) NN_NOEXCEPT;

/*!
    @brief      符号なし整数を書き込みます。（自動サイズ調整）

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。

    @details
                書き込みバイト数が最小になるよう、値によってデータ型が異なります。
*/
bool WriteUnsignedIntegerAutoSize(OutputStreamParam* stream, uint64_t value) NN_NOEXCEPT;

/*!
    @brief      単精度浮動小数点数を書き込みます。

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。
*/
bool WriteFloat32(OutputStreamParam* stream, float value) NN_NOEXCEPT;

/*!
    @brief      倍精度浮動小数点数を書き込みます。

    @param[in]  stream  出力ストリーム。
    @param[in]  value   値。

    @return     書き込めたかどうか。
*/
bool WriteFloat64(OutputStreamParam* stream, double value) NN_NOEXCEPT;

}}}}
