﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace spsm {

    /**
     * @brief 通知メッセージを表す列挙体です。同時に発生した場合、定義値の若いメッセージが優先的に受信されます。
     */
    enum NotificationMessage
    {
        NotificationMessage_None = 0, //!< 使用しません。

        // スリープ要求
        NotificationMessage_SleepRequiredByHighTemperature, //!< 本体高温のため、スリープが必要です。
        NotificationMessage_SleepRequiredByLowBattery, //!< 本体充電池僅少のため、スリープが必要です。
        NotificationMessage_AutoPowerDownTimerExpired, //!< 無操作時の自動スリープタイマーが発動しました。
        NotificationMessage_CecSystemStandbyReceived, //!< TV から CEC システムスタンバイコマンドを受信しました。

        // UI 操作
        NotificationMessage_PowerButtonPressedBriefly, //!< 電源ボタンが短押しされました。
        NotificationMessage_PowerButtonPressedFor3Sec, //!< 電源ボタンが押された状態が 3 秒間継続しました。
        NotificationMessage_PowerButtonPressedFor7Sec, //!< 電源ボタンが押された状態が 7 秒間継続しました。
        NotificationMessage_HomeButtonShortPressed, //!< HOME ボタンが短時間押されました。
        NotificationMessage_HomeButtonLongPressed, //!< HOME ボタンが長押しされました。

        NotificationMessage_Count, //!< 使用しません。
    };

    /**
     * @brief システム電源状態を表す列挙体です。
     */
    enum PowerState
    {
        // 以下は内部用ステート / 特殊ステートです
        PowerState_None,
        PowerState_Initial,
        PowerState_Error,

        PowerState_FullAwake, //!< システムが全て稼働している状態です
        PowerState_MinimumAwake, //!< システムの必要最小限の機能が稼働している状態です
        PowerState_MinimumAwakeForLowBatterySign, //!< 要充電画面を表示している状態です
        PowerState_MinimumAwakeForBackgroundTask, //!< バックグラウンドタスクを実行している状態です
        PowerState_SleepReady, //!< システムをスリープする準備が完了した状態、またはスリープから起床した直後の状態です。
        PowerState_ShutdownReady, //!< システムをシャットダウン・または再起動する準備が完了した状態です。
    };

    /**
     * @brief ウェイク要因を表すフラグセットです。
     */
    struct WakeReasonFlag
    {
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<0> PowerButtonPressed;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<1> AcOk;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<3> RtcForBackgroundTask;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<4> RtcForFullWakeup;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<5> BatteryChargerUpdateRequired;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<6> BatteryLevelTooLow;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<7> BluetoothActivityDetect;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<8> WiFiActivityDetect;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<9> SdCardDetect;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<10> GameCardDetect;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<11> ControllerDetectL;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<12> ControllerDetectR;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<13> ControllerButtonL;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<14> ControllerButtonR;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<15> ControllerBatteryLevelChangeL;
        typedef nn::util::BitFlagSet<32, WakeReasonFlag>::Flag<16> ControllerBatteryLevelChangeR;
    };
    typedef nn::util::BitFlagSet<32, WakeReasonFlag> WakeReasonFlagSet;
}}

