﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

namespace nn { namespace spsm { namespace server {

    // 値が若いものが高優先度
    // fwdbg の power_state_message_to_emulate で定義値を直接指定するので、定義値変更時は注意
    enum PowerStateMessage
    {
        PowerStateMessage_None = 0,

        // FATAL エラー
        // PowerStateMachine で特別扱いして処理され、ハンドラの OnMessage には渡されない特殊なメッセージ
        PowerStateMessage_PutErrorRequested,

        // ステートマシン開始
        // 起動時に一度しか発行しない
        PowerStateMessage_StartStateMachine,

        // シャットダウン要求
        PowerStateMessage_ShutdownRequested,
        PowerStateMessage_RebootRequested,
        PowerStateMessage_EventShutdownRequiredByVeryLowBattery,

        // スリープ要求
        PowerStateMessage_SleepRequested,
        PowerStateMessage_EventSleepRequiredByHighTemperature,
        PowerStateMessage_EventSleepRequiredByLowBattery,
        PowerStateMessage_EventAutoPowerDownTimerExpired,
        PowerStateMessage_EventCecSystemStandbyReceived,

        // UI 操作
        PowerStateMessage_EventPowerButtonStartedPressing,
        PowerStateMessage_EventPowerButtonPressedBriefly,
        PowerStateMessage_EventPowerButtonPressedFor3Sec,
        PowerStateMessage_EventPowerButtonPressedFor7Sec,
        PowerStateMessage_EventHomeButtonStartedPressing,
        PowerStateMessage_EventHomeButtonPressedBriefly,
        PowerStateMessage_EventHomeButtonPressedLong,

        // フルアウェイク要求
        PowerStateMessage_EventPowerSupplyChanged,
        PowerStateMessage_EventFullAwakeRequestedByController,
        PowerStateMessage_EventSdCardStateChanged,
        PowerStateMessage_EventGameCardStateChanged,

        // その他
        PowerStateMessage_EventBackgroundTaskDone,
        PowerStateMessage_EventInternalTimerExpired,

        PowerStateMessage_Count,
    };

    class PowerStateMessageWithMeta
    {
    public:
        PowerStateMessageWithMeta() NN_NOEXCEPT :
            m_Message(PowerStateMessage_None),
            m_TimeStamp(nn::os::GetSystemTick())
        {}

        // PowerStateMessage にタイムスタンプを自動的に付与しながら PowerStateMessageWithMeta に変換する機能
        //
        // これを implicit にしない理由は、PowerStateMessageWithMeta -> PowerStateMessage の暗黙キャストが用意されているため。
        // implicit にすると関数間で渡されていく過程で意図せず PowerStateMessageWithMeta -> PowerStateMessage -> PowerStateMessageWithMeta と
        // キャストされ知らないうちにタイムスタンプが化けるおそれがある
        explicit PowerStateMessageWithMeta(PowerStateMessage m) NN_NOEXCEPT :
            m_Message(m),
            m_TimeStamp(nn::os::GetSystemTick())
        {}

        // PowerStateMessageWithMeta から PowerStateMessage への暗黙のキャスト
        NN_IMPLICIT operator PowerStateMessage() NN_NOEXCEPT { return m_Message; }
        NN_IMPLICIT operator const PowerStateMessage() const NN_NOEXCEPT { return m_Message; }

        nn::os::Tick GetTimeStamp() const NN_NOEXCEPT { return m_TimeStamp; }

    private:
        PowerStateMessage   m_Message;
        nn::os::Tick        m_TimeStamp;
    };

    // やっていることをわかりやすくするためのシンタックスシュガー
    // PowerStateMessageQueue.Enqueue(AddTimeStamp(PowerStateMessage_EventPowerButtonStartedPressing)) のように書けるようにする
    inline PowerStateMessageWithMeta AddTimeStamp(PowerStateMessage m) NN_NOEXCEPT
    {
        return PowerStateMessageWithMeta(m);
    }

}}}
