﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ネットワーク設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

const int SsidLengthMax = 32;                   //!< SSID の最大長
const int KeyMaterialLengthMax = 64;            //!< Key Material の最大長
const int ProxyHostNameLengthMax = 128;         //!< Proxy サーバのホスト名の最大長
const int ProxyUserNameLengthMax = 32;          //!< Proxy サーバのユーザ認証のユーザ名の最大長
const int ProxyPasswordLengthMax = 32;          //!< Proxy サーバのパスワードの最大長
const int NetworkSettingsNameLengthMax = 64;    //!< ネットワーク設定の名前の最大長
const int NetworkSettingsIdLengthMax = 16;      //!< ネットワーク設定の ID の最大長
const int NetworkSettingsCountMax = 32;         //!< ネットワーク設定の最大数

/**
 * @brief       SSID の制御フラグ定義です。
 */
struct SsidFlag
{
    /**
     * @brief       SSID がステルス状態か否かを表すフラグです。
     */
    typedef ::nn::util::BitFlagSet<32, SsidFlag>::Flag<0> IsCloaked;
};

/**
 * @brief       SSID の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, SsidFlag> SsidFlagSet;

/**
 * @brief       SSID 設定を表す構造体です。
 */
struct SsidSettings
{
    SsidFlagSet flags;  //!< SSID の制御フラグ
    Bit8 ssid[32];      //!< SSID
    int32_t ssidLength; //!< SSID の長さ
};

/**
 * @brief       認証方式を表す列挙体です。
 */
enum AuthenticationMode
{
    AuthenticationType_Open,    //!< オープン認証
    AuthenticationType_Shared,  //!< 共有キー認証
    AuthenticationType_WpaPsk,  //!< WPA-PSK 認証
    AuthenticationType_Wpa2Psk, //!< WPA2-PSK 認証
};

/**
 * @brief       暗号化方式を表す列挙体です。
 */
enum EncryptionMode
{
    EncryptionMode_None,  //!< 暗号化無し
    EncryptionMode_Wep,   //!< WEP 暗号化方式
    EncryptionMode_Aes,   //!< AES 暗号化方式
};

/**
 * @brief       セキュリティ設定を表す構造体です。
 */
struct SecuritySettings
{
    int32_t authenticationMode;             //!< 認証方式
    int32_t encryptionMode;                 //!< 暗号化方式
    char keyMaterial[KeyMaterialLengthMax]; //!< Key Material
    int32_t keyMaterialLength;              //!< Key Material の長さ
};

/**
 * @brief       IP の制御フラグ定義です。
 */
struct IpFlag
{
    /**
     * @brief       IP アドレスの自動取得の有効化フラグです。
     */
    typedef ::nn::util::BitFlagSet<32, IpFlag>::Flag<0> EnablesAutoIp;

    /**
     * @brief       DNS サーバアドレスの自動取得の有効化フラグです。
     */
    typedef ::nn::util::BitFlagSet<32, IpFlag>::Flag<1> EnablesAutoDns;
};

/**
 * @brief       IP の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, IpFlag> IpFlagSet;

/**
 * @brief       IP 設定を表す構造体です。
 */
struct IpSettings
{
    IpFlagSet flags;        //!< IP の制御フラグ
    Bit8 ipAddress[4];      //!< IP アドレス
    Bit8 subnetMask[4];     //!< サブネットマスク
    Bit8 defaultGateway[4]; //!< デフォルトゲートウェイのアドレス
    Bit8 preferredDns[4];   //!< 優先 DNS サーバのアドレス
    Bit8 alternateDns[4];   //!< 代替 DNS サーバのアドレス
};

/**
 * @brief       Proxy の制御フラグ定義です。
 */
struct ProxyFlag
{
    /**
     * @brief       Proxy の使用の有効化フラグです。
     */
    typedef ::nn::util::BitFlagSet<32, ProxyFlag>::Flag<0> Enables;

    /**
     * @brief       Proxy のユーザ認証の有効化フラグです。
     */
    typedef ::nn::util::BitFlagSet<32, ProxyFlag>::Flag<1> Authenticates;
};

/**
 * @brief       Proxy の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, ProxyFlag> ProxyFlagSet;

/**
 * @brief       Proxy 設定を表す構造体です。
 */
struct ProxySettings
{
    ProxyFlagSet flags;                     //!< Proxy 制御フラグ
    char hostName[ProxyHostNameLengthMax];  //!< Proxy サーバのホスト名
    uint16_t portNumber;                    //!< Proxy サーバのポート番号
    NN_PADDING2;
    char userName[ProxyUserNameLengthMax];  //!< Proxy のユーザ認証のユーザ名
    char password[ProxyPasswordLengthMax];  //!< Proxy のユーザ認証のパスワード
};

/**
 * @brief       プロフィールの制御フラグ定義です。
 */
struct ProfileFlag
{
    /**
     * @brief       自動接続に利用するか否かを表すフラグです。
     */
    typedef ::nn::util::BitFlagSet<32, ProfileFlag>::Flag<0> AutoConnectable;

    /**
     * @brief       大容量通信を許可するか否かを表すフラグです。
     */
    typedef ::nn::util::BitFlagSet<32, ProfileFlag>::Flag<1> LargeCapacity;
};

/**
 * @brief       プロフィールの制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, ProfileFlag> ProfileFlagSet;

/**
 * @brief       NIC 種別を表す列挙体です。
 */
enum NicType
{
    NicType_Ieee80211,   //!< IEEE 802.11
    NicType_Ethernet,    //!< Ethernet
};

/**
 * @brief       ネットワーク設定を表す構造体です。
 */
struct NetworkSettings
{
    char name[NetworkSettingsNameLengthMax];        //!< ネットワーク設定の名前
    Bit8 id[NetworkSettingsIdLengthMax];            //!< ネットワーク設定の ID
    ProfileFlagSet flags;                           //!< プロフィール制御フラグ
    int32_t nicType;                                //!< NIC 種別
    SsidSettings ssidSettings;                      //!< SSID 設定
    SecuritySettings securitySettings;              //!< セキュリティ設定
    IpSettings ipSettings;                          //!< IP 設定
    ProxySettings proxySettings;                    //!< Proxy 設定
    uint16_t mtu;                                   //!< MTU
    NN_PADDING2;
};

//! @name ネットワーク設定関連 API
//! @{

/**
 * @brief       ネットワーク設定を取得します。
 *
 * @details     ネットワーク設定として指定されている値を取得可能な数だけ読み出します。
 *              取得可能なネットワーク設定の数より大きなバッファ（配列）が指定された場合、
 *              余った領域に対しては何も行いません。
 *              読み出し可能なネットワーク設定の最大数は NetworkSettingsCountMax 個です。
 *
 * @param[out]  pOutValues                  ネットワーク設定を読み出すバッファ（配列）
 * @param[in]   count                       pOutValues の数
 *
 * @return      読み出したネットワーク設定の数を返します。
 *
 * @pre
 *              - pOutValue != nullptr
 */
int GetNetworkSettings(NetworkSettings* pOutValues, int count) NN_NOEXCEPT;

/**
 * @brief       ネットワーク設定を変更します。
 *
 * @details     ネットワーク設定を指定された値に変更します。
 *              ネットワーク設定の変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能なネットワーク設定の最大数は NetworkSettingsCountMax 個です。
 *
 * @param[in]   values                      ネットワーク設定（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - 0 <= values && values <= NetworkSettingsCountMax
 */
void SetNetworkSettings(const NetworkSettings* values, int count) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
