﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   ファームウェアバージョン設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       ファームウェアバージョンを表す構造体です。
 */
struct NN_ALIGNAS(4) FirmwareVersion
{
    uint8_t major;              //!< ファームウェアのメジャーバージョン
    uint8_t minor;              //!< ファームウェアのマイナーバージョン
    uint8_t micro;              //!< ファームウェアのマイクロバージョン
    uint8_t _padding1;
    uint8_t majorRelstep;       //!< ファームウェアのメジャーリリース番号
    uint8_t minorRelstep;       //!< ファームウェアのマイナーリリース番号
    uint8_t _padding2;
    uint8_t _padding3;
    char platform[32];          //!< ファームウェアの対応するプラットフォーム
    char revision[64];          //!< ファームウェアのリビジョン
    char displayVersion[24];    //!< ファームウェアの表示バージョン
    char displayName[128];      //!< ファームウェアの表示名

    /**
     * @brief       ファームウェアバージョンの比較演算可能な整数表現を返します。
     *
     * @return      ファームウェアバージョンの比較演算可能な整数表現です。
     */
    int GetComparableVersion() const NN_NOEXCEPT
    {
        return (static_cast<int>(major) << 16 |
                static_cast<int>(minor) <<  8 |
                static_cast<int>(micro));
    }

    /**
     * @brief       ファームウェアバージョンが一致しているか否かを判断します。
     *
     * @param[in]   lhs                         比較するファームウェアバージョン
     * @param[in]   rhs                         比較するファームウェアバージョン
     *
     * @return      一致しているなら true そうでなければ false を返します。
     */
    friend bool operator==(const FirmwareVersion& lhs,
                           const FirmwareVersion& rhs) NN_NOEXCEPT
    {
        return (lhs.GetComparableVersion() == rhs.GetComparableVersion());
    }

    /**
     * @brief       ファームウェアバージョンが不一致か否かを判断します。
     *
     * @param[in]   lhs                         比較するファームウェアバージョン
     * @param[in]   rhs                         比較するファームウェアバージョン
     *
     * @return      不一致なら true そうでなければ false を返します。
     */

    friend bool operator!=(const FirmwareVersion& lhs,
                           const FirmwareVersion& rhs) NN_NOEXCEPT
    {
        return (lhs.GetComparableVersion() != rhs.GetComparableVersion());
    }

    /**
     * @brief       ファームウェアバージョンが小さいかを判断します。
     *
     * @param[in]   lhs                         比較するファームウェアバージョン
     * @param[in]   rhs                         比較するファームウェアバージョン
     *
     * @return      lhs が rhs より小さいなら true そうでなければ false を返します。
     */

    friend bool operator<(const FirmwareVersion& lhs,
                          const FirmwareVersion& rhs) NN_NOEXCEPT
    {
        return (lhs.GetComparableVersion() < rhs.GetComparableVersion());
    }

    /**
     * @brief       ファームウェアバージョンが大きいかを判断します。
     *
     * @param[in]   lhs                         比較するファームウェアバージョン
     * @param[in]   rhs                         比較するファームウェアバージョン
     *
     * @return      lhs が rhs より大きいなら true そうでなければ false を返します。
     */

    friend bool operator>(const FirmwareVersion& lhs,
                          const FirmwareVersion& rhs) NN_NOEXCEPT
    {
        return (lhs.GetComparableVersion() > rhs.GetComparableVersion());
    }

    /**
     * @brief       ファームウェアバージョンが小さいまたは等しいかを判断します。
     *
     * @param[in]   lhs                         比較するファームウェアバージョン
     * @param[in]   rhs                         比較するファームウェアバージョン
     *
     * @return      lhs が rhs より小さいかまたは等しければ true そうでなければ false を返します。
     */

    friend bool operator<=(const FirmwareVersion& lhs,
                           const FirmwareVersion& rhs) NN_NOEXCEPT
    {
        return (lhs.GetComparableVersion() <= rhs.GetComparableVersion());
    }

    /**
     * @brief       ファームウェアバージョンが大きいまたは等しいかを判断します。
     *
     * @param[in]   lhs                         比較するファームウェアバージョン
     * @param[in]   rhs                         比較するファームウェアバージョン
     *
     * @return      lhs が rhs より大きいかまたは等しければ true そうでなければ false を返します。
     */

    friend bool operator>=(const FirmwareVersion& lhs,
                           const FirmwareVersion& rhs) NN_NOEXCEPT
    {
        return (lhs.GetComparableVersion() >= rhs.GetComparableVersion());
    }
};


/**
 * @brief       ファームウェアバージョンのダイジェストを表す構造体です。
 */
struct FirmwareVersionDigest
{
    char data[64];  //!< ファームウェアバージョンのダイジェスト
};

/**
 * @brief       再起動不要 NUP のバージョンを表す構造体です。
 */
struct RebootlessSystemUpdateVersion
{
    uint32_t version;           //!< 再起動不要 NUP のバージョン
    uint8_t reserved[28];
    char displayVersion[32];    //!< 再起動不要 NUP の表示バージョン
};

//! @name ファームウェアバージョン設定関連 API
//! @{

/**
 * @brief   ファームウェアバージョンを取得します。
 *
 * @param[out]  pOutValue                   ファームウェアバージョン
 *
 * @pre
 *      - pOutValue != nullptr
 *
 * @details 現在のファームウェアバージョンを取得します。
 */
void GetFirmwareVersion(FirmwareVersion* pOutValue) NN_NOEXCEPT;

/**
 * @brief   ファームウェアバージョンのダイジェストを取得します。
 *
 * @param[out]  pOutValue                    ファームウェアバージョンのダイジェスト
 *
 * @pre
 *      - pOutValue != nullptr
 *
 * @details 現在のファームウェアバージョンのダイジェストを取得します。
 */
void GetFirmwareVersionDigest(FirmwareVersionDigest* pOutValue) NN_NOEXCEPT;

/**
 * @brief   再起動不要 NUP のバージョンを取得します。
 *
 * @param[out]  pOutValue                   再起動不要 NUP バージョン
 *
 * @pre
 *      - pOutValue != nullptr
 *
 * @details 現在の再起動不要 NUP のバージョンを取得します。
 */
void GetRebootlessSystemUpdateVersion(RebootlessSystemUpdateVersion* pOutValue) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
