﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       クロック設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/time/time_Types.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       クロックのソース ID を扱う型です。
 */
typedef nn::time::SourceId ClockSourceId;

/**
 * @brief       単調増加クロックが指し示す値を表す構造体です。
 */
typedef nn::time::SteadyClockTimePoint SteadyClockTimePoint;

/**
 * @brief       日時を返すクロックのコンテキストを表す構造体です。
 */
typedef nn::time::SystemClockContext SystemClockContext;

/**
 * @brief   ExternalSteadyClock のソース ID を取得します。
 *
 * @param[out]  pOutValue                   クロックのソース ID を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetExternalSteadyClockSourceId(ClockSourceId* pOutValue) NN_NOEXCEPT;

/**
 * @brief   ExternalSteadyClock のソース ID を変更します。
 *
 * @param[in]   value                       クロックのソース ID
 */
void SetExternalSteadyClockSourceId(const ClockSourceId& value) NN_NOEXCEPT;

/**
 * @brief   シャットダウン時の RTC 値を取得します。
 *
 * @return      RTC 値を返します。
 */
int64_t GetShutdownRtcValue() NN_NOEXCEPT;

/**
 * @brief   シャットダウン時の RTC 値を変更します。
 *
 * @param[in]   value                       RTC 値
 */
void SetShutdownRtcValue(int64_t value) NN_NOEXCEPT;

/**
 * @brief   ExternalSteadyClock の内部オフセットを取得します。
 *
 * @return      RTC 値を返します。
 */
int64_t GetExternalSteadyClockInternalOffset() NN_NOEXCEPT;

/**
 * @brief   ExternalSteadyClock の内部オフセットを設定します。
 *
 * @param[in]   value                       内部オフセット
 */
void SetExternalSteadyClockInternalOffset(int64_t value) NN_NOEXCEPT;

/**
 * @brief   UserSystemClock のコンテキストを取得します。
 *
 * @param[out]  pOutValue                   クロックのソース ID を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetUserSystemClockContext(SystemClockContext* pOutValue) NN_NOEXCEPT;

/**
 * @brief   UserSystemClock のコンテキストを変更します。
 *
 * @param[in]   value                       クロックのコンテキスト
 */
void SetUserSystemClockContext(const SystemClockContext& value) NN_NOEXCEPT;

/**
 * @brief   NetworkSystemClock のコンテキストを取得します。
 *
 * @param[out]  pOutValue                   クロックのソース ID を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetNetworkSystemClockContext(SystemClockContext* pOutValue) NN_NOEXCEPT;

/**
 * @brief   NetworkSystemClock のコンテキストを変更します。
 *
 * @param[in]   value                       クロックのコンテキスト
 */
void SetNetworkSystemClockContext(const SystemClockContext& value) NN_NOEXCEPT;


/**
 * @brief   UserSystemClock の自動補正フラグを取得します。
 *
 * @return      自動補正フラグを返します。
 */
bool IsUserSystemClockAutomaticCorrectionEnabled() NN_NOEXCEPT;

/**
 * @brief   UserSystemClock の自動補正フラグを変更します。
 *
 * @param[in]   value                       自動補正フラグ
 */
void SetUserSystemClockAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT;

/**
 * @brief   UserSystemClock の自動補正フラグの更新時間を取得します。
 *
 * @param[out]  pOutValue                   自動補正フラグ変更時間を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetUserSystemClockAutomaticCorrectionUpdatedTime(
    ::nn::time::SteadyClockTimePoint* pOutValue) NN_NOEXCEPT;

/**
 * @brief   UserSystemClock の自動補正フラグの更新時間を変更します。
 *
 * @param[in]   value                       自動補正フラグ変更時間
 */
void SetUserSystemClockAutomaticCorrectionUpdatedTime(
    const ::nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT;

}}} // namespace nn::settings::system
