﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   PWM ドライバライブラリのチャンネルアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/pwm/pwm_ChannelName.h>

namespace nn {
namespace pwm {
namespace driver {

namespace detail {
    const int ChannelSessionSize = 64;
    const int ChannelSessionAlignment = 8;
    struct ChannelSessionImplPadded;
}

/**
 * @brief PWM チャンネルのセッション用パラメータです。
 *
 * @details PWM チャンネルのセッション用パラメータです。内部の変数に直接アクセスしないでください。
 */
struct ChannelSession
{
    nn::util::TypedStorage<detail::ChannelSessionImplPadded, detail::ChannelSessionSize, detail::ChannelSessionAlignment> _impl;
};

//! @name チャンネルの初期化と終了
//! @{

/**
 * @brief 指定したチャンネルとのセッションをオープンします。
 *
 * @param[out]  pOutSession    オープンされたチャンネルセッション
 * @param[in]   name           チャンネル識別子
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したチャンネルは存在している必要があります。@n
 *  @a pOutSession はアクセス可能なアドレスを指している必要があります。
 *
 * @post
 *  指定したチャンネルが初期設定でアクセス可能な状態になります。@n
 *  また、@a pOutSession に有効なセッションが格納されます。
 *
 * @details
 *  指定したチャンネルとのセッションをオープンします。すなわち、チャンネルに接続されたデバイスの制御を可能にします。@n
 *  オープンされたチャンネルは以下の初期設定になっています。@n
 *
 *  @li 出力の有効状態 : 無効
 *  @li デューティ比 : 0
 *  @li 周期 : ボードにより異なる値
 *
 *  @a nn::pwm::SetEnabled() で信号の出力を有効化する前に、必ず明示的に @a nn::pwm::SetDuty() および
 *  @a nn::pwm::SetPeriod() で周期とデューティ比を設定してください。
 */
void OpenSession(ChannelSession* pOutSession, ChannelName name) NN_NOEXCEPT;

/**
 * @brief 指定したチャンネルとのセッションをクローズします。
 *
 * @param[in]   pSession        チャンネルセッション
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *
 * @post
 *  @a pSession が無効なセッションになります。
 *
 * @details
 *  指定したチャンネルとのセッションをクローズします。
 */
void CloseSession(ChannelSession* pSession) NN_NOEXCEPT;

//! @}
// ~チャンネルの初期化と終了

//! @name チャンネルの設定
//! @{

/**
 * @brief 出力信号の周期を設定します。
 *
 * @param[in]   pSession        チャンネルセッション
 * @param[in]   period          出力信号の周期
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  @a period はボードがサポートしている値である必要があります。
 *
 * @post
 *  チャンネルの出力信号が @a period で指定した周期になります。
 *
 * @details
 *  出力信号の周期を設定します。
 */
void SetPeriod(ChannelSession* pSession, nn::TimeSpan period) NN_NOEXCEPT;

/**
 * @brief 出力信号の周期を取得します。
 *
 * @param[in]   pSession        チャンネルセッション
 *
 * @return      出力信号の周期
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *
 * @details
 *  出力信号の周期を取得します。
 */
nn::TimeSpan GetPeriod(ChannelSession* pSession) NN_NOEXCEPT;

/**
 * @brief 出力信号のデューティ比を設定します。
 *
 * @param[in]   pSession        チャンネルセッション
 * @param[in]   duty            出力信号のデューティ比 [%]
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  @a duty は [0 - 100] の整数値である必要があります。
 *
 * @post
 *  チャンネルの出力信号が @a duty で指定したデューティ比になります。
 *
 * @details
 *  出力信号のデューティ比を設定します。
 */
void SetDuty(ChannelSession* pSession, int duty) NN_NOEXCEPT;

/**
 * @brief 出力信号のデューティ比を取得します。
 *
 * @param[in]   pSession        チャンネルセッション
 *
 * @return      出力信号のデューティ比
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *
 * @details
 *  出力信号のデューティ比を取得します。
 */
int GetDuty(ChannelSession* pSession) NN_NOEXCEPT;

/**
 * @brief 出力信号の有効／無効状態を設定します。
 *
 * @param[in]   pSession        チャンネルセッション
 * @param[in]   enable          出力信号の有効／無効状態の指定
 *                              false：無効化する
 *                              true：有効化する
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *
 * @post
 *  チャンネルの出力信号が @a enable で指定した状態になります。
 *
 * @details
 *  出力信号の有効／無効状態を設定します。@n
 *  無効状態になっているとき、出力信号の物理的な状態はデューティ比が 0 になっているときと等価ですが、
 *  ライブラリ内部では周期とデューティ比が保存された状態になっています。@n
 *  再度有効状態に設定すると、出力信号は以前の周期とデューティ比に戻ります。
 */
void SetEnabled(ChannelSession* pSession, bool enable) NN_NOEXCEPT;

/**
 * @brief 出力信号の有効／無効状態を取得します。
 *
 * @param[in]   pSession       チャンネルセッション
 *
 * @return      出力信号の有効／無効状態
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *
 * @details
 *  出力信号の有効／無効状態を取得します。
 */
bool GetEnabled(ChannelSession* pSession) NN_NOEXCEPT;

//! @}
// ~チャンネルの設定

} // driver
} // pwm
} // nn


