﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>
#include <nn/prepo/detail/service/core/prepo_ReportFileManager.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/nifm/nifm_NetworkConnection.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      アップロード処理の実行クラスです。
*/
class UploadExecuter
{
private:
    NN_DISALLOW_COPY(UploadExecuter);
    NN_DISALLOW_MOVE(UploadExecuter);

public:
    /*!
        @brief  コンストラクタです。
    */
    UploadExecuter() = default;

public:
    /*!
        @brief      ReportFileManager クラスを登録します。

        @param[in]  pManager    登録する ReportFileManager のポインタ。

        @return     現在登録されている ReportFileManager のポインタ。

        @details
                    登録解除するときは、nullptr を引数に指定します。
    */
    ReportFileManager* RegisterReportFileManager(ReportFileManager* pManager) NN_NOEXCEPT;

    /*!
        @brief      アップロード処理を実行します。

        @param[out] outSummary  アップロードしたレポートデータの概要。
        @param[in]  buffer      レポートデータのアップロードに利用するバッファ。
        @param[in]  bufferSize  @a buffer のバイトサイズ。
        @param[in]  isImmediate 即時実行するかどうか。
    */
    nn::Result Execute(ReportDataSummary* outSummary, nn::Bit8* buffer, size_t bufferSize, bool isImmediate) NN_NOEXCEPT;

    /*!
        @brief      転送状態を取得します。

        @return     転送状態。
    */
    TransmissionStatus GetTransmissionStatus() NN_NOEXCEPT;

    /*!
        @brief      最後に発生したエラーを取得します。

        @return     最後に発生したエラー。
    */
    nn::Result GetLastError() NN_NOEXCEPT;

private:
    //
    nn::Result ExecuteImpl(ReportDataSummary* outSummary, nn::Bit8* buffer, size_t bufferSize, bool isImmediate) NN_NOEXCEPT;
    //
    bool Connect(nn::nifm::NetworkConnection& nc, bool isImmediate) NN_NOEXCEPT;

private:
    //
    std::atomic<TransmissionStatus> m_TransmissionStatus { TransmissionStatus_Idle };
    //
    std::atomic<nn::Result> m_LastError { nn::ResultSuccess() };
    //
    ReportFileManager* m_pReportFileManager;
    nn::os::SdkMutex m_ReportFileManagerMutex;
};

}}}}}
