﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      スループットの履歴を記録するクラスです。

    @tparam     ValueCount  記録する履歴の個数。
*/
template <int ValueCount>
class ThroughputHistory
{
public:
    /*!
        @brief      コンストラクタです。

        @param[in]  interval    スループットを記録する間隔。
    */
    NN_IMPLICIT ThroughputHistory(nn::TimeSpan interval) NN_NOEXCEPT
        : m_IntervalTime(interval)
        , m_PreviousIndex(0)
    {
        std::memset(m_RingBuffer, 0, sizeof(m_RingBuffer));
    }

public:
    /*!
        @brief      履歴にアプリのデータの増分を追加します。

        @param[in]  value   データの増分。
    */
    void AddForApplication(uint16_t value) NN_NOEXCEPT
    {
        Add(value, 0u);
    }

    /*!
        @brief      履歴にシステムのデータの増分を追加します。

        @param[in]  value   データの増分。
    */
    void AddForSystem(uint16_t value) NN_NOEXCEPT
    {
        Add(0u, value);
    }

    /*!
        @brief      記録した履歴を取得します。

        @param[out] outCount    @a outArray に格納した個数。
        @param[out] outArray    取得した履歴を格納する配列。
        @param[in]  arrayCount  @a outArray の要素数。

        @pre
            - outArray != nullptr
            - outCount != nullptr
            - arrayCount > 0
    */
    void Get(int* outCount, ThroughputRecord* outArray, int arrayCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outArray);
        NN_SDK_REQUIRES_NOT_NULL(outCount);
        NN_SDK_REQUIRES_GREATER(arrayCount, 0);

        NN_UTIL_LOCK_GUARD(m_Mutex);

        const int64_t currentIndex = nn::os::GetSystemTick().ToTimeSpan().GetNanoSeconds() / m_IntervalTime.GetNanoSeconds();

        FillNoAccessRangeWithZero(currentIndex);

        const int copyCount = static_cast<int>(std::min(std::min(currentIndex + 1, static_cast<int64_t>(ValueCount)), static_cast<int64_t>(arrayCount)));
        const int currentIndexInBuffer = currentIndex % ValueCount;

        for (int i = 0; i < copyCount; i++)
        {
            outArray[i] = m_RingBuffer[((currentIndexInBuffer - i) + ValueCount) % ValueCount];
        }
        *outCount = copyCount;

        m_PreviousIndex = currentIndex;
    }

private:
    void Add(uint16_t valueForApp, uint16_t valueForSys) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        const int64_t currentIndex = nn::os::GetSystemTick().ToTimeSpan().GetNanoSeconds() / m_IntervalTime.GetNanoSeconds();

        FillNoAccessRangeWithZero(currentIndex);

        const int currentIndexInBuffer = currentIndex % ValueCount;

        const uint16_t capacityForApp = std::numeric_limits<uint16_t>::max() - m_RingBuffer[currentIndexInBuffer].app;
        m_RingBuffer[currentIndexInBuffer].app += std::min(valueForApp, capacityForApp);
        NN_SDK_ASSERT_GREATER(capacityForApp, 0);

        const uint16_t capacityForSys = std::numeric_limits<uint16_t>::max() - m_RingBuffer[currentIndexInBuffer].sys;
        m_RingBuffer[currentIndexInBuffer].sys += std::min(valueForSys, capacityForSys);
        NN_SDK_ASSERT_GREATER(capacityForSys, 0);

        m_PreviousIndex = currentIndex;
    }
    void FillNoAccessRangeWithZero(const int64_t currentIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

        const int fillCount = static_cast<int>(std::min(currentIndex - m_PreviousIndex, static_cast<int64_t>(ValueCount)));
        const int currentIndexInBuffer = currentIndex % ValueCount;

        for (int i = 0; i < fillCount; i++)
        {
            auto pRecord = &m_RingBuffer[((currentIndexInBuffer - i) + ValueCount) % ValueCount];
            pRecord->app = 0;
            pRecord->sys = 0;
        }
    }

private:
    const nn::TimeSpan m_IntervalTime;
    int64_t m_PreviousIndex;
    nn::os::SdkMutex m_Mutex;
    ThroughputRecord m_RingBuffer[ValueCount];
};

}}}}}
