﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      ストレージの管理クラスです。
*/
class Storage
{
private:
    NN_DISALLOW_COPY(Storage);
    NN_DISALLOW_MOVE(Storage);

private:
    static const int64_t FsBlockSize = 16 * 1024;

public:
    /*!
        @brief      テストモードを有効にします。

        @details
                    テストモードが有効な場合、ファイルシステムに Host FS を利用します。

                    テストモードを有効にするときは、ファイルシステムを利用する前に、この関数を呼び出してください。

                    Host FS を利用するには、desc の以下の項目に Debug を追加する必要があります。
                    - Desc/FsAccessControlDescriptor/FlagPresets
                    - Default/FsAccessControlData/FlagPresets
    */
    static void EnableTestMode() NN_NOEXCEPT;

public:
    /*!
        @brief  コンストラクタです。

        @param[in] mountName    マウント名。
        @param[in] saveDataId   セーブデータ ID。
        @param[in] totalSize    ジャーナル領域を含めたストレージ全体のバイトサイズ。
        @param[in] journalSize  ジャーナル領域のバイトサイズ。
    */
    Storage(const char* mountName, nn::fs::SystemSaveDataId saveDataId,
        int64_t totalSize, int64_t journalSize) NN_NOEXCEPT;

    /*!
        @brief      ストレージをマウントします。

        @return     処理結果。
    */
    nn::Result Mount();

    /*!
        @brief      ストレージをアンマウントします。
    */
    void Unmount() NN_NOEXCEPT;

    /*!
        @brief      ストレージをロックします。
    */
    void Lock() NN_NOEXCEPT;

    /*!
        @brief      ストレージをアンロックします。
    */
    void Unlock() NN_NOEXCEPT;

    /*!
        @brief      ストレージをコミットします。

        @return     処理結果。

        @details
                    ストレージに書き込んだファイルを確定するためには、コミットが必要です。
    */
    nn::Result Commit() NN_NOEXCEPT;

    /*!
        @brief      ストレージをロールバックします。

        @return     処理結果。

        @details
                    ロールバック中にストレージにアクセスしてはいけません。
    */
    nn::Result Rollback() NN_NOEXCEPT;

    /*!
        @brief      ストレージの内容をすべて削除します。

        @return     処理結果。

        @details
                    ストレージが空のときは、何もせずに成功します。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      マウント名を取得します。

        @return     マウント名。
    */
    const char* GetMountName() const NN_NOEXCEPT;

    /*!
        @brief      ストレージの使用量を取得します。

        @return     使用量。
    */
    int64_t GetUsage() const NN_NOEXCEPT;

    /*!
        @brief      ストレージの空き容量を取得します。

        @return     空き容量。
    */
    int64_t GetFreeSpaceSize() const NN_NOEXCEPT;

    /*!
        @brief      ストレージの全体容量を取得します。

        @return     全体容量。
    */
    int64_t GetTotalSpaceSize() const NN_NOEXCEPT;

    /*!
        @brief      ストレージが自スレッドによってロックされているか否かを返します。

        @return     ロックされているとき true、されていないとき false。
    */
    bool IsLockedByCurrentThread() const NN_NOEXCEPT;

private:
    nn::Result GetUsageImpl(int64_t* outFileUsage, int64_t* outFileCount, int64_t* outDirectoryCount, char* path, bool isRoot) const NN_NOEXCEPT;

private:
    static std::atomic<bool> g_IsTestMode;

private:
    const char* m_MountName;
    const nn::fs::SystemSaveDataId m_SaveDataId;
    const int64_t m_TotalSize;
    const int64_t m_JournalSize;
    nn::os::SdkMutex m_Mutex;
};

}}}}}
