﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      レポートデータの管理モジュールです。
*/
class ReportFileManager
{
private:
    NN_DISALLOW_COPY(ReportFileManager);
    NN_DISALLOW_MOVE(ReportFileManager);

public:
    /*!
        @brief      コンストラクタです。

        @param[in]  mountName                       レポートデータを管理するストレージのマウント名。
        @param[in]  isUserAgreementCheckRequired    「ユーザー情報を送信」に同意が必要なレポートデータを管理するか。
    */
    ReportFileManager(const char* mountName, bool isUserAgreementCheckRequired) NN_NOEXCEPT;

public:
    /*!
        @brief      レポートファイルのパスを生成します。

        @param[out] outPath     パスを格納するバッファ。
        @param[in]  dataSize    @a outPath のバイトサイズ。
        @param[in]  mountName   マウント名。
        @param[in]  id          ID。
    */
    static void MakeReportFilePath(char* outPath, size_t pathSize, const char* mountName, uint64_t id) NN_NOEXCEPT;

public:
    /*!
        @brief      レポートデータをファイルに書き込みます。

        @param[out] outSummary  ストレージの容量不足で削除したレポートデータの概要。
        @param[in]  data        データ。
        @param[in]  dataSize    データのサイズ。
        @param[in]  count       データの個数。

        @return     処理結果。
    */
    nn::Result WriteFile(ReportDataSummary* outSummary, void* data, size_t dataSize, int count) NN_NOEXCEPT;

    /*!
        @brief      アップロード対象のレポートファイルから、レポートデータをロードします。

        @param[out] outSummary  ロードしたレポートデータの概要。
        @param[in]  buffer      レポートデータをロードするバッファ。
        @param[in]  bufferSize  @a buffer のサイズ。

        @return     処理結果。

        @details
                    同意のないレポートを削除してから、レポートデータをロードします。
    */
    nn::Result LoadUploadFile(ReportDataSummary* outSummary, Bit8* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      アップロードが完了したファイルを削除します。

        @return     処理結果。
    */
    nn::Result CleanupUploadedFile() NN_NOEXCEPT;

    /*!
        @brief      ユーザー同意チェックが必要なレポートを管理しているか。

        @return     チェックが必要なとき true、そうでないとき false。
    */
    bool IsUserAgreementCheckRequired() const NN_NOEXCEPT;


private:
    /*!
        @brief      アップロードするファイルの名前です。
    */
    const char* UploadFileName = "post.msgpack";

private:
    //
    nn::Result WriteFileImpl(ReportDataSummary* outSummary, void* data, size_t dataSize, int count) NN_NOEXCEPT;
    //
    nn::Result OpenFile(ReportDataSummary* outSummary, nn::fs::FileHandle* outHandle, size_t appendSize) NN_NOEXCEPT;
    //
    nn::Result OpenExistingFile(nn::fs::FileHandle* outHandle, size_t appendSize) NN_NOEXCEPT;
    //
    nn::Result OpenNewFile(ReportDataSummary* outSummary, nn::fs::FileHandle* outHandle) NN_NOEXCEPT;
    //
    nn::Result IssueId(uint64_t* outId) NN_NOEXCEPT;
    //
    nn::Result DeleteOldestFile(size_t* outSize, int* outCount) NN_NOEXCEPT;
    //
    nn::Result DeleteOldFileIfNotEnoughSpace(ReportDataSummary* outSummary, int64_t requiredSize) NN_NOEXCEPT;
    //
    nn::Result GetReportFileStatus(size_t* outSize, int* outCount, const char* path) NN_NOEXCEPT;
    //
    nn::Result LoadUploadFileImpl(ReportDataSummary* outSummary, Bit8* buffer, size_t bufferSize) NN_NOEXCEPT;
    //
    nn::Result StashFile() NN_NOEXCEPT;
    //
    nn::Result GetOldestReportFilePath(char* outPath, size_t pathSize) NN_NOEXCEPT;
    //
    nn::Result ReadReportFileId(uint64_t* outId, nn::fs::DirectoryHandle handle) NN_NOEXCEPT;
    //
    bool DeleteDisagreedReport(ReportDataSummary* outSummary, Bit8* data, size_t dataSize) NN_NOEXCEPT;

private:
    //
    char m_MountName[nn::fs::MountNameLengthMax + 1];
    //
    char m_DataDirectoryPath[64];
    //
    char m_UploadFilePath[64];
    //
    uint64_t m_LastId;
    size_t m_LastFileSize;
    size_t m_LastReportCount;
    //
    const bool m_IsUserAgreementCheckRequired;
};

}}}}}
