﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      レポートデータのバッファリングモジュールです。

    @details
                レポートデータの登録毎にファイルシステムの書き込みを行わないようバッファリングを行います。@n
                バッファに貯めこまれたレポートデータは、 @ref ReportFileManager によってファイルに書き込まれます。
*/
class ReportBuffer
{
private:
    NN_DISALLOW_COPY(ReportBuffer);
    NN_DISALLOW_MOVE(ReportBuffer);

public:
    /*!
        @brief      フラッシュ関数です。

        @param[in]  category    分類。
        @param[in]  data        データ。
        @param[in]  dataSize    データのサイズ。
        @param[in]  count       データの個数。

        @return     処理結果。
    */
    using FlushFunction = nn::Result(*)(ReportCategory category, void* data, size_t dataSize, int count);

private:
    /*!
        @brief      コンストラクタです。
    */
    ReportBuffer() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ReportBuffer& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ReportBuffer, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      バッファにレポートデータがあるときに、シグナルされるイベントを登録します。

        @param[in]  event   イベント。

        @pre
            - event != nullptr

        @details
                    登録できるイベント数は 1 つのみです。
    */
    void RegisterAddEvent(nn::os::EventType* event) NN_NOEXCEPT;

    /*!
        @brief      バッファにレポートデータがあるときに、シグナルされるイベントの登録を解除します。

        @return     登録を解除したイベント。
    */
    nn::os::EventType* UnregisterAddEvent() NN_NOEXCEPT;

    /*!
        @brief      レポートデータを追加します。

        @param[in]  category    分類。
        @param[in]  data        データ。
        @param[in]  dataSize    データのサイズ。
        @param[in]  sysInfo     システム情報。
        @param[in]  sysInfoSize システム情報のサイズ。

        @return     処理結果。

        @pre
            - data != nullptr
            - dataSize > 0
            - sysInfo != nullptr
            - sysInfoSize > 0

        @details
                    バッファに空きが無い場合は、即座に返ります。
    */
    nn::Result Add(ReportCategory category, const void* data, size_t dataSize, const void* sysInfo, size_t sysInfoSize) NN_NOEXCEPT;

    /*!
        @brief      レポートデータをフラッシュします。

        @param[in]  flush       フラッシュ関数。

        @return     処理結果。

        @details
                    引数に指定したフラッシュ関数を使用して、レポートデータをフラッシュします。@n
                    フラッシュするレポートデータが無い場合は、フラッシュ関数を呼び出さず、即座に返ります。
    */
    nn::Result Flush(FlushFunction flush) NN_NOEXCEPT;

private:
    /*!
        @brief      バッファサイズです。
    */
    static const size_t BufferSize = ReportFileSizeMax - ReportFileHeaderSize;

    /*!
        @brief      バッファです。
    */
    struct Buffer
    {
        Bit8 buffer[BufferSize]; //!< バッファ。
        size_t filledSize;       //!< 書き込んだバイト数。
        int count;               //!< レポートデータ数。

        /*!
            @brief      書き込み位置のポインタを取得します。

            @return     書き込み位置のポインタ。
        */
        inline Bit8* GetWritePointer() NN_NOEXCEPT
        {
            return &(buffer[filledSize]);
        }

        /*!
            @brief      データの残りサイズを取得します。

            @return     データの残りサイズ。
        */
        inline size_t GetRemainSize() const NN_NOEXCEPT
        {
            return BufferSize - filledSize;
        }
    };

private:
    //
    nn::os::SdkMutex m_ForegroundBufferMutex;
    nn::os::SdkMutex m_BackgroundBufferMutex;
    nn::os::SdkMutex m_AddEventMutex;
    //
    Buffer m_Buffers[2];
    Buffer m_PickupBuffer;
    Buffer* m_pForegroundBuffer;
    Buffer* m_pBackgroundBuffer;
    //
    nn::os::EventType* m_AddEvent;
};

}}}}}
